/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.snapshots.impl;

import io.camunda.zeebe.snapshots.PersistedSnapshot;
import io.camunda.zeebe.snapshots.ReceivedSnapshot;
import io.camunda.zeebe.snapshots.SnapshotChunk;
import io.camunda.zeebe.snapshots.SnapshotId;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshot;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshotMetadata;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshotStore;
import io.camunda.zeebe.snapshots.impl.SnapshotChunkUtil;
import io.camunda.zeebe.snapshots.impl.SnapshotWriteException;
import io.camunda.zeebe.util.FileUtil;
import io.camunda.zeebe.util.sched.ActorControl;
import io.camunda.zeebe.util.sched.future.ActorFuture;
import io.camunda.zeebe.util.sched.future.CompletableActorFuture;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileBasedReceivedSnapshot
implements ReceivedSnapshot {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileBasedReceivedSnapshot.class);
    private static final int BLOCK_SIZE = 524288;
    private final Path directory;
    private final ActorControl actor;
    private final FileBasedSnapshotStore snapshotStore;
    private final FileBasedSnapshotMetadata metadata;
    private long expectedSnapshotChecksum;
    private int expectedTotalCount;

    FileBasedReceivedSnapshot(FileBasedSnapshotMetadata metadata, Path directory, FileBasedSnapshotStore snapshotStore, ActorControl actor) {
        this.metadata = metadata;
        this.snapshotStore = snapshotStore;
        this.directory = directory;
        this.actor = actor;
        this.expectedSnapshotChecksum = Long.MIN_VALUE;
        this.expectedTotalCount = Integer.MIN_VALUE;
    }

    @Override
    public long index() {
        return this.metadata.getIndex();
    }

    @Override
    public ActorFuture<Void> apply(SnapshotChunk snapshotChunk) {
        return this.actor.call(() -> {
            this.applyInternal(snapshotChunk);
            return null;
        });
    }

    private boolean containsChunk(String chunkId) {
        return Files.exists(this.directory.resolve(chunkId), new LinkOption[0]);
    }

    private void applyInternal(SnapshotChunk snapshotChunk) throws SnapshotWriteException {
        if (this.containsChunk(snapshotChunk.getChunkName())) {
            return;
        }
        this.checkSnapshotIdIsValid(snapshotChunk.getSnapshotId());
        long currentSnapshotChecksum = snapshotChunk.getSnapshotChecksum();
        this.checkSnapshotChecksumIsValid(currentSnapshotChecksum);
        int currentTotalCount = snapshotChunk.getTotalCount();
        this.checkTotalCountIsValid(currentTotalCount);
        String snapshotId = snapshotChunk.getSnapshotId();
        String chunkName = snapshotChunk.getChunkName();
        if (this.snapshotStore.hasSnapshotId(snapshotId)) {
            LOGGER.debug("Ignore snapshot snapshotChunk {}, because snapshot {} already exists.", (Object)chunkName, (Object)snapshotId);
            return;
        }
        this.checkChunkChecksumIsValid(snapshotChunk, snapshotId, chunkName);
        Path tmpSnapshotDirectory = this.directory;
        try {
            FileUtil.ensureDirectoryExists((Path)tmpSnapshotDirectory);
        }
        catch (IOException e) {
            throw new SnapshotWriteException(String.format("Failed to ensure that directory %s exists.", tmpSnapshotDirectory), e);
        }
        Path snapshotFile = tmpSnapshotDirectory.resolve(chunkName);
        if (Files.exists(snapshotFile, new LinkOption[0])) {
            throw new SnapshotWriteException(String.format("Received a snapshot snapshotChunk which already exist '%s'.", snapshotFile));
        }
        LOGGER.trace("Consume snapshot snapshotChunk {} of snapshot {}", (Object)chunkName, (Object)snapshotId);
        this.writeReceivedSnapshotChunk(snapshotChunk, snapshotFile);
    }

    private void checkChunkChecksumIsValid(SnapshotChunk snapshotChunk, String snapshotId, String chunkName) throws SnapshotWriteException {
        long actualChecksum;
        long expectedChecksum = snapshotChunk.getChecksum();
        if (expectedChecksum != (actualChecksum = SnapshotChunkUtil.createChecksum(snapshotChunk.getContent()))) {
            throw new SnapshotWriteException(String.format("Expected to have checksum %d for snapshot chunk %s (%s), but calculated %d", expectedChecksum, chunkName, snapshotId, actualChecksum));
        }
    }

    private void checkSnapshotChecksumIsValid(long currentSnapshotChecksum) throws SnapshotWriteException {
        if (this.expectedSnapshotChecksum == Long.MIN_VALUE) {
            this.expectedSnapshotChecksum = currentSnapshotChecksum;
        }
        if (this.expectedSnapshotChecksum != currentSnapshotChecksum) {
            throw new SnapshotWriteException(String.format("Expected snapshot chunk with equal snapshot checksum %d, but got chunk with snapshot checksum %d.", this.expectedSnapshotChecksum, currentSnapshotChecksum));
        }
    }

    private void checkTotalCountIsValid(int currentTotalCount) throws SnapshotWriteException {
        if (this.expectedTotalCount == Integer.MIN_VALUE) {
            this.expectedTotalCount = currentTotalCount;
        }
        if (this.expectedTotalCount != currentTotalCount) {
            throw new SnapshotWriteException(String.format("Expected snapshot chunk with equal snapshot total count %d, but got chunk with total count %d.", this.expectedTotalCount, currentTotalCount));
        }
    }

    private void checkSnapshotIdIsValid(String snapshotId) throws SnapshotWriteException {
        Optional<FileBasedSnapshotMetadata> receivedSnapshotId = FileBasedSnapshotMetadata.ofFileName(snapshotId);
        if (receivedSnapshotId.isEmpty()) {
            throw new SnapshotWriteException(String.format("Snapshot file name '%s' has unexpected format", snapshotId));
        }
        FileBasedSnapshotMetadata chunkMetadata = receivedSnapshotId.get();
        if (this.metadata.compareTo(chunkMetadata) != 0) {
            throw new SnapshotWriteException(String.format("Expected snapshot chunk metadata to match metadata '%s' but was '%s' instead", this.metadata, chunkMetadata));
        }
    }

    private void writeReceivedSnapshotChunk(SnapshotChunk snapshotChunk, Path snapshotFile) throws SnapshotWriteException {
        try (FileChannel channel = FileChannel.open(snapshotFile, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);){
            ByteBuffer buffer = ByteBuffer.wrap(snapshotChunk.getContent());
            while (buffer.hasRemaining()) {
                int newLimit = Math.min(buffer.capacity(), buffer.position() + 524288);
                channel.write(buffer.limit(newLimit));
                buffer.limit(buffer.capacity());
            }
            channel.force(true);
        }
        catch (IOException e) {
            throw new SnapshotWriteException(String.format("Failed to write snapshot chunk %s", snapshotChunk), e);
        }
        LOGGER.trace("Wrote replicated snapshot chunk to file {}", (Object)snapshotFile);
    }

    @Override
    public ActorFuture<Void> abort() {
        CompletableActorFuture abortFuture = new CompletableActorFuture();
        this.actor.run(() -> {
            this.abortInternal();
            abortFuture.complete(null);
        });
        return abortFuture;
    }

    @Override
    public ActorFuture<PersistedSnapshot> persist() {
        CompletableActorFuture future = new CompletableActorFuture();
        this.actor.call(() -> this.persistInternal((CompletableActorFuture<PersistedSnapshot>)future));
        return future;
    }

    @Override
    public SnapshotId snapshotId() {
        return this.metadata;
    }

    @Override
    public Path getPath() {
        return this.directory;
    }

    private void abortInternal() {
        try {
            LOGGER.debug("Aborting received snapshot in dir {}", (Object)this.directory);
            FileUtil.deleteFolderIfExists((Path)this.directory);
        }
        catch (IOException e) {
            LOGGER.warn("Failed to delete pending snapshot {}", (Object)this, (Object)e);
        }
        finally {
            this.snapshotStore.removePendingSnapshot(this);
        }
    }

    private void persistInternal(CompletableActorFuture<PersistedSnapshot> future) {
        if (this.snapshotStore.hasSnapshotId(this.metadata.getSnapshotIdAsString())) {
            this.abortInternal();
            future.complete((Object)this.snapshotStore.getLatestSnapshot().orElseThrow());
            return;
        }
        Object[] files = this.directory.toFile().listFiles();
        try {
            Objects.requireNonNull(files, "No chunks have been applied yet");
        }
        catch (Exception e) {
            future.completeExceptionally((Throwable)e);
            return;
        }
        if (files.length != this.expectedTotalCount) {
            future.completeExceptionally((Throwable)new IllegalStateException(String.format("Expected '%d' chunk files for this snapshot, but found '%d'. Files are: %s.", this.expectedTotalCount, files.length, Arrays.toString(files))));
            return;
        }
        try {
            FileBasedSnapshot value = this.snapshotStore.newSnapshot(this.metadata, this.directory, this.expectedSnapshotChecksum);
            future.complete((Object)value);
        }
        catch (Exception e) {
            future.completeExceptionally((Throwable)e);
        }
        this.snapshotStore.removePendingSnapshot(this);
    }

    public String toString() {
        return "FileBasedReceivedSnapshot{directory=" + this.directory + ", snapshotStore=" + this.snapshotStore + ", metadata=" + this.metadata + "}";
    }
}

