/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.gateway.impl.broker;

import io.camunda.zeebe.gateway.cmd.BrokerErrorException;
import io.camunda.zeebe.gateway.cmd.BrokerRejectionException;
import io.camunda.zeebe.gateway.cmd.BrokerResponseException;
import io.camunda.zeebe.gateway.cmd.ClientResponseException;
import io.camunda.zeebe.gateway.cmd.IllegalBrokerResponseException;
import io.camunda.zeebe.gateway.cmd.NoTopologyAvailableException;
import io.camunda.zeebe.gateway.cmd.PartitionNotFoundException;
import io.camunda.zeebe.gateway.impl.ErrorResponseHandler;
import io.camunda.zeebe.gateway.impl.broker.RequestDispatchStrategy;
import io.camunda.zeebe.gateway.impl.broker.cluster.BrokerClusterState;
import io.camunda.zeebe.gateway.impl.broker.cluster.BrokerTopologyManagerImpl;
import io.camunda.zeebe.gateway.impl.broker.request.BrokerPublishMessageRequest;
import io.camunda.zeebe.gateway.impl.broker.request.BrokerRequest;
import io.camunda.zeebe.gateway.impl.broker.response.BrokerResponse;
import io.camunda.zeebe.gateway.metrics.GatewayMetrics;
import io.camunda.zeebe.protocol.impl.SubscriptionUtil;
import io.camunda.zeebe.protocol.record.ErrorCode;
import io.camunda.zeebe.protocol.record.MessageHeaderDecoder;
import io.camunda.zeebe.transport.ClientRequest;
import io.camunda.zeebe.transport.ClientTransport;
import io.camunda.zeebe.util.buffer.BufferUtil;
import io.camunda.zeebe.util.sched.Actor;
import io.camunda.zeebe.util.sched.future.ActorFuture;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;
import java.util.function.ToIntFunction;
import org.agrona.DirectBuffer;

final class BrokerRequestManager
extends Actor {
    private static final TransportRequestSender SENDER_WITH_RETRY = (c, s, r, t) -> c.sendRequestWithRetry(s, BrokerRequestManager::responseValidation, r, t);
    private static final TransportRequestSender SENDER_WITHOUT_RETRY = ClientTransport::sendRequest;
    private final ClientTransport clientTransport;
    private final RequestDispatchStrategy dispatchStrategy;
    private final BrokerTopologyManagerImpl topologyManager;
    private final Duration requestTimeout;

    BrokerRequestManager(ClientTransport clientTransport, BrokerTopologyManagerImpl topologyManager, RequestDispatchStrategy dispatchStrategy, Duration requestTimeout) {
        this.clientTransport = clientTransport;
        this.dispatchStrategy = dispatchStrategy;
        this.topologyManager = topologyManager;
        this.requestTimeout = requestTimeout;
    }

    private static boolean responseValidation(DirectBuffer responseContent) {
        ErrorResponseHandler errorHandler = new ErrorResponseHandler();
        MessageHeaderDecoder headerDecoder = new MessageHeaderDecoder();
        headerDecoder.wrap(responseContent, 0);
        if (errorHandler.handlesResponse(headerDecoder)) {
            errorHandler.wrap(responseContent, headerDecoder.encodedLength(), headerDecoder.blockLength(), headerDecoder.version());
            ErrorCode errorCode = errorHandler.getErrorCode();
            return errorCode != ErrorCode.PARTITION_LEADER_MISMATCH;
        }
        return true;
    }

    <T> CompletableFuture<BrokerResponse<T>> sendRequestWithRetry(BrokerRequest<T> request) {
        return this.sendRequestWithRetry(request, this.requestTimeout);
    }

    <T> CompletableFuture<BrokerResponse<T>> sendRequest(BrokerRequest<T> request) {
        return this.sendRequest(request, this.requestTimeout);
    }

    <T> CompletableFuture<BrokerResponse<T>> sendRequest(BrokerRequest<T> request, Duration timeout) {
        return this.sendRequestInternal(request, SENDER_WITHOUT_RETRY, timeout);
    }

    <T> CompletableFuture<BrokerResponse<T>> sendRequestWithRetry(BrokerRequest<T> request, Duration requestTimeout) {
        return this.sendRequestInternal(request, SENDER_WITH_RETRY, requestTimeout);
    }

    private <T> CompletableFuture<BrokerResponse<T>> sendRequestInternal(BrokerRequest<T> request, TransportRequestSender sender, Duration requestTimeout) {
        CompletableFuture responseFuture = new CompletableFuture();
        request.serializeValue();
        this.actor.run(() -> this.sendRequestInternal(request, responseFuture, sender, requestTimeout));
        return responseFuture;
    }

    private <T> void sendRequestInternal(BrokerRequest<T> request, CompletableFuture<BrokerResponse<T>> returnFuture, TransportRequestSender sender, Duration requestTimeout) {
        BrokerAddressProvider nodeIdProvider;
        try {
            nodeIdProvider = this.determineBrokerNodeIdProvider(request);
        }
        catch (PartitionNotFoundException e) {
            returnFuture.completeExceptionally(e);
            GatewayMetrics.registerFailedRequest(request.getPartitionId(), request.getType(), "PARTITION_NOT_FOUND");
            return;
        }
        catch (NoTopologyAvailableException e) {
            returnFuture.completeExceptionally(e);
            GatewayMetrics.registerFailedRequest(request.getPartitionId(), request.getType(), "NO_TOPOLOGY");
            return;
        }
        ActorFuture<DirectBuffer> responseFuture = sender.send(this.clientTransport, nodeIdProvider, request, requestTimeout);
        long startTime = System.currentTimeMillis();
        this.actor.runOnCompletion(responseFuture, (clientResponse, error) -> {
            RequestResult result = null;
            try {
                if (error == null) {
                    BrokerResponse response = request.getResponse((DirectBuffer)clientResponse);
                    result = this.handleResponse(response, returnFuture);
                    if (result.wasProcessed()) {
                        long elapsedTime = System.currentTimeMillis() - startTime;
                        GatewayMetrics.registerSuccessfulRequest(request.getPartitionId(), request.getType(), elapsedTime);
                        return;
                    }
                } else {
                    returnFuture.completeExceptionally((Throwable)error);
                }
            }
            catch (RuntimeException e) {
                returnFuture.completeExceptionally(new ClientResponseException(e));
            }
            this.registerFailure(request, result, (Throwable)error);
        });
    }

    private <T> void registerFailure(BrokerRequest<T> request, RequestResult result, Throwable error) {
        if (result != null && result.getErrorCode() == ErrorCode.RESOURCE_EXHAUSTED) {
            return;
        }
        String code = result != null && result.getErrorCode() != ErrorCode.NULL_VAL ? result.getErrorCode().toString() : (error != null && error.getClass().equals(TimeoutException.class) ? "TIMEOUT" : "UNKNOWN");
        GatewayMetrics.registerFailedRequest(request.getPartitionId(), request.getType(), code);
    }

    private <T> RequestResult handleResponse(BrokerResponse<T> response, CompletableFuture<BrokerResponse<T>> responseFuture) {
        try {
            if (response.isResponse()) {
                responseFuture.complete(response);
                return RequestResult.processed();
            }
            if (response.isRejection()) {
                responseFuture.completeExceptionally(new BrokerRejectionException(response.getRejection()));
                return RequestResult.processed();
            }
            if (response.isError()) {
                responseFuture.completeExceptionally(new BrokerErrorException(response.getError()));
                return RequestResult.failed(response.getError().getCode());
            }
            responseFuture.completeExceptionally(new IllegalBrokerResponseException("Expected broker response to be either response, rejection, or error, but is neither of them"));
        }
        catch (RuntimeException e) {
            responseFuture.completeExceptionally(new BrokerResponseException(e));
        }
        return RequestResult.failed(ErrorCode.NULL_VAL);
    }

    private BrokerAddressProvider determineBrokerNodeIdProvider(BrokerRequest<?> request) {
        if (request.addressesSpecificPartition()) {
            BrokerClusterState topology = this.topologyManager.getTopology();
            if (topology != null && !topology.getPartitions().contains(request.getPartitionId())) {
                throw new PartitionNotFoundException(request.getPartitionId());
            }
            return new BrokerAddressProvider(request.getPartitionId());
        }
        if (request.requiresPartitionId()) {
            if (request instanceof BrokerPublishMessageRequest) {
                this.determinePartitionIdForPublishMessageRequest((BrokerPublishMessageRequest)request);
            } else {
                int partitionId = this.dispatchStrategy.determinePartition();
                if (partitionId == -3) {
                    partitionId = 1;
                }
                request.setPartitionId(partitionId);
            }
            return new BrokerAddressProvider(request.getPartitionId());
        }
        return new BrokerAddressProvider();
    }

    private void determinePartitionIdForPublishMessageRequest(BrokerPublishMessageRequest request) {
        BrokerClusterState topology = this.topologyManager.getTopology();
        if (topology == null || topology.getPartitionsCount() == 0) {
            throw new NoTopologyAvailableException(String.format("Expected to pick partition for message with correlation key '%s', but no topology is available", BufferUtil.bufferAsString((DirectBuffer)request.getCorrelationKey())));
        }
        int partitionId = SubscriptionUtil.getSubscriptionPartitionId((DirectBuffer)request.getCorrelationKey(), (int)topology.getPartitionsCount());
        request.setPartitionId(partitionId);
    }

    private static interface TransportRequestSender {
        public ActorFuture<DirectBuffer> send(ClientTransport var1, Supplier<String> var2, ClientRequest var3, Duration var4);
    }

    private class BrokerAddressProvider
    implements Supplier<String> {
        private final ToIntFunction<BrokerClusterState> nodeIdSelector;

        BrokerAddressProvider() {
            this(BrokerClusterState::getRandomBroker);
        }

        BrokerAddressProvider(int partitionId) {
            this(state -> state.getLeaderForPartition(partitionId));
        }

        BrokerAddressProvider(ToIntFunction<BrokerClusterState> nodeIdSelector) {
            this.nodeIdSelector = nodeIdSelector;
        }

        @Override
        public String get() {
            BrokerClusterState topology = BrokerRequestManager.this.topologyManager.getTopology();
            if (topology != null) {
                return topology.getBrokerAddress(this.nodeIdSelector.applyAsInt(topology));
            }
            return null;
        }
    }

    private static class RequestResult {
        private final boolean processed;
        private final ErrorCode errorCode;

        RequestResult(boolean processed, ErrorCode errorCode) {
            this.processed = processed;
            this.errorCode = errorCode;
        }

        boolean wasProcessed() {
            return this.processed;
        }

        public ErrorCode getErrorCode() {
            return this.errorCode;
        }

        static RequestResult processed() {
            return new RequestResult(true, null);
        }

        static RequestResult failed(ErrorCode code) {
            return new RequestResult(false, code);
        }
    }
}

