/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.journal.file;

import io.camunda.zeebe.journal.CorruptedJournalException;
import io.camunda.zeebe.journal.JournalException;
import io.camunda.zeebe.journal.file.JournalIndex;
import io.camunda.zeebe.journal.file.Segment;
import io.camunda.zeebe.journal.file.SegmentAllocator;
import io.camunda.zeebe.journal.file.SegmentDescriptor;
import io.camunda.zeebe.journal.file.SegmentFile;
import io.camunda.zeebe.journal.file.UninitializedSegment;
import io.camunda.zeebe.journal.file.UnknownVersionException;
import io.camunda.zeebe.util.FileUtil;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class SegmentLoader {
    private static final Logger LOGGER = LoggerFactory.getLogger(SegmentLoader.class);
    private static final ByteOrder ENDIANNESS = ByteOrder.LITTLE_ENDIAN;
    private final SegmentAllocator allocator;
    private final long minFreeDiskSpace;

    SegmentLoader(int minFreeDiskSpace) {
        this(SegmentAllocator.fill(), minFreeDiskSpace);
    }

    SegmentLoader(SegmentAllocator allocator, long minFreeDiskSpace) {
        this.allocator = allocator;
        this.minFreeDiskSpace = minFreeDiskSpace;
    }

    Segment createSegment(Path segmentFile, SegmentDescriptor descriptor, long lastWrittenAsqn, JournalIndex journalIndex) {
        MappedByteBuffer mappedSegment;
        try {
            mappedSegment = this.mapNewSegment(segmentFile, descriptor);
        }
        catch (IOException e) {
            throw new JournalException(String.format("Failed to create new segment file %s", segmentFile), e);
        }
        try {
            descriptor.copyTo(mappedSegment);
            mappedSegment.force();
        }
        catch (InternalError e) {
            throw new JournalException(String.format("Failed to ensure durability of segment %s with descriptor %s, rolling back", segmentFile, descriptor), e);
        }
        try {
            FileUtil.flushDirectory((Path)segmentFile.getParent());
        }
        catch (IOException e) {
            throw new JournalException(String.format("Failed to flush journal directory after creating segment %s", segmentFile), e);
        }
        return this.loadSegment(segmentFile, mappedSegment, descriptor, lastWrittenAsqn, journalIndex);
    }

    UninitializedSegment createUninitializedSegment(Path segmentFile, SegmentDescriptor descriptor, JournalIndex journalIndex) {
        MappedByteBuffer mappedSegment;
        try {
            mappedSegment = this.mapNewSegment(segmentFile, descriptor);
        }
        catch (IOException e) {
            throw new JournalException(String.format("Failed to create new segment file %s", segmentFile), e);
        }
        try {
            FileUtil.flushDirectory((Path)segmentFile.getParent());
        }
        catch (IOException e) {
            throw new JournalException(String.format("Failed to flush journal directory after creating segment %s", segmentFile), e);
        }
        return new UninitializedSegment(new SegmentFile(segmentFile.toFile()), descriptor.id(), descriptor.maxSegmentSize(), mappedSegment, journalIndex);
    }

    Segment loadExistingSegment(Path segmentFile, long lastWrittenAsqn, JournalIndex journalIndex) {
        MappedByteBuffer mappedSegment;
        SegmentDescriptor descriptor = this.readDescriptor(segmentFile);
        try (FileChannel channel = FileChannel.open(segmentFile, StandardOpenOption.READ, StandardOpenOption.WRITE);){
            mappedSegment = this.mapSegment(channel, descriptor.maxSegmentSize());
        }
        catch (IOException e) {
            throw new JournalException(String.format("Failed to load existing segment %s", segmentFile), e);
        }
        return this.loadSegment(segmentFile, mappedSegment, descriptor, lastWrittenAsqn, journalIndex);
    }

    private Segment loadSegment(Path file, MappedByteBuffer buffer, SegmentDescriptor descriptor, long lastWrittenAsqn, JournalIndex journalIndex) {
        SegmentFile segmentFile = new SegmentFile(file.toFile());
        return new Segment(segmentFile, descriptor, buffer, lastWrittenAsqn, journalIndex);
    }

    private MappedByteBuffer mapSegment(FileChannel channel, long segmentSize) throws IOException {
        MappedByteBuffer mappedSegment = channel.map(FileChannel.MapMode.READ_WRITE, 0L, segmentSize);
        mappedSegment.order(ENDIANNESS);
        return mappedSegment;
    }

    private SegmentDescriptor readDescriptor(Path file) {
        SegmentDescriptor segmentDescriptor;
        block12: {
            String fileName = file.getFileName().toString();
            FileChannel channel = FileChannel.open(file, StandardOpenOption.READ);
            try {
                long fileSize = Files.size(file);
                byte version = this.readVersion(channel, fileName);
                int length = SegmentDescriptor.getEncodingLengthForVersion(version);
                if (fileSize < (long)length) {
                    throw new CorruptedJournalException(String.format("Expected segment '%s' with version %d to be at least %d bytes long but it only has %d.", fileName, version, length, fileSize));
                }
                ByteBuffer buffer = ByteBuffer.allocate(length);
                int readBytes = channel.read(buffer, 0L);
                if (readBytes != -1 && readBytes < length) {
                    throw new JournalException(String.format("Expected to read %d bytes of segment '%s' with %d version but only read %d bytes.", length, fileName, version, readBytes));
                }
                buffer.flip();
                segmentDescriptor = new SegmentDescriptor(buffer);
                if (channel == null) break block12;
            }
            catch (Throwable throwable) {
                try {
                    if (channel != null) {
                        try {
                            channel.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IndexOutOfBoundsException e) {
                    throw new JournalException(String.format("Expected to read descriptor of segment '%s', but nothing was read.", fileName), e);
                }
                catch (UnknownVersionException e) {
                    throw new CorruptedJournalException(String.format("Couldn't read or recognize version of segment '%s'.", fileName), e);
                }
                catch (IOException e) {
                    throw new JournalException(e);
                }
            }
            channel.close();
        }
        return segmentDescriptor;
    }

    private byte readVersion(FileChannel channel, String fileName) throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(1);
        int readBytes = channel.read(buffer);
        if (readBytes == 0) {
            throw new JournalException(String.format("Expected to read the version byte from segment '%s' but nothing was read.", fileName));
        }
        if (readBytes == -1) {
            throw new CorruptedJournalException(String.format("Expected to read the version byte from segment '%s' but got EOF instead.", fileName));
        }
        return buffer.get(0);
    }

    private MappedByteBuffer mapNewSegment(Path segmentPath, SegmentDescriptor descriptor) throws IOException {
        MappedByteBuffer mappedByteBuffer;
        block8: {
            int maxSegmentSize = descriptor.maxSegmentSize();
            this.checkDiskSpace(segmentPath, maxSegmentSize);
            FileChannel channel = FileChannel.open(segmentPath, StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.CREATE_NEW);
            try {
                this.allocator.allocate(channel, maxSegmentSize);
                mappedByteBuffer = this.mapSegment(channel, maxSegmentSize);
                if (channel == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (channel != null) {
                        try {
                            channel.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (FileAlreadyExistsException e) {
                    LOGGER.warn("Failed to create segment {}: an unused file already existed, and will be replaced", (Object)segmentPath, (Object)e);
                    Files.delete(segmentPath);
                    return this.mapNewSegment(segmentPath, descriptor);
                }
            }
            channel.close();
        }
        return mappedByteBuffer;
    }

    private void checkDiskSpace(Path segmentPath, int maxSegmentSize) {
        long required;
        long available = segmentPath.getParent().toFile().getUsableSpace();
        if (available < (required = Math.max((long)maxSegmentSize, this.minFreeDiskSpace))) {
            throw new JournalException.OutOfDiskSpace("Not enough space to allocate a new journal segment. Required: %s, Available: %s".formatted(required, available));
        }
    }
}

