/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.backup.management;

import io.camunda.zeebe.backup.api.Backup;
import io.camunda.zeebe.backup.api.BackupIdentifier;
import io.camunda.zeebe.backup.api.BackupStatus;
import io.camunda.zeebe.backup.api.BackupStatusCode;
import io.camunda.zeebe.backup.api.BackupStore;
import io.camunda.zeebe.backup.common.BackupIdentifierImpl;
import io.camunda.zeebe.backup.common.BackupIdentifierWildcardImpl;
import io.camunda.zeebe.backup.management.BackupAlreadyExistsException;
import io.camunda.zeebe.backup.management.InProgressBackup;
import io.camunda.zeebe.scheduler.ConcurrencyControl;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import io.camunda.zeebe.scheduler.future.CompletableActorFuture;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class BackupServiceImpl {
    private static final Logger LOG = LoggerFactory.getLogger(BackupServiceImpl.class);
    private final Set<InProgressBackup> backupsInProgress = new HashSet<InProgressBackup>();
    private final BackupStore backupStore;
    private ConcurrencyControl concurrencyControl;

    BackupServiceImpl(BackupStore backupStore) {
        this.backupStore = backupStore;
    }

    void close() {
        this.backupsInProgress.forEach(InProgressBackup::close);
    }

    ActorFuture<Void> takeBackup(InProgressBackup inProgressBackup, ConcurrencyControl concurrencyControl) {
        this.concurrencyControl = concurrencyControl;
        this.backupsInProgress.add(inProgressBackup);
        CompletableFuture<BackupStatus> checkCurrentBackup = this.backupStore.getStatus(inProgressBackup.id());
        ActorFuture backupSaved = concurrencyControl.createFuture();
        checkCurrentBackup.whenCompleteAsync((status, error) -> {
            if (error != null) {
                backupSaved.completeExceptionally(error);
            } else {
                this.takeBackupIfDoesNotExist((BackupStatus)status, inProgressBackup, concurrencyControl, (ActorFuture<Void>)backupSaved);
            }
        }, arg_0 -> ((ConcurrencyControl)concurrencyControl).run(arg_0));
        backupSaved.onComplete((ignore, error) -> this.closeInProgressBackup(inProgressBackup));
        return backupSaved;
    }

    private void takeBackupIfDoesNotExist(BackupStatus status, InProgressBackup inProgressBackup, ConcurrencyControl concurrencyControl, ActorFuture<Void> backupSaved) {
        switch (status.statusCode()) {
            case COMPLETED: {
                LOG.debug("Backup {} is already completed, will not take a new one", (Object)inProgressBackup.id());
                backupSaved.complete(null);
                break;
            }
            case FAILED: 
            case IN_PROGRESS: {
                LOG.error("Backup {} already exists with status {}, will not take a new one", (Object)inProgressBackup.id(), (Object)status);
                backupSaved.completeExceptionally((Throwable)new BackupAlreadyExistsException(inProgressBackup.id(), status));
                break;
            }
            default: {
                ActorFuture snapshotFound = concurrencyControl.createFuture();
                ActorFuture snapshotReserved = concurrencyControl.createFuture();
                ActorFuture snapshotFilesCollected = concurrencyControl.createFuture();
                ActorFuture<Void> segmentFilesCollected = inProgressBackup.findSegmentFiles();
                segmentFilesCollected.onComplete(this.proceed(arg_0 -> ((ActorFuture)snapshotFound).completeExceptionally(arg_0), () -> inProgressBackup.findValidSnapshot().onComplete((BiConsumer)snapshotFound)));
                snapshotFound.onComplete(this.proceed(arg_0 -> ((ActorFuture)snapshotReserved).completeExceptionally(arg_0), () -> inProgressBackup.reserveSnapshot().onComplete((BiConsumer)snapshotReserved)));
                snapshotReserved.onComplete(this.proceed(arg_0 -> ((ActorFuture)snapshotFilesCollected).completeExceptionally(arg_0), () -> inProgressBackup.findSnapshotFiles().onComplete((BiConsumer)snapshotFilesCollected)));
                snapshotFilesCollected.onComplete(this.proceed(error -> this.failBackup(inProgressBackup, backupSaved, (Throwable)error), () -> this.saveBackup(inProgressBackup, backupSaved)));
            }
        }
    }

    private void saveBackup(InProgressBackup inProgressBackup, ActorFuture<Void> backupSaved) {
        this.saveBackup(inProgressBackup).onComplete(this.proceed(error -> this.failBackup(inProgressBackup, backupSaved, (Throwable)error), () -> backupSaved.complete(null)));
    }

    private ActorFuture<Void> saveBackup(InProgressBackup inProgressBackup) {
        ActorFuture future = this.concurrencyControl.createFuture();
        Backup backup = inProgressBackup.createBackup();
        this.backupStore.save(backup).whenComplete((ignore, error) -> {
            if (error == null) {
                future.complete(null);
            } else {
                future.completeExceptionally("Failed to save backup", error);
            }
        });
        return future;
    }

    private void failBackup(InProgressBackup inProgressBackup, ActorFuture<Void> backupSaved, Throwable error) {
        backupSaved.completeExceptionally(error);
        this.backupStore.markFailed(inProgressBackup.id(), error.getMessage());
    }

    private void closeInProgressBackup(InProgressBackup inProgressBackup) {
        this.backupsInProgress.remove(inProgressBackup);
        inProgressBackup.close();
    }

    private BiConsumer<Void, Throwable> proceed(Consumer<Throwable> onError, Runnable nextStep) {
        return (ignore, error) -> {
            if (error != null) {
                onError.accept((Throwable)error);
            } else {
                nextStep.run();
            }
        };
    }

    ActorFuture<Optional<BackupStatus>> getBackupStatus(int partitionId, long checkpointId, ConcurrencyControl executor) {
        CompletableActorFuture future = new CompletableActorFuture();
        executor.run(() -> this.backupStore.list(new BackupIdentifierWildcardImpl(Optional.empty(), Optional.of(partitionId), Optional.of(checkpointId))).whenComplete((backupStatuses, throwable) -> {
            if (throwable != null) {
                future.completeExceptionally(throwable);
            } else {
                future.complete(backupStatuses.stream().max(BackupStatusCode.BY_STATUS));
            }
        }));
        return future;
    }

    void failInProgressBackups(int partitionId, long lastCheckpointId, Collection<Integer> brokers, ConcurrencyControl executor) {
        if (lastCheckpointId != -1L) {
            executor.run(() -> {
                List<BackupIdentifierImpl> backupIds = brokers.stream().map(b -> new BackupIdentifierImpl((int)b, partitionId, lastCheckpointId)).toList();
                backupIds.forEach(this::failInProgressBackup);
            });
        }
    }

    private void failInProgressBackup(BackupIdentifier backupId) {
        ((CompletableFuture)this.backupStore.getStatus(backupId).thenAccept(status -> {
            if (status.statusCode() == BackupStatusCode.IN_PROGRESS) {
                LOG.info("The backup {} initiated by previous leader is still in progress. Marking it as failed.", (Object)backupId);
                ((CompletableFuture)this.backupStore.markFailed(backupId, "Backup is cancelled due to leader change.").thenAccept(ignore -> LOG.trace("Marked backup {} as failed.", (Object)backupId))).exceptionally(failed -> {
                    LOG.warn("Failed to mark backup {} as failed", (Object)backupId, failed);
                    return null;
                });
            }
        })).exceptionally(error -> {
            LOG.warn("Failed to retrieve status of backup {}", (Object)backupId);
            return null;
        });
    }
}

