/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.streamprocessor;

import io.camunda.zeebe.db.TransactionContext;
import io.camunda.zeebe.db.ZeebeDbTransaction;
import io.camunda.zeebe.engine.api.CommandResponseWriter;
import io.camunda.zeebe.engine.api.EmptyProcessingResult;
import io.camunda.zeebe.engine.api.ProcessingResponse;
import io.camunda.zeebe.engine.api.ProcessingResult;
import io.camunda.zeebe.engine.api.RecordProcessor;
import io.camunda.zeebe.engine.api.TypedRecord;
import io.camunda.zeebe.engine.api.records.RecordBatchEntry;
import io.camunda.zeebe.engine.metrics.StreamProcessorMetrics;
import io.camunda.zeebe.engine.processing.streamprocessor.RecordValues;
import io.camunda.zeebe.logstreams.impl.Loggers;
import io.camunda.zeebe.logstreams.log.LogStreamBatchWriter;
import io.camunda.zeebe.logstreams.log.LogStreamReader;
import io.camunda.zeebe.logstreams.log.LoggedEvent;
import io.camunda.zeebe.protocol.impl.record.RecordMetadata;
import io.camunda.zeebe.protocol.impl.record.UnifiedRecordValue;
import io.camunda.zeebe.protocol.record.RecordType;
import io.camunda.zeebe.scheduler.ActorControl;
import io.camunda.zeebe.scheduler.clock.ActorClock;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import io.camunda.zeebe.scheduler.retry.AbortableRetryStrategy;
import io.camunda.zeebe.scheduler.retry.RecoverableRetryStrategy;
import io.camunda.zeebe.scheduler.retry.RetryStrategy;
import io.camunda.zeebe.streamprocessor.BufferedProcessingResultBuilder;
import io.camunda.zeebe.streamprocessor.EventFilter;
import io.camunda.zeebe.streamprocessor.LastProcessingPositions;
import io.camunda.zeebe.streamprocessor.MetadataEventFilter;
import io.camunda.zeebe.streamprocessor.MetadataFilter;
import io.camunda.zeebe.streamprocessor.RecordProtocolVersionFilter;
import io.camunda.zeebe.streamprocessor.StreamProcessorContext;
import io.camunda.zeebe.streamprocessor.StreamProcessorListener;
import io.camunda.zeebe.streamprocessor.TypedRecordImpl;
import io.camunda.zeebe.streamprocessor.state.MutableLastProcessedPositionState;
import io.camunda.zeebe.util.buffer.BufferReader;
import io.camunda.zeebe.util.buffer.BufferUtil;
import io.camunda.zeebe.util.buffer.BufferWriter;
import io.camunda.zeebe.util.exception.RecoverableException;
import io.camunda.zeebe.util.exception.UnrecoverableException;
import io.prometheus.client.Histogram;
import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.function.BooleanSupplier;
import org.slf4j.Logger;

public final class ProcessingStateMachine {
    private static final Logger LOG = Loggers.PROCESSOR_LOGGER;
    private static final String ERROR_MESSAGE_WRITE_RECORD_ABORTED = "Expected to write one or more follow-up records for record '{} {}' without errors, but exception was thrown.";
    private static final String ERROR_MESSAGE_ROLLBACK_ABORTED = "Expected to roll back the current transaction for record '{} {}' successfully, but exception was thrown.";
    private static final String ERROR_MESSAGE_EXECUTE_SIDE_EFFECT_ABORTED = "Expected to execute side effects for record '{} {}' successfully, but exception was thrown.";
    private static final String ERROR_MESSAGE_UPDATE_STATE_FAILED = "Expected to successfully update state for record '{} {}', but caught an exception. Retry.";
    private static final String ERROR_MESSAGE_PROCESSING_FAILED_RETRY_PROCESSING = "Expected to process record '{} {}' successfully on stream processor, but caught recoverable exception. Retry processing.";
    private static final String NOTIFY_PROCESSED_LISTENER_ERROR_MESSAGE = "Expected to invoke processed listener for record {} successfully, but exception was thrown.";
    private static final String NOTIFY_SKIPPED_LISTENER_ERROR_MESSAGE = "Expected to invoke skipped listener for record '{} {}' successfully, but exception was thrown.";
    private static final Duration PROCESSING_RETRY_DELAY = Duration.ofMillis(250L);
    private static final MetadataFilter PROCESSING_FILTER = recordMetadata -> recordMetadata.getRecordType() == RecordType.COMMAND;
    private final EventFilter eventFilter = new MetadataEventFilter(new RecordProtocolVersionFilter().and(PROCESSING_FILTER));
    private final EventFilter commandFilter = new MetadataEventFilter(recordMetadata -> recordMetadata.getRecordType() != RecordType.COMMAND);
    private final MutableLastProcessedPositionState lastProcessedPositionState;
    private final RecordMetadata metadata = new RecordMetadata();
    private final ActorControl actor;
    private final LogStreamReader logStreamReader;
    private final TransactionContext transactionContext;
    private final RetryStrategy writeRetryStrategy;
    private final RetryStrategy sideEffectsRetryStrategy;
    private final RetryStrategy updateStateRetryStrategy;
    private final BooleanSupplier shouldProcessNext;
    private final BooleanSupplier abortCondition;
    private final RecordValues recordValues;
    private final TypedRecordImpl typedCommand;
    private final StreamProcessorMetrics metrics;
    private final StreamProcessorListener streamProcessorListener;
    private LoggedEvent currentRecord;
    private ZeebeDbTransaction zeebeDbTransaction;
    private long writtenPosition = -1L;
    private long lastSuccessfulProcessedRecordPosition = -1L;
    private long lastWrittenPosition = -1L;
    private volatile boolean onErrorHandlingLoop;
    private int onErrorRetries;
    private Histogram.Timer processingTimer;
    private boolean reachedEnd = true;
    private final StreamProcessorContext context;
    private final List<RecordProcessor> recordProcessors;
    private ProcessingResult currentProcessingResult;
    private RecordProcessor currentProcessor;
    private final LogStreamBatchWriter logStreamBatchWriter;
    private boolean inProcessing;

    public ProcessingStateMachine(StreamProcessorContext context, BooleanSupplier shouldProcessNext, List<RecordProcessor> recordProcessors) {
        this.context = context;
        this.recordProcessors = recordProcessors;
        this.actor = context.getActor();
        this.recordValues = context.getRecordValues();
        this.logStreamReader = context.getLogStreamReader();
        this.logStreamBatchWriter = context.getLogStreamBatchWriter();
        this.transactionContext = context.getTransactionContext();
        this.abortCondition = context.getAbortCondition();
        this.lastProcessedPositionState = context.getLastProcessedPositionState();
        this.writeRetryStrategy = new AbortableRetryStrategy(this.actor);
        this.sideEffectsRetryStrategy = new AbortableRetryStrategy(this.actor);
        this.updateStateRetryStrategy = new RecoverableRetryStrategy(this.actor);
        this.shouldProcessNext = shouldProcessNext;
        int partitionId = context.getLogStream().getPartitionId();
        this.typedCommand = new TypedRecordImpl(partitionId);
        this.metrics = new StreamProcessorMetrics(partitionId);
        this.streamProcessorListener = context.getStreamProcessorListener();
    }

    private void skipRecord() {
        this.notifySkippedListener(this.currentRecord);
        this.inProcessing = false;
        this.actor.submit(this::readNextRecord);
        this.metrics.eventSkipped();
    }

    void readNextRecord() {
        if (this.onErrorRetries > 0) {
            this.onErrorHandlingLoop = false;
            this.onErrorRetries = 0;
        }
        this.tryToReadNextRecord();
    }

    private void tryToReadNextRecord() {
        boolean hasNext = this.logStreamReader.hasNext();
        if (this.currentRecord != null) {
            LoggedEvent previousRecord = this.currentRecord;
            boolean bl = this.reachedEnd = this.commandFilter.applies(previousRecord) && !hasNext && this.lastWrittenPosition <= previousRecord.getPosition();
        }
        if (this.shouldProcessNext.getAsBoolean() && hasNext && !this.inProcessing) {
            this.currentRecord = (LoggedEvent)this.logStreamReader.next();
            if (this.eventFilter.applies(this.currentRecord)) {
                this.processCommand(this.currentRecord);
            } else {
                this.skipRecord();
            }
        }
    }

    public boolean hasReachedEnd() {
        return this.reachedEnd;
    }

    private void processCommand(LoggedEvent command) {
        this.inProcessing = true;
        this.currentProcessingResult = EmptyProcessingResult.INSTANCE;
        this.metadata.reset();
        command.readMetadata((BufferReader)this.metadata);
        long processingStartTime = ActorClock.currentTimeMillis();
        this.processingTimer = this.metrics.startProcessingDurationTimer(this.metadata.getRecordType());
        try {
            UnifiedRecordValue value = this.recordValues.readRecordValue(command, this.metadata.getValueType());
            this.typedCommand.wrap(command, this.metadata, value);
            long position = this.typedCommand.getPosition();
            BufferedProcessingResultBuilder processingResultBuilder = new BufferedProcessingResultBuilder((arg_0, arg_1) -> ((LogStreamBatchWriter)this.logStreamBatchWriter).canWriteAdditionalEvent(arg_0, arg_1));
            this.metrics.processingLatency(command.getTimestamp(), processingStartTime);
            this.currentProcessor = this.recordProcessors.stream().filter(p -> p.accepts(this.typedCommand.getValueType())).findFirst().orElse(null);
            this.zeebeDbTransaction = this.transactionContext.getCurrentTransaction();
            this.zeebeDbTransaction.run(() -> {
                if (this.currentProcessor != null) {
                    this.currentProcessingResult = this.currentProcessor.process(this.typedCommand, processingResultBuilder);
                }
                this.lastProcessedPositionState.markAsProcessed(position);
            });
            this.metrics.commandsProcessed();
            if (EmptyProcessingResult.INSTANCE == this.currentProcessingResult) {
                this.skipRecord();
                return;
            }
            this.writeRecords();
        }
        catch (RecoverableException recoverableException) {
            LOG.error(ERROR_MESSAGE_PROCESSING_FAILED_RETRY_PROCESSING, new Object[]{command, this.metadata, recoverableException});
            this.actor.runDelayed(PROCESSING_RETRY_DELAY, () -> this.processCommand(this.currentRecord));
        }
        catch (UnrecoverableException unrecoverableException) {
            throw unrecoverableException;
        }
        catch (Exception e) {
            this.onError(e, this::writeRecords);
        }
    }

    private void onError(Throwable processingException, Runnable nextStep) {
        ++this.onErrorRetries;
        if (this.onErrorRetries > 1) {
            this.onErrorHandlingLoop = true;
        }
        ActorFuture retryFuture = this.updateStateRetryStrategy.runWithRetry(() -> {
            this.zeebeDbTransaction.rollback();
            return true;
        }, this.abortCondition);
        this.actor.runOnCompletion(retryFuture, (bool, throwable) -> {
            if (throwable != null) {
                LOG.error(ERROR_MESSAGE_ROLLBACK_ABORTED, new Object[]{this.currentRecord, this.metadata, throwable});
            }
            try {
                this.errorHandlingInTransaction(processingException);
                nextStep.run();
            }
            catch (Exception ex) {
                this.onError(ex, nextStep);
            }
        });
    }

    private void errorHandlingInTransaction(Throwable processingException) throws Exception {
        this.zeebeDbTransaction = this.transactionContext.getCurrentTransaction();
        this.zeebeDbTransaction.run(() -> {
            BufferedProcessingResultBuilder processingResultBuilder = new BufferedProcessingResultBuilder((arg_0, arg_1) -> ((LogStreamBatchWriter)this.logStreamBatchWriter).canWriteAdditionalEvent(arg_0, arg_1));
            this.currentProcessingResult = this.currentProcessor.onProcessingError(processingException, this.typedCommand, processingResultBuilder);
        });
    }

    private void writeRecords() {
        ActorFuture retryFuture = this.writeRetryStrategy.runWithRetry(() -> {
            this.logStreamBatchWriter.reset();
            this.logStreamBatchWriter.sourceRecordPosition(this.typedCommand.getPosition());
            this.currentProcessingResult.getRecordBatch().forEach(entry -> this.logStreamBatchWriter.event().key(entry.key()).metadataWriter((BufferWriter)entry.recordMetadata()).sourceIndex(entry.sourceIndex()).valueWriter((BufferWriter)entry.recordValue()).done());
            long position = this.logStreamBatchWriter.tryWrite();
            if (position > 0L) {
                this.writtenPosition = position;
            }
            return position >= 0L;
        }, this.abortCondition);
        this.actor.runOnCompletion(retryFuture, (bool, t) -> {
            if (t != null) {
                LOG.error(ERROR_MESSAGE_WRITE_RECORD_ABORTED, new Object[]{this.currentRecord, this.metadata, t});
                this.onError((Throwable)t, this::writeRecords);
            } else {
                long amount = this.writtenPosition - this.lastWrittenPosition;
                this.metrics.recordsWritten(amount);
                this.updateState();
            }
        });
    }

    private void updateState() {
        ActorFuture retryFuture = this.updateStateRetryStrategy.runWithRetry(() -> {
            this.zeebeDbTransaction.commit();
            this.lastSuccessfulProcessedRecordPosition = this.currentRecord.getPosition();
            this.metrics.setLastProcessedPosition(this.lastSuccessfulProcessedRecordPosition);
            this.lastWrittenPosition = this.writtenPosition;
            return true;
        }, this.abortCondition);
        this.actor.runOnCompletion(retryFuture, (bool, throwable) -> {
            if (throwable != null) {
                LOG.error(ERROR_MESSAGE_UPDATE_STATE_FAILED, new Object[]{this.currentRecord, this.metadata, throwable});
                this.onError((Throwable)throwable, this::updateState);
            } else {
                this.executeSideEffects();
            }
        });
    }

    private void executeSideEffects() {
        ActorFuture retryFuture = this.sideEffectsRetryStrategy.runWithRetry(() -> {
            Optional<ProcessingResponse> processingResponseOptional = this.currentProcessingResult.getProcessingResponse();
            if (processingResponseOptional.isPresent()) {
                RecordBatchEntry responseValue;
                RecordMetadata recordMetadata;
                ProcessingResponse processingResponse = processingResponseOptional.get();
                CommandResponseWriter responseWriter = this.context.getCommandResponseWriter();
                boolean responseSent = responseWriter.intent((recordMetadata = (responseValue = processingResponse.responseValue()).recordMetadata()).getIntent()).key(responseValue.key()).recordType(recordMetadata.getRecordType()).rejectionReason(BufferUtil.wrapString((String)recordMetadata.getRejectionReason())).rejectionType(recordMetadata.getRejectionType()).partitionId(this.context.getPartitionId()).valueType(recordMetadata.getValueType()).valueWriter((BufferWriter)responseValue.recordValue()).tryWriteResponse(processingResponse.requestStreamId(), processingResponse.requestId());
                if (!responseSent) {
                    return false;
                }
                return this.currentProcessingResult.executePostCommitTasks();
            }
            return this.currentProcessingResult.executePostCommitTasks();
        }, this.abortCondition);
        this.actor.runOnCompletion(retryFuture, (bool, throwable) -> {
            if (throwable != null) {
                LOG.error(ERROR_MESSAGE_EXECUTE_SIDE_EFFECT_ABORTED, new Object[]{this.currentRecord, this.metadata, throwable});
            }
            this.notifyProcessedListener(this.typedCommand);
            this.processingTimer.close();
            this.inProcessing = false;
            this.actor.submit(this::readNextRecord);
        });
    }

    private void notifyProcessedListener(TypedRecord processedRecord) {
        try {
            this.streamProcessorListener.onProcessed(processedRecord);
        }
        catch (Exception e) {
            LOG.error(NOTIFY_PROCESSED_LISTENER_ERROR_MESSAGE, (Object)processedRecord, (Object)e);
        }
    }

    private void notifySkippedListener(LoggedEvent skippedRecord) {
        try {
            this.streamProcessorListener.onSkipped(skippedRecord);
        }
        catch (Exception e) {
            LOG.error(NOTIFY_SKIPPED_LISTENER_ERROR_MESSAGE, new Object[]{skippedRecord, this.metadata, e});
        }
    }

    public long getLastSuccessfulProcessedRecordPosition() {
        return this.lastSuccessfulProcessedRecordPosition;
    }

    public long getLastWrittenPosition() {
        return this.lastWrittenPosition;
    }

    public boolean isMakingProgress() {
        return !this.onErrorHandlingLoop;
    }

    public void startProcessing(LastProcessingPositions lastProcessingPositions) {
        long lastProcessedPosition = lastProcessingPositions.getLastProcessedPosition();
        this.logStreamReader.seekToNextEvent(lastProcessedPosition);
        if (this.lastSuccessfulProcessedRecordPosition == -1L) {
            this.lastSuccessfulProcessedRecordPosition = lastProcessedPosition;
        }
        if (this.lastWrittenPosition == -1L) {
            this.lastWrittenPosition = lastProcessingPositions.getLastWrittenPosition();
        }
        this.actor.submit(this::readNextRecord);
    }
}

