/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.journal.file;

import com.google.common.base.Preconditions;
import io.camunda.zeebe.journal.CorruptedJournalException;
import io.camunda.zeebe.journal.JournalException;
import io.camunda.zeebe.journal.file.JournalIndex;
import io.camunda.zeebe.journal.file.JournalMetrics;
import io.camunda.zeebe.journal.file.Segment;
import io.camunda.zeebe.journal.file.SegmentDescriptor;
import io.camunda.zeebe.journal.file.SegmentFile;
import io.camunda.zeebe.journal.file.SegmentLoader;
import io.prometheus.client.Gauge;
import java.io.File;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.SortedMap;
import java.util.concurrent.ConcurrentSkipListMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class SegmentsManager {
    private static final long FIRST_SEGMENT_ID = 1L;
    private static final long INITIAL_INDEX = 1L;
    private static final long INITIAL_ASQN = -1L;
    private static final Logger LOG = LoggerFactory.getLogger(SegmentsManager.class);
    private final JournalMetrics journalMetrics;
    private final NavigableMap<Long, Segment> segments = new ConcurrentSkipListMap<Long, Segment>();
    private volatile Segment currentSegment;
    private final JournalIndex journalIndex;
    private final int maxSegmentSize;
    private final File directory;
    private final SegmentLoader segmentLoader;
    private final long lastWrittenIndex;
    private final String name;

    SegmentsManager(JournalIndex journalIndex, int maxSegmentSize, File directory, long lastWrittenIndex, String name, SegmentLoader segmentLoader) {
        this.name = (String)Preconditions.checkNotNull((Object)name, (Object)"name cannot be null");
        this.journalMetrics = new JournalMetrics(name);
        this.journalIndex = journalIndex;
        this.maxSegmentSize = maxSegmentSize;
        this.directory = directory;
        this.lastWrittenIndex = lastWrittenIndex;
        this.segmentLoader = segmentLoader;
    }

    void close() {
        this.segments.values().forEach(segment -> {
            LOG.debug("Closing segment: {}", segment);
            segment.close();
        });
        this.currentSegment = null;
    }

    Segment getCurrentSegment() {
        return this.currentSegment;
    }

    Segment getFirstSegment() {
        Map.Entry<Long, Segment> segment = this.segments.firstEntry();
        return segment != null ? segment.getValue() : null;
    }

    Segment getLastSegment() {
        Map.Entry<Long, Segment> segment = this.segments.lastEntry();
        return segment != null ? segment.getValue() : null;
    }

    Segment getNextSegment() {
        Segment lastSegment = this.getLastSegment();
        SegmentDescriptor descriptor = SegmentDescriptor.builder().withId(lastSegment != null ? lastSegment.descriptor().id() + 1L : 1L).withIndex(this.currentSegment.lastIndex() + 1L).withMaxSegmentSize(this.maxSegmentSize).build();
        this.currentSegment = this.createSegment(descriptor, lastSegment != null ? lastSegment.lastAsqn() : -1L);
        this.segments.put(descriptor.index(), this.currentSegment);
        this.journalMetrics.incSegmentCount();
        return this.currentSegment;
    }

    Segment getNextSegment(long index) {
        Map.Entry<Long, Segment> nextSegment = this.segments.higherEntry(index);
        return nextSegment != null ? nextSegment.getValue() : null;
    }

    Segment getSegment(long index) {
        if (this.currentSegment != null && index > this.currentSegment.index()) {
            return this.currentSegment;
        }
        Map.Entry<Long, Segment> segment = this.segments.floorEntry(index);
        if (segment != null) {
            return segment.getValue();
        }
        return this.getFirstSegment();
    }

    private long getFirstIndex() {
        Segment firstSegment = this.getFirstSegment();
        return firstSegment != null ? firstSegment.index() : 0L;
    }

    void deleteUntil(long index) {
        Map.Entry<Long, Segment> segmentEntry = this.segments.floorEntry(index);
        if (segmentEntry != null) {
            SortedMap<Long, Segment> compactSegments = this.segments.headMap(segmentEntry.getValue().index());
            if (compactSegments.isEmpty()) {
                LOG.debug("No segments can be deleted with index < {} (first log index: {})", (Object)index, (Object)this.getFirstIndex());
                return;
            }
            LOG.debug("{} - Deleting log up from {} up to {} (removing {} segments)", new Object[]{this.name, this.getFirstIndex(), ((Segment)compactSegments.get(compactSegments.lastKey())).index(), compactSegments.size()});
            for (Segment segment : compactSegments.values()) {
                LOG.trace("{} - Deleting segment: {}", (Object)this.name, (Object)segment);
                segment.delete();
                this.journalMetrics.decSegmentCount();
            }
            compactSegments.clear();
            this.journalIndex.deleteUntil(index);
        }
    }

    Segment resetSegments(long index) {
        for (Segment segment : this.segments.values()) {
            segment.delete();
            this.journalMetrics.decSegmentCount();
        }
        this.segments.clear();
        SegmentDescriptor descriptor = SegmentDescriptor.builder().withId(1L).withIndex(index).withMaxSegmentSize(this.maxSegmentSize).build();
        this.currentSegment = this.createSegment(descriptor, -1L);
        this.segments.put(index, this.currentSegment);
        this.journalMetrics.incSegmentCount();
        return this.currentSegment;
    }

    void removeSegment(Segment segment) {
        this.segments.remove(segment.index());
        this.journalMetrics.decSegmentCount();
        segment.delete();
        this.resetCurrentSegment();
    }

    private void resetCurrentSegment() {
        Segment lastSegment = this.getLastSegment();
        if (lastSegment != null) {
            this.currentSegment = lastSegment;
        } else {
            SegmentDescriptor descriptor = SegmentDescriptor.builder().withId(1L).withIndex(1L).withMaxSegmentSize(this.maxSegmentSize).build();
            this.currentSegment = this.createSegment(descriptor, -1L);
            this.segments.put(1L, this.currentSegment);
            this.journalMetrics.incSegmentCount();
        }
    }

    void open() {
        Gauge.Timer openDurationTimer = this.journalMetrics.startJournalOpenDurationTimer();
        for (Segment segment : this.loadSegments()) {
            this.segments.put(segment.descriptor().index(), segment);
            this.journalMetrics.incSegmentCount();
        }
        if (!this.segments.isEmpty()) {
            this.currentSegment = this.segments.lastEntry().getValue();
        } else {
            SegmentDescriptor descriptor = SegmentDescriptor.builder().withId(1L).withIndex(1L).withMaxSegmentSize(this.maxSegmentSize).build();
            this.currentSegment = this.createSegment(descriptor, -1L);
            this.segments.put(1L, this.currentSegment);
            this.journalMetrics.incSegmentCount();
        }
        openDurationTimer.close();
        this.deleteDeferredFiles();
    }

    private Segment createSegment(SegmentDescriptor descriptor, long lastWrittenAsqn) {
        File segmentFile = SegmentFile.createSegmentFile(this.name, this.directory, descriptor.id());
        return this.segmentLoader.createSegment(segmentFile.toPath(), descriptor, this.lastWrittenIndex, lastWrittenAsqn, this.journalIndex);
    }

    private Collection<Segment> loadSegments() {
        this.directory.mkdirs();
        ArrayList<Segment> segments = new ArrayList<Segment>();
        List<File> files = this.getSortedLogSegments();
        Segment previousSegment = null;
        for (int i = 0; i < files.size(); ++i) {
            File file = files.get(i);
            try {
                LOG.debug("Found segment file: {}", (Object)file.getName());
                Segment segment = this.segmentLoader.loadExistingSegment(file.toPath(), this.lastWrittenIndex, previousSegment != null ? previousSegment.lastAsqn() : -1L, this.journalIndex);
                if (i > 0) {
                    this.checkForIndexGaps((Segment)segments.get(i - 1), segment);
                }
                segments.add(segment);
                previousSegment = segment;
                continue;
            }
            catch (CorruptedJournalException e) {
                if (this.handleSegmentCorruption(files, segments, i)) {
                    return segments;
                }
                throw e;
            }
        }
        return segments;
    }

    private void checkForIndexGaps(Segment prevSegment, Segment segment) {
        if (prevSegment.lastIndex() != segment.index() - 1L) {
            throw new CorruptedJournalException(String.format("Log segment %s is not aligned with previous segment %s (last index: %d).", segment, prevSegment, prevSegment.lastIndex()));
        }
    }

    private boolean handleSegmentCorruption(List<File> files, List<Segment> segments, int failedIndex) {
        long lastSegmentIndex = 0L;
        if (!segments.isEmpty()) {
            Segment previousSegment = segments.get(segments.size() - 1);
            lastSegmentIndex = previousSegment.lastIndex();
        }
        if (this.lastWrittenIndex > lastSegmentIndex) {
            return false;
        }
        LOG.debug("Found corrupted segment after last ack'ed index {}. Deleting segments {} - {}", new Object[]{this.lastWrittenIndex, files.get(failedIndex).getName(), files.get(files.size() - 1).getName()});
        for (int i = failedIndex; i < files.size(); ++i) {
            File file = files.get(i);
            try {
                Files.delete(file.toPath());
                continue;
            }
            catch (IOException e) {
                throw new JournalException(String.format("Failed to delete log segment '%s' when handling corruption.", file.getName()), e);
            }
        }
        return true;
    }

    private List<File> getSortedLogSegments() {
        File[] files = this.directory.listFiles(file -> file.isFile() && SegmentFile.isSegmentFile(this.name, file));
        if (files == null) {
            throw new IllegalStateException(String.format("Could not list files in directory '%s'. Either the path doesn't point to a directory or an I/O error occurred.", this.directory));
        }
        Arrays.sort(files, Comparator.comparingInt(f -> SegmentFile.getSegmentIdFromPath(f.getName())));
        return Arrays.asList(files);
    }

    private void deleteDeferredFiles() {
        try (DirectoryStream<Path> segmentsToDelete = Files.newDirectoryStream(this.directory.toPath(), path -> SegmentFile.isDeletedSegmentFile(this.name, path.getFileName().toString()));){
            segmentsToDelete.forEach(this::deleteDeferredFile);
        }
        catch (IOException e) {
            LOG.warn("Could not delete segment files marked for deletion in {}. This can result in unnecessary disk usage.", (Object)this.directory.toPath(), (Object)e);
        }
    }

    private void deleteDeferredFile(Path segmentFileToDelete) {
        try {
            Files.deleteIfExists(segmentFileToDelete);
        }
        catch (IOException e) {
            LOG.warn("Could not delete file {} which is marked for deletion. This can result in unnecessary disk usage.", (Object)segmentFileToDelete, (Object)e);
        }
    }
}

