/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.restore;

import io.atomix.raft.partition.RaftPartition;
import io.camunda.zeebe.backup.api.Backup;
import io.camunda.zeebe.backup.api.BackupDescriptor;
import io.camunda.zeebe.backup.api.BackupIdentifier;
import io.camunda.zeebe.backup.api.BackupStatus;
import io.camunda.zeebe.backup.api.BackupStatusCode;
import io.camunda.zeebe.backup.api.BackupStore;
import io.camunda.zeebe.backup.common.BackupIdentifierImpl;
import io.camunda.zeebe.journal.JournalMetaStore;
import io.camunda.zeebe.journal.JournalReader;
import io.camunda.zeebe.journal.JournalRecord;
import io.camunda.zeebe.journal.file.SegmentedJournal;
import io.camunda.zeebe.restore.BackupNotFoundException;
import io.camunda.zeebe.snapshots.RestorableSnapshotStore;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshotStoreFactory;
import io.camunda.zeebe.util.FileUtil;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PartitionRestoreService {
    private static final Logger LOG = LoggerFactory.getLogger(PartitionRestoreService.class);
    final BackupStore backupStore;
    final int partitionId;
    final Set<Integer> brokerIds;
    final Path rootDirectory;
    private final RaftPartition partition;
    private final int localBrokerId;

    public PartitionRestoreService(BackupStore backupStore, RaftPartition partition, Set<Integer> brokerIds, int localNodeId) {
        this.backupStore = backupStore;
        this.partitionId = (Integer)partition.id().id();
        this.rootDirectory = partition.dataDirectory().toPath();
        this.partition = partition;
        this.brokerIds = brokerIds;
        this.localBrokerId = localNodeId;
    }

    public CompletableFuture<BackupDescriptor> restore(long backupId) {
        return this.getTargetDirectory(backupId).thenCompose(targetDirectory -> this.download(backupId, (Path)targetDirectory)).thenApply(this::moveFilesToDataDirectory).thenApply(backup -> {
            this.resetLogToCheckpointPosition(backup.descriptor().checkpointPosition(), this.rootDirectory);
            return backup.descriptor();
        }).toCompletableFuture();
    }

    private CompletionStage<Path> getTargetDirectory(long backupId) {
        try {
            if (!FileUtil.isEmpty((Path)this.rootDirectory)) {
                LOG.error("Partition's data directory {} is not empty. Aborting restore to avoid overwriting data. Please restart with a clean directory.", (Object)this.rootDirectory);
                return CompletableFuture.failedFuture(new DirectoryNotEmptyException(this.rootDirectory.toString()));
            }
            Path tempTargetDirectory = this.rootDirectory.resolve("restoring-" + backupId);
            FileUtil.ensureDirectoryExists((Path)tempTargetDirectory);
            return CompletableFuture.completedFuture(tempTargetDirectory);
        }
        catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private void resetLogToCheckpointPosition(long checkpointPosition, Path dataDirectory) {
        try (SegmentedJournal journal = SegmentedJournal.builder().withDirectory(dataDirectory.toFile()).withName(this.partition.name()).withMetaStore((JournalMetaStore)new JournalMetaStore.InMemory()).build();){
            this.resetJournal(checkpointPosition, journal);
        }
    }

    private void resetJournal(long checkpointPosition, SegmentedJournal journal) {
        try (JournalReader reader = journal.openReader();){
            reader.seekToAsqn(checkpointPosition);
            if (reader.hasNext()) {
                JournalRecord checkpointRecord = (JournalRecord)reader.next();
                if (checkpointRecord.asqn() != checkpointPosition) {
                    PartitionRestoreService.failedToFindCheckpointRecord(checkpointPosition, reader);
                }
                journal.deleteAfter(checkpointRecord.index());
            } else {
                PartitionRestoreService.failedToFindCheckpointRecord(checkpointPosition, reader);
            }
        }
    }

    private static void failedToFindCheckpointRecord(long checkpointPosition, JournalReader reader) {
        reader.seekToFirst();
        JournalRecord firstEntry = reader.hasNext() ? (JournalRecord)reader.next() : null;
        reader.seekToLast();
        JournalRecord lastEntry = reader.hasNext() ? (JournalRecord)reader.next() : null;
        LOG.error("Cannot find the checkpoint record at position {}. Log contains first record: (index = {}, position= {}) last record: (index = {}, position= {}). Restoring from this state can lead to inconsistent state. Aborting restore.", new Object[]{checkpointPosition, firstEntry != null ? firstEntry.index() : -1L, firstEntry != null ? firstEntry.asqn() : -1L, lastEntry != null ? lastEntry.index() : -1L, lastEntry != null ? lastEntry.asqn() : -1L});
        throw new IllegalStateException("Failed to restore from backup. Cannot find a record at checkpoint position %d in the log.".formatted(checkpointPosition));
    }

    private Backup moveFilesToDataDirectory(Backup backup) {
        this.moveSegmentFiles(backup);
        this.moveSnapshotFiles(backup);
        return backup;
    }

    private void moveSegmentFiles(Backup backup) {
        LOG.info("Moving journal segment files to {}", (Object)this.rootDirectory);
        Map segmentFileSet = backup.segments().namedFiles();
        Set segmentFileNames = segmentFileSet.keySet();
        segmentFileNames.forEach(name -> this.copyNamedFileToDirectory((String)name, (Path)segmentFileSet.get(name), this.rootDirectory));
        try {
            FileUtil.flushDirectory((Path)this.rootDirectory);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void moveSnapshotFiles(Backup backup) {
        if (backup.descriptor().snapshotId().isEmpty()) {
            return;
        }
        RestorableSnapshotStore snapshotStore = FileBasedSnapshotStoreFactory.createRestorableSnapshotStore((Path)this.partition.dataDirectory().toPath(), (int)((Integer)this.partition.id().id()), (int)this.localBrokerId);
        try {
            snapshotStore.restore((String)backup.descriptor().snapshotId().orElseThrow(), backup.snapshot().namedFiles());
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void copyNamedFileToDirectory(String name, Path source, Path targetDirectory) {
        Path targetFilePath = targetDirectory.resolve(name);
        try {
            Files.move(source, targetFilePath, new CopyOption[0]);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private CompletionStage<Backup> download(long checkpointId, Path tempRestoringDirectory) {
        return this.findValidBackup(checkpointId).thenCompose(backup -> {
            LOG.info("Downloading backup {} to {}", backup, (Object)tempRestoringDirectory);
            return this.backupStore.restore(backup, tempRestoringDirectory);
        });
    }

    private CompletionStage<BackupIdentifier> findValidBackup(long checkpointId) {
        LOG.info("Searching for a completed backup with id {}", (Object)checkpointId);
        List<CompletableFuture> futures = this.brokerIds.stream().map(brokerId -> new BackupIdentifierImpl(brokerId.intValue(), this.partitionId, checkpointId)).map(arg_0 -> ((BackupStore)this.backupStore).getStatus(arg_0)).toList();
        return CompletableFuture.allOf((CompletableFuture[])futures.toArray(CompletableFuture[]::new)).thenApply(ignore -> {
            List<BackupStatus> backupStatuses = futures.stream().map(CompletableFuture::join).toList();
            return this.findCompletedBackup(backupStatuses).orElseThrow(() -> {
                LOG.error("Could not find a valid backup with id {}. Found {}", (Object)checkpointId, (Object)backupStatuses);
                return new BackupNotFoundException(checkpointId);
            });
        });
    }

    private Optional<BackupIdentifier> findCompletedBackup(List<BackupStatus> backupStatuses) {
        return backupStatuses.stream().filter(s -> s.statusCode() == BackupStatusCode.COMPLETED).findFirst().map(BackupStatus::id);
    }
}

