/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.stream.impl;

import io.camunda.zeebe.db.TransactionContext;
import io.camunda.zeebe.db.ZeebeDbTransaction;
import io.camunda.zeebe.logstreams.impl.Loggers;
import io.camunda.zeebe.logstreams.log.LogAppendEntry;
import io.camunda.zeebe.logstreams.log.LogStreamReader;
import io.camunda.zeebe.logstreams.log.LogStreamWriter;
import io.camunda.zeebe.logstreams.log.LoggedEvent;
import io.camunda.zeebe.msgpack.UnpackedObject;
import io.camunda.zeebe.protocol.impl.record.RecordMetadata;
import io.camunda.zeebe.protocol.impl.record.UnifiedRecordValue;
import io.camunda.zeebe.protocol.record.RecordType;
import io.camunda.zeebe.protocol.record.RecordValue;
import io.camunda.zeebe.protocol.record.RejectionType;
import io.camunda.zeebe.scheduler.ActorControl;
import io.camunda.zeebe.scheduler.clock.ActorClock;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import io.camunda.zeebe.scheduler.future.CompletableActorFuture;
import io.camunda.zeebe.scheduler.retry.AbortableRetryStrategy;
import io.camunda.zeebe.scheduler.retry.RecoverableRetryStrategy;
import io.camunda.zeebe.scheduler.retry.RetryStrategy;
import io.camunda.zeebe.stream.api.CommandResponseWriter;
import io.camunda.zeebe.stream.api.EmptyProcessingResult;
import io.camunda.zeebe.stream.api.EventFilter;
import io.camunda.zeebe.stream.api.ProcessingResponse;
import io.camunda.zeebe.stream.api.ProcessingResult;
import io.camunda.zeebe.stream.api.RecordProcessor;
import io.camunda.zeebe.stream.api.records.ExceededBatchRecordSizeException;
import io.camunda.zeebe.stream.api.records.TypedRecord;
import io.camunda.zeebe.stream.api.scheduling.ScheduledCommandCache;
import io.camunda.zeebe.stream.api.state.MutableLastProcessedPositionState;
import io.camunda.zeebe.stream.impl.BufferedProcessingResultBuilder;
import io.camunda.zeebe.stream.impl.LastProcessingPositions;
import io.camunda.zeebe.stream.impl.MetadataEventFilter;
import io.camunda.zeebe.stream.impl.StreamProcessorContext;
import io.camunda.zeebe.stream.impl.StreamProcessorListener;
import io.camunda.zeebe.stream.impl.metrics.ProcessingMetrics;
import io.camunda.zeebe.stream.impl.metrics.StreamProcessorMetrics;
import io.camunda.zeebe.stream.impl.records.RecordBatchEntry;
import io.camunda.zeebe.stream.impl.records.RecordValues;
import io.camunda.zeebe.stream.impl.records.TypedRecordImpl;
import io.camunda.zeebe.stream.impl.records.UnwrittenRecord;
import io.camunda.zeebe.util.Either;
import io.camunda.zeebe.util.buffer.BufferReader;
import io.camunda.zeebe.util.buffer.BufferUtil;
import io.camunda.zeebe.util.buffer.BufferWriter;
import io.camunda.zeebe.util.exception.RecoverableException;
import io.camunda.zeebe.util.exception.UnrecoverableException;
import io.prometheus.client.Histogram;
import java.time.Duration;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.function.BooleanSupplier;
import org.slf4j.Logger;

public final class ProcessingStateMachine {
    public static final String WARN_MESSAGE_BATCH_PROCESSING_RETRY = "Expected to process commands in a batch, but exceeded the resulting batch size after processing {} commands (maxCommandsInBatch: {}).";
    private static final Logger LOG = Loggers.PROCESSOR_LOGGER;
    private static final String ERROR_MESSAGE_WRITE_RECORD_ABORTED = "Expected to write one or more follow-up records for record '{} {}' without errors, but exception was thrown.";
    private static final String ERROR_MESSAGE_ROLLBACK_ABORTED = "Expected to roll back the current transaction for record '{} {}' successfully, but exception was thrown.";
    private static final String ERROR_MESSAGE_EXECUTE_SIDE_EFFECT_ABORTED = "Expected to execute side effects for record '{} {}' successfully, but exception was thrown.";
    private static final String ERROR_MESSAGE_UPDATE_STATE_FAILED = "Expected to successfully update state for record '{} {}', but caught an exception. Retry.";
    private static final String ERROR_MESSAGE_PROCESSING_FAILED_RETRY_PROCESSING = "Expected to process record '{} {}' successfully on stream processor, but caught recoverable exception. Retry processing.";
    private static final String ERROR_MESSAGE_PROCESSING_FAILED_UNRECOVERABLE = "Expected to process record '{} {}' successfully on stream processor, but caught unrecoverable exception.";
    private static final String NOTIFY_PROCESSED_LISTENER_ERROR_MESSAGE = "Expected to invoke processed listener for record {} successfully, but exception was thrown.";
    private static final String NOTIFY_SKIPPED_LISTENER_ERROR_MESSAGE = "Expected to invoke skipped listener for record '{} {}' successfully, but exception was thrown.";
    private static final Duration PROCESSING_RETRY_DELAY = Duration.ofMillis(250L);
    private static final String ERROR_MESSAGE_HANDLING_PROCESSING_ERROR_FAILED = "Expected to process command '{} {}' successfully on stream processor, but caught unexpected exception. Failed to handle the exception gracefully.";
    private final EventFilter processingFilter;
    private final EventFilter isEventOrRejection = new MetadataEventFilter(recordMetadata -> {
        RecordType recordType = recordMetadata.getRecordType();
        return recordType == RecordType.EVENT || recordType == RecordType.COMMAND_REJECTION;
    });
    private final MutableLastProcessedPositionState lastProcessedPositionState;
    private final RecordMetadata metadata = new RecordMetadata();
    private final ActorControl actor;
    private final LogStreamReader logStreamReader;
    private final TransactionContext transactionContext;
    private final RetryStrategy writeRetryStrategy;
    private final RetryStrategy sideEffectsRetryStrategy;
    private final RetryStrategy updateStateRetryStrategy;
    private final BooleanSupplier shouldProcessNext;
    private final BooleanSupplier abortCondition;
    private final RecordValues recordValues;
    private final TypedRecordImpl typedCommand;
    private final StreamProcessorMetrics metrics;
    private final StreamProcessorListener streamProcessorListener;
    private LoggedEvent currentRecord;
    private ZeebeDbTransaction zeebeDbTransaction;
    private long writtenPosition = -1L;
    private long lastSuccessfulProcessedRecordPosition = -1L;
    private long lastWrittenPosition = -1L;
    private int onErrorRetries;
    private Histogram.Timer processingTimer;
    private boolean reachedEnd = true;
    private final StreamProcessorContext context;
    private final List<RecordProcessor> recordProcessors;
    private ProcessingResult currentProcessingResult;
    private List<LogAppendEntry> pendingWrites;
    private Collection<ProcessingResponse> pendingResponses;
    private RecordProcessor currentProcessor;
    private final LogStreamWriter logStreamWriter;
    private boolean inProcessing;
    private final int maxCommandsInBatch;
    private int processedCommandsCount;
    private final ProcessingMetrics processingMetrics;
    private final ScheduledCommandCache scheduledCommandCache;
    private volatile ErrorHandlingPhase errorHandlingPhase = ErrorHandlingPhase.NO_ERROR;

    public ProcessingStateMachine(StreamProcessorContext context, BooleanSupplier shouldProcessNext, List<RecordProcessor> recordProcessors, ScheduledCommandCache scheduledCommandCache) {
        this.context = context;
        this.recordProcessors = recordProcessors;
        this.scheduledCommandCache = scheduledCommandCache;
        this.actor = context.getActor();
        this.recordValues = context.getRecordValues();
        this.logStreamReader = context.getLogStreamReader();
        this.logStreamWriter = context.getLogStreamWriter();
        this.transactionContext = context.getTransactionContext();
        this.abortCondition = context.getAbortCondition();
        this.lastProcessedPositionState = context.getLastProcessedPositionState();
        this.maxCommandsInBatch = context.getMaxCommandsInBatch();
        this.writeRetryStrategy = new AbortableRetryStrategy(this.actor);
        this.sideEffectsRetryStrategy = new AbortableRetryStrategy(this.actor);
        this.updateStateRetryStrategy = new RecoverableRetryStrategy(this.actor);
        this.shouldProcessNext = shouldProcessNext;
        int partitionId = context.getLogStream().getPartitionId();
        this.typedCommand = new TypedRecordImpl(partitionId);
        this.metrics = new StreamProcessorMetrics(partitionId);
        this.streamProcessorListener = context.getStreamProcessorListener();
        this.processingMetrics = new ProcessingMetrics(Integer.toString(partitionId));
        this.processingFilter = new MetadataEventFilter(recordMetadata -> recordMetadata.getRecordType() == RecordType.COMMAND).and(record -> !record.shouldSkipProcessing()).and(context.processingFilter());
    }

    private void skipRecord() {
        this.notifySkippedListener(this.currentRecord);
        this.inProcessing = false;
        this.actor.submit(this::readNextRecord);
        this.metrics.eventSkipped();
    }

    void readNextRecord() {
        if (this.onErrorRetries > 0) {
            this.onErrorRetries = 0;
            this.errorHandlingPhase = ErrorHandlingPhase.NO_ERROR;
        }
        this.tryToReadNextRecord();
    }

    private void tryToReadNextRecord() {
        boolean hasNext = this.logStreamReader.hasNext();
        if (this.currentRecord != null) {
            LoggedEvent previousRecord = this.currentRecord;
            boolean bl = this.reachedEnd = this.isEventOrRejection.applies(previousRecord) && !hasNext && this.lastWrittenPosition <= previousRecord.getPosition();
        }
        if (this.shouldProcessNext.getAsBoolean() && hasNext && !this.inProcessing) {
            this.currentRecord = (LoggedEvent)this.logStreamReader.next();
            if (this.processingFilter.applies(this.currentRecord)) {
                this.processCommand(this.currentRecord);
            } else {
                this.skipRecord();
            }
        }
    }

    public boolean hasReachedEnd() {
        return this.reachedEnd;
    }

    private void processCommand(LoggedEvent loggedEvent) {
        this.inProcessing = true;
        this.currentProcessingResult = EmptyProcessingResult.INSTANCE;
        this.metadata.reset();
        loggedEvent.readMetadata((BufferReader)this.metadata);
        try {
            long processingStartTime = ActorClock.currentTimeMillis();
            this.metrics.processingLatency(loggedEvent.getTimestamp(), processingStartTime);
            this.processingTimer = this.metrics.startProcessingDurationTimer(this.metadata.getRecordType());
            UnifiedRecordValue value = this.recordValues.readRecordValue(loggedEvent, this.metadata.getValueType());
            this.typedCommand.wrap(loggedEvent, this.metadata, value);
            this.zeebeDbTransaction = this.transactionContext.getCurrentTransaction();
            try (Histogram.Timer timer = this.processingMetrics.startBatchProcessingDurationTimer();){
                this.zeebeDbTransaction.run(() -> this.batchProcessing(this.typedCommand));
                this.processingMetrics.observeCommandCount(this.processedCommandsCount);
            }
            this.finalizeCommandProcessing();
            this.writeRecords();
        }
        catch (RecoverableException recoverableException) {
            LOG.error(ERROR_MESSAGE_PROCESSING_FAILED_RETRY_PROCESSING, new Object[]{loggedEvent, this.metadata, recoverableException});
            this.actor.runDelayed(PROCESSING_RETRY_DELAY, () -> this.processCommand(this.currentRecord));
        }
        catch (UnrecoverableException unrecoverableException) {
            LOG.error(ERROR_MESSAGE_PROCESSING_FAILED_UNRECOVERABLE, (Object)loggedEvent, (Object)this.metadata);
            throw unrecoverableException;
        }
        catch (ExceededBatchRecordSizeException exceededBatchRecordSizeException) {
            if (this.processedCommandsCount > 0) {
                LOG.warn(WARN_MESSAGE_BATCH_PROCESSING_RETRY, new Object[]{this.processedCommandsCount, this.maxCommandsInBatch, exceededBatchRecordSizeException});
                this.processingMetrics.countRetry();
                this.onError(exceededBatchRecordSizeException, () -> this.processCommand(loggedEvent));
            } else {
                this.onError(exceededBatchRecordSizeException, () -> {
                    this.errorHandlingInTransaction(exceededBatchRecordSizeException);
                    this.writeRecords();
                });
            }
        }
        catch (Exception e) {
            this.onError(e, () -> {
                this.errorHandlingInTransaction(e);
                this.writeRecords();
            });
        }
    }

    private void finalizeCommandProcessing() {
        this.lastProcessedPositionState.markAsProcessed(this.typedCommand.getPosition());
        this.processedCommandsCount = 0;
    }

    private void batchProcessing(TypedRecord<?> initialCommand) {
        BufferedProcessingResultBuilder processingResultBuilder = new BufferedProcessingResultBuilder((arg_0, arg_1) -> ((LogStreamWriter)this.logStreamWriter).canWriteEvents(arg_0, arg_1));
        int lastProcessingResultSize = 0;
        int currentProcessingBatchLimit = this.processedCommandsCount > 0 ? this.processedCommandsCount : this.maxCommandsInBatch;
        this.processedCommandsCount = 0;
        this.pendingWrites = new ArrayList<LogAppendEntry>();
        this.pendingResponses = Collections.newSetFromMap(new IdentityHashMap(2));
        ArrayDeque pendingCommands = new ArrayDeque();
        pendingCommands.addLast(initialCommand);
        while (!pendingCommands.isEmpty() && this.processedCommandsCount < currentProcessingBatchLimit) {
            TypedRecord command = (TypedRecord)pendingCommands.removeFirst();
            this.currentProcessor = this.recordProcessors.stream().filter(p -> p.accepts(command.getValueType())).findFirst().orElse(null);
            if (this.currentProcessor != null) {
                this.currentProcessingResult = this.currentProcessor.process(command, processingResultBuilder);
                BatchProcessingStepResult batchProcessingStepResult = this.collectBatchProcessingStepResult(this.currentProcessingResult, lastProcessingResultSize, pendingCommands.size() + this.processedCommandsCount + 1, currentProcessingBatchLimit);
                pendingCommands.addAll(batchProcessingStepResult.toProcess());
                this.pendingWrites.addAll(batchProcessingStepResult.toWrite());
                this.currentProcessingResult.getProcessingResponse().ifPresent(this.pendingResponses::add);
            }
            lastProcessingResultSize = this.currentProcessingResult.getRecordBatch().entries().size();
            ++this.processedCommandsCount;
            this.metrics.commandsProcessed();
        }
    }

    private BatchProcessingStepResult collectBatchProcessingStepResult(ProcessingResult processingResult, int lastProcessingResultSize, int currentBatchSize, int currentProcessingBatchLimit) {
        ArrayList commandsToProcess = new ArrayList();
        ArrayList<LogAppendEntry> toWriteEntries = new ArrayList<LogAppendEntry>();
        processingResult.getRecordBatch().entries().stream().skip(lastProcessingResultSize).forEachOrdered(entry -> {
            LogAppendEntry toWriteEntry = entry;
            int potentialBatchSize = currentBatchSize + commandsToProcess.size();
            if (entry.recordMetadata().getRecordType() == RecordType.COMMAND && potentialBatchSize < currentProcessingBatchLimit) {
                commandsToProcess.add(new UnwrittenRecord(entry.key(), this.context.getPartitionId(), entry.recordValue(), entry.recordMetadata()));
                toWriteEntry = LogAppendEntry.ofProcessed((LogAppendEntry)entry);
            }
            toWriteEntries.add(toWriteEntry);
        });
        return new BatchProcessingStepResult(commandsToProcess, toWriteEntries);
    }

    private void onError(Throwable error, NextProcessingStep nextStep) {
        ++this.onErrorRetries;
        this.switchErrorPhase();
        ActorFuture retryFuture = this.updateStateRetryStrategy.runWithRetry(() -> {
            this.zeebeDbTransaction.rollback();
            return true;
        }, this.abortCondition);
        this.actor.runOnCompletion(retryFuture, (bool, throwable) -> {
            if (throwable != null) {
                LOG.error(ERROR_MESSAGE_ROLLBACK_ABORTED, new Object[]{this.currentRecord, this.metadata, throwable});
            }
            try {
                if (this.tryExitOutOfErrorLoop(error)) {
                    return;
                }
                nextStep.run();
            }
            catch (Exception ex) {
                this.onError(ex, nextStep);
            }
        });
    }

    private boolean tryExitOutOfErrorLoop(Throwable error) {
        try {
            if (this.errorHandlingPhase == ErrorHandlingPhase.USER_COMMAND_PROCESSING_ERROR_FAILED) {
                LOG.debug(ERROR_MESSAGE_HANDLING_PROCESSING_ERROR_FAILED, new Object[]{this.currentRecord, this.metadata, error});
                this.tryRejectingIfUserCommand(error.getMessage());
                return true;
            }
            if (this.errorHandlingPhase == ErrorHandlingPhase.USER_COMMAND_REJECT_FAILED) {
                LOG.warn(ERROR_MESSAGE_HANDLING_PROCESSING_ERROR_FAILED, new Object[]{this.currentRecord, this.metadata, error});
                this.tryRejectingIfUserCommand(String.format("Expected to process command, but caught an exception. Check broker logs (partition %s) for details.", this.context.getPartitionId()));
                return true;
            }
        }
        catch (Exception e) {
            LOG.error("Expected to write rejection for command '{} {}', but failed with unexpected error.", new Object[]{this.currentRecord, this.metadata, e});
            this.pendingResponses.clear();
            this.pendingWrites.clear();
        }
        return false;
    }

    private void startErrorLoop(boolean isUserCommand) {
        if (this.errorHandlingPhase == ErrorHandlingPhase.NO_ERROR) {
            this.errorHandlingPhase = isUserCommand ? ErrorHandlingPhase.USER_COMMAND_PROCESSING_FAILED : ErrorHandlingPhase.PROCESSING_FAILED;
        }
    }

    private void switchErrorPhase() {
        this.errorHandlingPhase = switch (this.errorHandlingPhase) {
            default -> throw new IncompatibleClassChangeError();
            case ErrorHandlingPhase.NO_ERROR -> ErrorHandlingPhase.NO_ERROR;
            case ErrorHandlingPhase.PROCESSING_FAILED -> ErrorHandlingPhase.PROCESSING_ERROR_FAILED;
            case ErrorHandlingPhase.USER_COMMAND_PROCESSING_FAILED -> ErrorHandlingPhase.USER_COMMAND_PROCESSING_ERROR_FAILED;
            case ErrorHandlingPhase.USER_COMMAND_PROCESSING_ERROR_FAILED -> ErrorHandlingPhase.USER_COMMAND_REJECT_FAILED;
            case ErrorHandlingPhase.USER_COMMAND_REJECT_FAILED -> ErrorHandlingPhase.USER_COMMAND_REJECT_SIMPLE_REJECT_FAILED;
            case ErrorHandlingPhase.PROCESSING_ERROR_FAILED, ErrorHandlingPhase.USER_COMMAND_REJECT_SIMPLE_REJECT_FAILED -> {
                LOG.error("Failed to process command '{} {}' retries. Entering endless error loop.", (Object)this.currentRecord, (Object)this.metadata);
                yield ErrorHandlingPhase.ENDLESS_ERROR_LOOP;
            }
            case ErrorHandlingPhase.ENDLESS_ERROR_LOOP -> ErrorHandlingPhase.ENDLESS_ERROR_LOOP;
        };
    }

    private void tryRejectingIfUserCommand(String errorMessage) {
        String rejectionReason = errorMessage != null ? errorMessage : "";
        BufferedProcessingResultBuilder processingResultBuilder = new BufferedProcessingResultBuilder((arg_0, arg_1) -> ((LogStreamWriter)this.logStreamWriter).canWriteEvents(arg_0, arg_1));
        this.typedCommand.getValue().reset();
        RecordMetadata rejectionMetadata = new RecordMetadata().recordType(RecordType.COMMAND_REJECTION).intent(this.typedCommand.getIntent()).rejectionType(RejectionType.PROCESSING_ERROR).rejectionReason(rejectionReason);
        processingResultBuilder.appendRecord(this.currentRecord.getKey(), rejectionMetadata.getRecordType(), rejectionMetadata.getIntent(), rejectionMetadata.getRejectionType(), rejectionMetadata.getRejectionReason(), (RecordValue)this.typedCommand.getValue());
        processingResultBuilder.withResponse(RecordType.COMMAND_REJECTION, this.typedCommand.getKey(), this.typedCommand.getIntent(), (UnpackedObject)this.typedCommand.getValue(), this.typedCommand.getValueType(), RejectionType.PROCESSING_ERROR, rejectionReason, this.typedCommand.getRequestId(), this.typedCommand.getRequestStreamId());
        this.currentProcessingResult = processingResultBuilder.build();
        this.pendingWrites = this.currentProcessingResult.getRecordBatch().entries();
        this.pendingResponses = this.currentProcessingResult.getProcessingResponse().stream().toList();
        this.finalizeCommandProcessing();
        this.writeRecords();
    }

    private void errorHandlingInTransaction(Throwable processingException) throws Exception {
        this.startErrorLoop(this.typedCommand.hasRequestMetadata());
        this.zeebeDbTransaction = this.transactionContext.getCurrentTransaction();
        this.zeebeDbTransaction.run(() -> {
            BufferedProcessingResultBuilder processingResultBuilder = new BufferedProcessingResultBuilder((arg_0, arg_1) -> ((LogStreamWriter)this.logStreamWriter).canWriteEvents(arg_0, arg_1));
            this.currentProcessingResult = this.currentProcessor.onProcessingError(processingException, this.typedCommand, processingResultBuilder);
            this.pendingWrites = this.currentProcessingResult.getRecordBatch().entries();
            this.pendingResponses = this.currentProcessingResult.getProcessingResponse().stream().toList();
            this.finalizeCommandProcessing();
        });
    }

    private ActorFuture<Boolean> writeWithRetryAsync() {
        Object writeFuture;
        long sourceRecordPosition = this.typedCommand.getPosition();
        if (this.currentProcessingResult.isEmpty()) {
            this.notifySkippedListener(this.currentRecord);
            this.metrics.eventSkipped();
            writeFuture = CompletableActorFuture.completed((Object)true);
        } else {
            writeFuture = this.pendingWrites.isEmpty() ? CompletableActorFuture.completed((Object)true) : this.writeRetryStrategy.runWithRetry(() -> {
                Either writeResult = this.logStreamWriter.tryWrite(this.pendingWrites, sourceRecordPosition);
                if (writeResult.isRight()) {
                    this.writtenPosition = (Long)writeResult.get();
                    return true;
                }
                return false;
            }, this.abortCondition);
        }
        return writeFuture;
    }

    private void writeRecords() {
        ActorFuture<Boolean> writeFuture = this.writeWithRetryAsync();
        this.actor.runOnCompletion(writeFuture, (bool, t) -> {
            if (t != null) {
                LOG.error(ERROR_MESSAGE_WRITE_RECORD_ABORTED, new Object[]{this.currentRecord, this.metadata, t});
                this.onError((Throwable)t, () -> {
                    this.errorHandlingInTransaction((Throwable)t);
                    this.writeRecords();
                });
            } else {
                long amount = this.writtenPosition - this.lastWrittenPosition;
                this.metrics.recordsWritten(amount);
                this.updateState();
            }
        });
    }

    private void updateState() {
        ActorFuture retryFuture = this.updateStateRetryStrategy.runWithRetry(() -> {
            this.zeebeDbTransaction.commit();
            this.lastSuccessfulProcessedRecordPosition = this.currentRecord.getPosition();
            this.metrics.setLastProcessedPosition(this.lastSuccessfulProcessedRecordPosition);
            this.lastWrittenPosition = this.writtenPosition;
            return true;
        }, this.abortCondition);
        this.actor.runOnCompletion(retryFuture, (bool, throwable) -> {
            if (throwable != null) {
                LOG.error(ERROR_MESSAGE_UPDATE_STATE_FAILED, new Object[]{this.currentRecord, this.metadata, throwable});
                this.onError((Throwable)throwable, () -> {
                    this.errorHandlingInTransaction((Throwable)throwable);
                    this.updateState();
                });
            } else {
                this.scheduledCommandCache.remove(this.metadata.getIntent(), this.currentRecord.getKey());
                this.executeSideEffects();
            }
        });
    }

    private void executeSideEffects() {
        ActorFuture retryFuture = this.sideEffectsRetryStrategy.runWithRetry(() -> {
            for (ProcessingResponse processingResponse : this.pendingResponses) {
                CommandResponseWriter responseWriter = this.context.getCommandResponseWriter();
                RecordBatchEntry responseValue = processingResponse.responseValue();
                RecordMetadata recordMetadata = responseValue.recordMetadata();
                responseWriter.intent(recordMetadata.getIntent()).key(responseValue.key()).recordType(recordMetadata.getRecordType()).rejectionReason(BufferUtil.wrapString((String)recordMetadata.getRejectionReason())).rejectionType(recordMetadata.getRejectionType()).partitionId(this.context.getPartitionId()).valueType(recordMetadata.getValueType()).valueWriter((BufferWriter)responseValue.recordValue()).tryWriteResponse(processingResponse.requestStreamId(), processingResponse.requestId());
            }
            return this.executePostCommitTasks();
        }, this.abortCondition);
        this.actor.runOnCompletion(retryFuture, (bool, throwable) -> {
            if (throwable != null) {
                LOG.error(ERROR_MESSAGE_EXECUTE_SIDE_EFFECT_ABORTED, new Object[]{this.currentRecord, this.metadata, throwable});
            }
            this.notifyProcessedListener(this.typedCommand);
            this.processingTimer.close();
            this.inProcessing = false;
            this.actor.submit(this::readNextRecord);
        });
    }

    private boolean executePostCommitTasks() {
        try (Histogram.Timer timer = this.processingMetrics.startBatchProcessingPostCommitTasksTimer();){
            boolean bl = this.currentProcessingResult.executePostCommitTasks();
            return bl;
        }
    }

    private void notifyProcessedListener(TypedRecord processedRecord) {
        try {
            this.streamProcessorListener.onProcessed(processedRecord);
        }
        catch (Exception e) {
            LOG.error(NOTIFY_PROCESSED_LISTENER_ERROR_MESSAGE, (Object)processedRecord, (Object)e);
        }
    }

    private void notifySkippedListener(LoggedEvent skippedRecord) {
        try {
            this.streamProcessorListener.onSkipped(skippedRecord);
        }
        catch (Exception e) {
            LOG.error(NOTIFY_SKIPPED_LISTENER_ERROR_MESSAGE, new Object[]{skippedRecord, this.metadata, e});
        }
    }

    public long getLastSuccessfulProcessedRecordPosition() {
        return this.lastSuccessfulProcessedRecordPosition;
    }

    public long getLastWrittenPosition() {
        return this.lastWrittenPosition;
    }

    public boolean isMakingProgress() {
        return this.errorHandlingPhase != ErrorHandlingPhase.ENDLESS_ERROR_LOOP;
    }

    public void startProcessing(LastProcessingPositions lastProcessingPositions) {
        long lastProcessedPosition = lastProcessingPositions.getLastProcessedPosition();
        this.logStreamReader.seekToNextEvent(lastProcessedPosition);
        if (this.lastSuccessfulProcessedRecordPosition == -1L) {
            this.lastSuccessfulProcessedRecordPosition = lastProcessedPosition;
        }
        if (this.lastWrittenPosition == -1L) {
            this.lastWrittenPosition = lastProcessingPositions.getLastWrittenPosition();
        }
        this.actor.submit(this::readNextRecord);
    }

    private static enum ErrorHandlingPhase {
        NO_ERROR,
        USER_COMMAND_PROCESSING_FAILED,
        PROCESSING_FAILED,
        PROCESSING_ERROR_FAILED,
        USER_COMMAND_PROCESSING_ERROR_FAILED,
        USER_COMMAND_REJECT_FAILED,
        USER_COMMAND_REJECT_SIMPLE_REJECT_FAILED,
        ENDLESS_ERROR_LOOP;

    }

    @FunctionalInterface
    private static interface NextProcessingStep {
        public void run() throws Exception;
    }

    private record BatchProcessingStepResult(List<TypedRecord<?>> toProcess, List<LogAppendEntry> toWrite) {
    }
}

