/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.backup.gcs;

import com.google.cloud.storage.BucketInfo;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageOptions;
import io.camunda.zeebe.backup.api.Backup;
import io.camunda.zeebe.backup.api.BackupDescriptor;
import io.camunda.zeebe.backup.api.BackupIdentifier;
import io.camunda.zeebe.backup.api.BackupIdentifierWildcard;
import io.camunda.zeebe.backup.api.BackupStatus;
import io.camunda.zeebe.backup.api.BackupStatusCode;
import io.camunda.zeebe.backup.api.BackupStore;
import io.camunda.zeebe.backup.api.NamedFileSet;
import io.camunda.zeebe.backup.common.BackupImpl;
import io.camunda.zeebe.backup.common.BackupStatusImpl;
import io.camunda.zeebe.backup.gcs.FileSetManager;
import io.camunda.zeebe.backup.gcs.GcsBackupConfig;
import io.camunda.zeebe.backup.gcs.GcsBackupStoreException;
import io.camunda.zeebe.backup.gcs.ManifestManager;
import io.camunda.zeebe.backup.gcs.manifest.Manifest;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

public final class GcsBackupStore
implements BackupStore {
    public static final String ERROR_MSG_BACKUP_NOT_FOUND = "Expected to restore from backup with id '%s', but does not exist.";
    public static final String ERROR_MSG_BACKUP_WRONG_STATE_TO_RESTORE = "Expected to restore from completed backup with id '%s', but was in state '%s'";
    public static final String ERROR_MSG_ACCESS_FAILED = "Expected to access bucket '%s', but failed";
    public static final String SNAPSHOT_FILESET_NAME = "snapshot";
    public static final String SEGMENTS_FILESET_NAME = "segments";
    private final ExecutorService executor;
    private final ManifestManager manifestManager;
    private final FileSetManager fileSetManager;
    private final Storage client;

    public GcsBackupStore(GcsBackupConfig config) {
        this(config, GcsBackupStore.buildClient(config));
    }

    public GcsBackupStore(GcsBackupConfig config, Storage client) {
        BucketInfo bucketInfo = BucketInfo.of((String)config.bucketName());
        String basePath = Optional.ofNullable(config.basePath()).map(s -> s + "/").orElse("");
        this.client = client;
        this.executor = Executors.newWorkStealingPool(4);
        this.manifestManager = new ManifestManager(client, bucketInfo, basePath);
        this.fileSetManager = new FileSetManager(client, bucketInfo, basePath);
    }

    public CompletableFuture<Void> save(Backup backup) {
        return CompletableFuture.runAsync(() -> {
            ManifestManager.PersistedManifest persistedManifest = this.manifestManager.createInitialManifest(backup);
            try {
                this.fileSetManager.save(backup.id(), SNAPSHOT_FILESET_NAME, backup.snapshot());
                this.fileSetManager.save(backup.id(), SEGMENTS_FILESET_NAME, backup.segments());
                this.manifestManager.completeManifest(persistedManifest);
            }
            catch (Exception e) {
                this.manifestManager.markAsFailed(persistedManifest.manifest(), e.getMessage());
                throw e;
            }
        }, this.executor);
    }

    public CompletableFuture<BackupStatus> getStatus(BackupIdentifier id) {
        return CompletableFuture.supplyAsync(() -> {
            Manifest manifest = this.manifestManager.getManifest(id);
            if (manifest == null) {
                return BackupStatusImpl.doesNotExist((BackupIdentifier)id);
            }
            return GcsBackupStore.toStatus(manifest);
        }, this.executor);
    }

    public CompletableFuture<Collection<BackupStatus>> list(BackupIdentifierWildcard wildcard) {
        return CompletableFuture.supplyAsync(() -> this.manifestManager.listManifests(wildcard).stream().map(GcsBackupStore::toStatus).toList(), this.executor);
    }

    public CompletableFuture<Void> delete(BackupIdentifier id) {
        return CompletableFuture.runAsync(() -> {
            this.manifestManager.deleteManifest(id);
            this.fileSetManager.delete(id, SNAPSHOT_FILESET_NAME);
            this.fileSetManager.delete(id, SEGMENTS_FILESET_NAME);
        }, this.executor);
    }

    public CompletableFuture<Backup> restore(BackupIdentifier id, Path targetFolder) {
        return CompletableFuture.supplyAsync(() -> {
            Manifest manifest = this.manifestManager.getManifest(id);
            if (manifest == null) {
                throw new RuntimeException(ERROR_MSG_BACKUP_NOT_FOUND.formatted(id));
            }
            switch (manifest.statusCode()) {
                default: {
                    throw new IncompatibleClassChangeError();
                }
                case IN_PROGRESS: 
                case FAILED: {
                    throw new RuntimeException(ERROR_MSG_BACKUP_WRONG_STATE_TO_RESTORE.formatted(new Object[]{id, manifest.statusCode()}));
                }
                case COMPLETED: 
            }
            Manifest.CompletedManifest completed = manifest.asCompleted();
            NamedFileSet snapshot = this.fileSetManager.restore(id, SNAPSHOT_FILESET_NAME, completed.snapshot(), targetFolder);
            NamedFileSet segments = this.fileSetManager.restore(id, SEGMENTS_FILESET_NAME, completed.segments(), targetFolder);
            return new BackupImpl(id, (BackupDescriptor)manifest.descriptor(), snapshot, segments);
        }, this.executor);
    }

    public CompletableFuture<BackupStatusCode> markFailed(BackupIdentifier id, String failureReason) {
        return CompletableFuture.supplyAsync(() -> {
            this.manifestManager.markAsFailed(id, failureReason);
            return BackupStatusCode.FAILED;
        }, this.executor);
    }

    public CompletableFuture<Void> closeAsync() {
        return CompletableFuture.runAsync(() -> {
            try {
                this.executor.shutdown();
                boolean closed = this.executor.awaitTermination(1L, TimeUnit.MINUTES);
                if (!closed) {
                    this.executor.shutdownNow();
                }
                this.client.close();
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
    }

    private static BackupStatus toStatus(Manifest manifest) {
        return switch (manifest.statusCode()) {
            default -> throw new IncompatibleClassChangeError();
            case Manifest.StatusCode.IN_PROGRESS -> new BackupStatusImpl((BackupIdentifier)manifest.id(), Optional.ofNullable(manifest.descriptor()), BackupStatusCode.IN_PROGRESS, Optional.empty(), Optional.ofNullable(manifest.createdAt()), Optional.ofNullable(manifest.modifiedAt()));
            case Manifest.StatusCode.COMPLETED -> new BackupStatusImpl((BackupIdentifier)manifest.id(), Optional.ofNullable(manifest.descriptor()), BackupStatusCode.COMPLETED, Optional.empty(), Optional.ofNullable(manifest.createdAt()), Optional.ofNullable(manifest.modifiedAt()));
            case Manifest.StatusCode.FAILED -> new BackupStatusImpl((BackupIdentifier)manifest.id(), Optional.ofNullable(manifest.descriptor()), BackupStatusCode.FAILED, Optional.ofNullable(manifest.asFailed().failureReason()), Optional.ofNullable(manifest.createdAt()), Optional.ofNullable(manifest.modifiedAt()));
        };
    }

    public static Storage buildClient(GcsBackupConfig config) {
        return (Storage)((StorageOptions.Builder)((StorageOptions.Builder)StorageOptions.newBuilder().setHost(config.connection().host())).setCredentials(config.connection().auth().credentials())).build().getService();
    }

    public static void validateConfig(GcsBackupConfig config) {
        try (Storage storage = GcsBackupStore.buildClient(config);){
            storage.list(config.bucketName(), new Storage.BlobListOption[]{Storage.BlobListOption.pageSize((long)1L)});
        }
        catch (Exception e) {
            throw new GcsBackupStoreException.ConfigurationException.CouldNotAccessBucketException(ERROR_MSG_ACCESS_FAILED.formatted(config.bucketName()), e);
        }
    }
}

