/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.exporter;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.camunda.zeebe.exporter.ElasticsearchClient;
import io.camunda.zeebe.exporter.ElasticsearchExporterConfiguration;
import io.camunda.zeebe.exporter.ElasticsearchExporterException;
import io.camunda.zeebe.exporter.ElasticsearchExporterMetadata;
import io.camunda.zeebe.exporter.ElasticsearchRecordCounters;
import io.camunda.zeebe.exporter.RecordSequence;
import io.camunda.zeebe.exporter.api.Exporter;
import io.camunda.zeebe.exporter.api.ExporterException;
import io.camunda.zeebe.exporter.api.context.Context;
import io.camunda.zeebe.exporter.api.context.Controller;
import io.camunda.zeebe.protocol.record.Record;
import io.camunda.zeebe.protocol.record.RecordType;
import io.camunda.zeebe.protocol.record.ValueType;
import java.io.IOException;
import java.time.Duration;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ElasticsearchExporter
implements Exporter {
    private static final String PATTERN_MIN_AGE_FORMAT = "^[0-9]+[dhms]$";
    private static final Predicate<String> CHECKER_MIN_AGE = Pattern.compile("^[0-9]+[dhms]$").asPredicate();
    private static final int RECOMMENDED_MAX_BULK_MEMORY_LIMIT = 0x6400000;
    private Logger log = LoggerFactory.getLogger((String)this.getClass().getPackageName());
    private final ObjectMapper exporterMetadataObjectMapper = new ObjectMapper();
    private final ElasticsearchExporterMetadata exporterMetadata = new ElasticsearchExporterMetadata();
    private Controller controller;
    private ElasticsearchExporterConfiguration configuration;
    private ElasticsearchClient client;
    private ElasticsearchRecordCounters recordCounters;
    private long lastPosition = -1L;
    private boolean indexTemplatesCreated;

    public void configure(Context context) {
        this.log = context.getLogger();
        this.configuration = (ElasticsearchExporterConfiguration)context.getConfiguration().instantiate(ElasticsearchExporterConfiguration.class);
        this.log.debug("Exporter configured with {}", (Object)this.configuration);
        this.validate(this.configuration);
        context.setFilter((Context.RecordFilter)new ElasticsearchRecordFilter(this.configuration));
    }

    public void open(Controller controller) {
        this.controller = controller;
        this.client = this.createClient();
        this.recordCounters = controller.readMetadata().map(this::deserializeExporterMetadata).map(ElasticsearchExporterMetadata::getRecordCountersByValueType).map(ElasticsearchRecordCounters::new).orElse(new ElasticsearchRecordCounters());
        this.scheduleDelayedFlush();
        this.log.info("Exporter opened");
    }

    public void close() {
        try {
            this.flush();
            this.updateLastExportedPosition();
        }
        catch (Exception e) {
            this.log.warn("Failed to flush records before closing exporter.", (Throwable)e);
        }
        try {
            this.client.close();
        }
        catch (Exception e) {
            this.log.warn("Failed to close elasticsearch client", (Throwable)e);
        }
        this.log.info("Exporter closed");
    }

    public void export(Record<?> record) {
        if (!this.indexTemplatesCreated) {
            this.createIndexTemplates();
        }
        RecordSequence recordSequence = this.recordCounters.getNextRecordSequence(record);
        this.client.index(record, recordSequence);
        this.lastPosition = record.getPosition();
        if (this.client.shouldFlush()) {
            this.flush();
            this.recordCounters.updateRecordCounters(record, recordSequence);
            this.updateLastExportedPosition();
        } else {
            this.recordCounters.updateRecordCounters(record, recordSequence);
        }
    }

    private void validate(ElasticsearchExporterConfiguration configuration) {
        Integer numberOfShards;
        if (configuration.index.prefix != null && configuration.index.prefix.contains("_")) {
            throw new ExporterException(String.format("Elasticsearch prefix must not contain underscore. Current value: %s", configuration.index.prefix));
        }
        if (configuration.bulk.memoryLimit > 0x6400000) {
            this.log.warn("The bulk memory limit is set to more than {} bytes. It is recommended to set the limit between 5 to 15 MB.", (Object)0x6400000);
        }
        if ((numberOfShards = configuration.index.getNumberOfShards()) != null && numberOfShards < 1) {
            throw new ExporterException(String.format("Elasticsearch numberOfShards must be >= 1. Current value: %d", numberOfShards));
        }
        Integer numberOfReplicas = configuration.index.getNumberOfReplicas();
        if (numberOfReplicas != null && numberOfReplicas < 0) {
            throw new ExporterException(String.format("Elasticsearch numberOfReplicas must be >= 0. Current value: %d", numberOfReplicas));
        }
        String minimumAge = configuration.retention.getMinimumAge();
        if (minimumAge != null && !CHECKER_MIN_AGE.test(minimumAge)) {
            throw new ExporterException(String.format("Elasticsearch minimumAge '%s' must match pattern '%s', but didn't.", minimumAge, PATTERN_MIN_AGE_FORMAT));
        }
    }

    protected ElasticsearchClient createClient() {
        return new ElasticsearchClient(this.configuration);
    }

    private void flushAndReschedule() {
        try {
            this.flush();
            this.updateLastExportedPosition();
        }
        catch (Exception e) {
            this.log.warn("Unexpected exception occurred on periodically flushing bulk, will retry later.", (Throwable)e);
        }
        this.scheduleDelayedFlush();
    }

    private void scheduleDelayedFlush() {
        this.controller.scheduleCancellableTask(Duration.ofSeconds(this.configuration.bulk.delay), this::flushAndReschedule);
    }

    private void flush() {
        this.client.flush();
    }

    private void updateLastExportedPosition() {
        this.exporterMetadata.setRecordCountersByValueType(this.recordCounters.getRecordCounters());
        byte[] serializeExporterMetadata = this.serializeExporterMetadata(this.exporterMetadata);
        this.controller.updateLastExportedRecordPosition(this.lastPosition, serializeExporterMetadata);
    }

    private byte[] serializeExporterMetadata(ElasticsearchExporterMetadata metadata) {
        try {
            return this.exporterMetadataObjectMapper.writeValueAsBytes((Object)metadata);
        }
        catch (JsonProcessingException e) {
            throw new ElasticsearchExporterException("Failed to serialize exporter metadata", e);
        }
    }

    private ElasticsearchExporterMetadata deserializeExporterMetadata(byte[] metadata) {
        try {
            return (ElasticsearchExporterMetadata)this.exporterMetadataObjectMapper.readValue(metadata, ElasticsearchExporterMetadata.class);
        }
        catch (IOException e) {
            throw new ElasticsearchExporterException("Failed to deserialize exporter metadata", e);
        }
    }

    private void createIndexTemplates() {
        if (this.configuration.retention.isEnabled()) {
            this.createIndexLifecycleManagementPolicy();
        }
        ElasticsearchExporterConfiguration.IndexConfiguration index = this.configuration.index;
        if (index.createTemplate) {
            this.createComponentTemplate();
            if (index.deployment) {
                this.createValueIndexTemplate(ValueType.DEPLOYMENT);
            }
            if (index.process) {
                this.createValueIndexTemplate(ValueType.PROCESS);
            }
            if (index.error) {
                this.createValueIndexTemplate(ValueType.ERROR);
            }
            if (index.incident) {
                this.createValueIndexTemplate(ValueType.INCIDENT);
            }
            if (index.job) {
                this.createValueIndexTemplate(ValueType.JOB);
            }
            if (index.jobBatch) {
                this.createValueIndexTemplate(ValueType.JOB_BATCH);
            }
            if (index.message) {
                this.createValueIndexTemplate(ValueType.MESSAGE);
            }
            if (index.messageBatch) {
                this.createValueIndexTemplate(ValueType.MESSAGE_BATCH);
            }
            if (index.messageSubscription) {
                this.createValueIndexTemplate(ValueType.MESSAGE_SUBSCRIPTION);
            }
            if (index.variable) {
                this.createValueIndexTemplate(ValueType.VARIABLE);
            }
            if (index.variableDocument) {
                this.createValueIndexTemplate(ValueType.VARIABLE_DOCUMENT);
            }
            if (index.processInstance) {
                this.createValueIndexTemplate(ValueType.PROCESS_INSTANCE);
            }
            if (index.processInstanceBatch) {
                this.createValueIndexTemplate(ValueType.PROCESS_INSTANCE_BATCH);
            }
            if (index.processInstanceCreation) {
                this.createValueIndexTemplate(ValueType.PROCESS_INSTANCE_CREATION);
            }
            if (index.processInstanceModification) {
                this.createValueIndexTemplate(ValueType.PROCESS_INSTANCE_MODIFICATION);
            }
            if (index.processMessageSubscription) {
                this.createValueIndexTemplate(ValueType.PROCESS_MESSAGE_SUBSCRIPTION);
            }
            if (index.decisionRequirements) {
                this.createValueIndexTemplate(ValueType.DECISION_REQUIREMENTS);
            }
            if (index.decision) {
                this.createValueIndexTemplate(ValueType.DECISION);
            }
            if (index.decisionEvaluation) {
                this.createValueIndexTemplate(ValueType.DECISION_EVALUATION);
            }
            if (index.checkpoint) {
                this.createValueIndexTemplate(ValueType.CHECKPOINT);
            }
            if (index.timer) {
                this.createValueIndexTemplate(ValueType.TIMER);
            }
            if (index.messageStartEventSubscription) {
                this.createValueIndexTemplate(ValueType.MESSAGE_START_EVENT_SUBSCRIPTION);
            }
            if (index.processEvent) {
                this.createValueIndexTemplate(ValueType.PROCESS_EVENT);
            }
            if (index.deploymentDistribution) {
                this.createValueIndexTemplate(ValueType.DEPLOYMENT_DISTRIBUTION);
            }
            if (index.escalation) {
                this.createValueIndexTemplate(ValueType.ESCALATION);
            }
            if (index.signal) {
                this.createValueIndexTemplate(ValueType.SIGNAL);
            }
            if (index.signalSubscription) {
                this.createValueIndexTemplate(ValueType.SIGNAL_SUBSCRIPTION);
            }
            if (index.resourceDeletion) {
                this.createValueIndexTemplate(ValueType.RESOURCE_DELETION);
            }
            if (index.commandDistribution) {
                this.createValueIndexTemplate(ValueType.COMMAND_DISTRIBUTION);
            }
            if (index.form) {
                this.createValueIndexTemplate(ValueType.FORM);
            }
        }
        this.indexTemplatesCreated = true;
    }

    private void createIndexLifecycleManagementPolicy() {
        if (!this.client.putIndexLifecycleManagementPolicy()) {
            this.log.warn("Failed to acknowledge the creation or update of the Index Lifecycle Management Policy");
        }
    }

    private void createComponentTemplate() {
        if (!this.client.putComponentTemplate()) {
            this.log.warn("Failed to acknowledge the creation or update of the component template");
        }
    }

    private void createValueIndexTemplate(ValueType valueType) {
        if (!this.client.putIndexTemplate(valueType)) {
            this.log.warn("Failed to acknowledge the creation or update of the index template for value type {}", (Object)valueType);
        }
    }

    private static class ElasticsearchRecordFilter
    implements Context.RecordFilter {
        private final ElasticsearchExporterConfiguration configuration;

        ElasticsearchRecordFilter(ElasticsearchExporterConfiguration configuration) {
            this.configuration = configuration;
        }

        public boolean acceptType(RecordType recordType) {
            return this.configuration.shouldIndexRecordType(recordType);
        }

        public boolean acceptValue(ValueType valueType) {
            return this.configuration.shouldIndexValueType(valueType);
        }
    }
}

