/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.broker.exporter.metrics;

import io.camunda.zeebe.broker.exporter.metrics.ExecutionLatencyMetrics;
import io.camunda.zeebe.broker.exporter.metrics.TtlKeyCache;
import io.camunda.zeebe.broker.system.configuration.ExporterCfg;
import io.camunda.zeebe.exporter.api.Exporter;
import io.camunda.zeebe.exporter.api.context.Context;
import io.camunda.zeebe.exporter.api.context.Controller;
import io.camunda.zeebe.protocol.record.Record;
import io.camunda.zeebe.protocol.record.RecordType;
import io.camunda.zeebe.protocol.record.ValueType;
import io.camunda.zeebe.protocol.record.intent.Intent;
import io.camunda.zeebe.protocol.record.intent.JobBatchIntent;
import io.camunda.zeebe.protocol.record.intent.JobIntent;
import io.camunda.zeebe.protocol.record.intent.ProcessInstanceIntent;
import io.camunda.zeebe.protocol.record.value.BpmnElementType;
import io.camunda.zeebe.protocol.record.value.JobBatchRecordValue;
import io.camunda.zeebe.protocol.record.value.ProcessInstanceRecordValue;
import io.camunda.zeebe.scheduler.clock.ActorClock;
import java.time.Duration;
import java.util.Iterator;
import java.util.Set;

public class MetricsExporter
implements Exporter {
    public static final Duration TIME_TO_LIVE = Duration.ofSeconds(10L);
    private final ExecutionLatencyMetrics executionLatencyMetrics;
    private final TtlKeyCache processInstanceCache;
    private final TtlKeyCache jobCache;
    private Controller controller;

    public MetricsExporter() {
        this(new ExecutionLatencyMetrics());
    }

    public MetricsExporter(ExecutionLatencyMetrics executionLatencyMetrics) {
        this(executionLatencyMetrics, new TtlKeyCache(TIME_TO_LIVE.toMillis()), new TtlKeyCache(TIME_TO_LIVE.toMillis()));
    }

    MetricsExporter(ExecutionLatencyMetrics executionLatencyMetrics, TtlKeyCache processInstanceCache, TtlKeyCache jobCache) {
        this.executionLatencyMetrics = executionLatencyMetrics;
        this.processInstanceCache = processInstanceCache;
        this.jobCache = jobCache;
    }

    public void configure(Context context) throws Exception {
        context.setFilter(new Context.RecordFilter(){
            private static final Set<ValueType> ACCEPTED_VALUE_TYPES = Set.of(ValueType.JOB, ValueType.JOB_BATCH, ValueType.PROCESS_INSTANCE);

            public boolean acceptType(RecordType recordType) {
                return recordType == RecordType.EVENT;
            }

            public boolean acceptValue(ValueType valueType) {
                return ACCEPTED_VALUE_TYPES.contains(valueType);
            }
        });
    }

    public void open(Controller controller) {
        this.controller = controller;
        controller.scheduleCancellableTask(TIME_TO_LIVE, this::cleanUp);
    }

    public void close() {
        this.processInstanceCache.clear();
        this.jobCache.clear();
    }

    public void export(Record<?> record) {
        if (record.getRecordType() != RecordType.EVENT) {
            this.controller.updateLastExportedRecordPosition(record.getPosition());
            return;
        }
        int partitionId = record.getPartitionId();
        long recordKey = record.getKey();
        ValueType currentValueType = record.getValueType();
        if (currentValueType == ValueType.JOB) {
            this.handleJobRecord(record, partitionId, recordKey);
        } else if (currentValueType == ValueType.JOB_BATCH) {
            this.handleJobBatchRecord(record, partitionId);
        } else if (currentValueType == ValueType.PROCESS_INSTANCE) {
            this.handleProcessInstanceRecord(record, partitionId, recordKey);
        }
        this.controller.updateLastExportedRecordPosition(record.getPosition());
    }

    private void handleProcessInstanceRecord(Record<?> record, int partitionId, long recordKey) {
        Intent currentIntent = record.getIntent();
        if (currentIntent == ProcessInstanceIntent.ELEMENT_ACTIVATING && MetricsExporter.isProcessInstanceRecord(record)) {
            this.processInstanceCache.store(recordKey, record.getTimestamp());
        } else if (currentIntent == ProcessInstanceIntent.ELEMENT_COMPLETED && MetricsExporter.isProcessInstanceRecord(record)) {
            long creationTime = this.processInstanceCache.remove(recordKey);
            this.executionLatencyMetrics.observeProcessInstanceExecutionTime(partitionId, creationTime, record.getTimestamp());
        }
    }

    private void handleJobRecord(Record<?> record, int partitionId, long recordKey) {
        Intent currentIntent = record.getIntent();
        if (currentIntent == JobIntent.CREATED) {
            this.jobCache.store(recordKey, record.getTimestamp());
        } else if (currentIntent == JobIntent.COMPLETED) {
            long creationTime = this.jobCache.remove(recordKey);
            this.executionLatencyMetrics.observeJobLifeTime(partitionId, creationTime, record.getTimestamp());
        }
    }

    private void handleJobBatchRecord(Record<?> record, int partitionId) {
        Intent currentIntent = record.getIntent();
        if (currentIntent == JobBatchIntent.ACTIVATED) {
            JobBatchRecordValue value = (JobBatchRecordValue)record.getValue();
            Iterator iterator = value.getJobKeys().iterator();
            while (iterator.hasNext()) {
                long jobKey = (Long)iterator.next();
                long creationTime = this.jobCache.remove(jobKey);
                this.executionLatencyMetrics.observeJobActivationTime(partitionId, creationTime, record.getTimestamp());
            }
        }
    }

    private void cleanUp() {
        long currentTimeMillis = ActorClock.currentTimeMillis();
        long deadTime = currentTimeMillis - TIME_TO_LIVE.toMillis();
        this.processInstanceCache.cleanup(deadTime);
        this.jobCache.cleanup(deadTime);
        this.controller.scheduleCancellableTask(TIME_TO_LIVE, this::cleanUp);
    }

    public static ExporterCfg defaultConfig() {
        ExporterCfg exporterCfg = new ExporterCfg();
        exporterCfg.setClassName(MetricsExporter.class.getName());
        return exporterCfg;
    }

    public static String defaultExporterId() {
        return MetricsExporter.class.getSimpleName();
    }

    private static boolean isProcessInstanceRecord(Record<?> record) {
        ProcessInstanceRecordValue recordValue = (ProcessInstanceRecordValue)record.getValue();
        return BpmnElementType.PROCESS == recordValue.getBpmnElementType();
    }
}

