/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.broker.exporter.stream;

import io.camunda.zeebe.broker.Loggers;
import io.camunda.zeebe.broker.exporter.stream.ExporterContainer;
import io.camunda.zeebe.broker.exporter.stream.ExporterDirectorContext;
import io.camunda.zeebe.broker.exporter.stream.ExporterMetrics;
import io.camunda.zeebe.broker.exporter.stream.ExporterPhase;
import io.camunda.zeebe.broker.exporter.stream.ExporterStateDistributeMessage;
import io.camunda.zeebe.broker.exporter.stream.ExporterStateDistributionService;
import io.camunda.zeebe.broker.exporter.stream.ExportersState;
import io.camunda.zeebe.broker.system.partitions.PartitionMessagingService;
import io.camunda.zeebe.db.ZeebeDb;
import io.camunda.zeebe.logstreams.log.LogRecordAwaiter;
import io.camunda.zeebe.logstreams.log.LogStream;
import io.camunda.zeebe.logstreams.log.LogStreamReader;
import io.camunda.zeebe.logstreams.log.LoggedEvent;
import io.camunda.zeebe.protocol.ZbColumnFamilies;
import io.camunda.zeebe.protocol.impl.record.RecordMetadata;
import io.camunda.zeebe.protocol.impl.record.UnifiedRecordValue;
import io.camunda.zeebe.protocol.record.RecordType;
import io.camunda.zeebe.protocol.record.ValueType;
import io.camunda.zeebe.scheduler.Actor;
import io.camunda.zeebe.scheduler.ActorControl;
import io.camunda.zeebe.scheduler.ActorSchedulingService;
import io.camunda.zeebe.scheduler.SchedulingHints;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import io.camunda.zeebe.scheduler.future.CompletableActorFuture;
import io.camunda.zeebe.scheduler.retry.BackOffRetryStrategy;
import io.camunda.zeebe.scheduler.retry.EndlessRetryStrategy;
import io.camunda.zeebe.scheduler.retry.RetryStrategy;
import io.camunda.zeebe.stream.api.EventFilter;
import io.camunda.zeebe.stream.api.records.TypedRecord;
import io.camunda.zeebe.stream.impl.records.RecordValues;
import io.camunda.zeebe.stream.impl.records.TypedRecordImpl;
import io.camunda.zeebe.util.buffer.BufferReader;
import io.camunda.zeebe.util.exception.UnrecoverableException;
import io.camunda.zeebe.util.health.FailureListener;
import io.camunda.zeebe.util.health.HealthMonitorable;
import io.camunda.zeebe.util.health.HealthReport;
import java.time.Duration;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.agrona.LangUtil;
import org.slf4j.Logger;

public final class ExporterDirector
extends Actor
implements HealthMonitorable,
LogRecordAwaiter {
    private static final String ERROR_MESSAGE_EXPORTING_ABORTED = "Expected to export record '{}' successfully, but exception was thrown.";
    private static final String ERROR_MESSAGE_RECOVER_FROM_SNAPSHOT_FAILED = "Expected to find event with the snapshot position %s in log stream, but nothing was found. Failed to recover '%s'.";
    private static final String EXPORTER_STATE_TOPIC_FORMAT = "exporterState-%d";
    private static final Logger LOG = Loggers.EXPORTER_LOGGER;
    private final AtomicBoolean isOpened = new AtomicBoolean(false);
    private final List<ExporterContainer> containers;
    private final LogStream logStream;
    private final RecordExporter recordExporter;
    private final ZeebeDb zeebeDb;
    private final ExporterMetrics metrics;
    private final String name;
    private final RetryStrategy exportingRetryStrategy;
    private final RetryStrategy recordWrapStrategy;
    private final Set<FailureListener> listeners = new HashSet<FailureListener>();
    private LogStreamReader logStreamReader;
    private EventFilter eventFilter;
    private ExportersState state;
    private volatile HealthReport healthReport = HealthReport.healthy((HealthMonitorable)this);
    private boolean inExportingPhase;
    private boolean isPaused;
    private ExporterPhase exporterPhase;
    private final PartitionMessagingService partitionMessagingService;
    private final String exporterPositionsTopic;
    private final ExporterDirectorContext.ExporterMode exporterMode;
    private final Duration distributionInterval;
    private ExporterStateDistributionService exporterDistributionService;
    private final int partitionId;
    private final EventFilter positionsToSkipFilter;

    public ExporterDirector(ExporterDirectorContext context, boolean shouldPauseOnStart) {
        this.name = context.getName();
        this.containers = context.getDescriptors().stream().map(ExporterContainer::new).collect(Collectors.toList());
        this.logStream = Objects.requireNonNull(context.getLogStream());
        this.partitionId = this.logStream.getPartitionId();
        this.metrics = new ExporterMetrics(this.partitionId);
        this.recordExporter = new RecordExporter(this.metrics, this.containers, this.partitionId);
        this.exportingRetryStrategy = new BackOffRetryStrategy(this.actor, Duration.ofSeconds(10L));
        this.recordWrapStrategy = new EndlessRetryStrategy(this.actor);
        this.zeebeDb = context.getZeebeDb();
        this.isPaused = shouldPauseOnStart;
        this.partitionMessagingService = context.getPartitionMessagingService();
        this.exporterPositionsTopic = String.format(EXPORTER_STATE_TOPIC_FORMAT, this.partitionId);
        this.exporterMode = context.getExporterMode();
        this.distributionInterval = context.getDistributionInterval();
        this.positionsToSkipFilter = context.getPositionsToSkipFilter();
    }

    public ActorFuture<Void> startAsync(ActorSchedulingService actorSchedulingService) {
        return actorSchedulingService.submitActor((Actor)this, SchedulingHints.ioBound());
    }

    public ActorFuture<Void> stopAsync() {
        return this.actor.close();
    }

    public ActorFuture<Void> pauseExporting() {
        if (this.actor.isClosed()) {
            return CompletableActorFuture.completed(null);
        }
        return this.actor.call(() -> {
            this.isPaused = true;
            this.exporterPhase = ExporterPhase.PAUSED;
        });
    }

    public ActorFuture<Void> resumeExporting() {
        if (this.actor.isClosed()) {
            return CompletableActorFuture.completed(null);
        }
        return this.actor.call(() -> {
            this.isPaused = false;
            this.exporterPhase = ExporterPhase.EXPORTING;
            if (this.exporterMode == ExporterDirectorContext.ExporterMode.ACTIVE) {
                this.actor.submit(this::readNextEvent);
            }
        });
    }

    public ActorFuture<ExporterPhase> getPhase() {
        if (this.actor.isClosed()) {
            return CompletableActorFuture.completed((Object)((Object)ExporterPhase.CLOSED));
        }
        return this.actor.call(() -> this.exporterPhase);
    }

    protected Map<String, String> createContext() {
        Map context = super.createContext();
        context.put("partitionId", Integer.toString(this.partitionId));
        return context;
    }

    public String getName() {
        return this.name;
    }

    protected void onActorStarting() {
        if (this.exporterMode == ExporterDirectorContext.ExporterMode.ACTIVE) {
            ActorFuture newReaderFuture = this.logStream.newLogStreamReader();
            this.actor.runOnCompletionBlockingCurrentPhase(newReaderFuture, (reader, errorOnReceivingReader) -> {
                if (errorOnReceivingReader == null) {
                    this.logStreamReader = reader;
                } else {
                    LOG.error("Unexpected error on retrieving reader from log {}", (Object)this.logStream.getLogName(), errorOnReceivingReader);
                    this.actor.close();
                }
            });
        }
    }

    protected void onActorStarted() {
        try {
            LOG.debug("Recovering exporter from snapshot");
            this.recoverFromSnapshot();
            this.exporterDistributionService = new ExporterStateDistributionService(this::consumeExporterStateFromLeader, this.partitionMessagingService, this.exporterPositionsTopic);
            this.initContainers();
        }
        catch (Exception e) {
            this.onFailure();
            LangUtil.rethrowUnchecked((Throwable)e);
        }
        this.isOpened.set(true);
        this.clearExporterState();
        if (this.exporterMode == ExporterDirectorContext.ExporterMode.ACTIVE) {
            this.startActiveExportingMode();
        } else {
            this.startPassiveExportingMode();
        }
    }

    protected void onActorClosing() {
        if (this.logStreamReader != null) {
            this.logStreamReader.close();
        }
        this.logStream.removeRecordAvailableListener((LogRecordAwaiter)this);
    }

    protected void onActorClosed() {
        LOG.debug("Closed exporter director '{}'.", (Object)this.getName());
        this.exporterPhase = ExporterPhase.CLOSED;
    }

    protected void onActorCloseRequested() {
        this.isOpened.set(false);
        if (this.exporterMode == ExporterDirectorContext.ExporterMode.ACTIVE) {
            this.containers.forEach(ExporterContainer::close);
        } else {
            this.exporterDistributionService.close();
        }
    }

    protected void handleFailure(Throwable failure) {
        LOG.error("Actor '{}' failed in phase {} with: {} .", new Object[]{this.name, this.actor.getLifecyclePhase(), failure, failure});
        this.actor.fail(failure);
        if (failure instanceof UnrecoverableException) {
            this.healthReport = HealthReport.dead((HealthMonitorable)this).withIssue(failure);
            for (FailureListener listener : this.listeners) {
                listener.onUnrecoverableFailure(this.healthReport);
            }
        } else {
            this.healthReport = HealthReport.unhealthy((HealthMonitorable)this).withIssue(failure);
            for (FailureListener listener : this.listeners) {
                listener.onFailure(this.healthReport);
            }
        }
    }

    private void consumeExporterStateFromLeader(String exporterId, ExporterStateDistributeMessage.ExporterStateEntry exporterState) {
        if (this.state.getPosition(exporterId) < exporterState.position()) {
            this.state.setExporterState(exporterId, exporterState.position(), exporterState.metadata());
        }
    }

    private void initContainers() throws Exception {
        for (ExporterContainer container : this.containers) {
            container.initContainer(this.actor, this.metrics, this.state);
            container.configureExporter();
        }
        this.eventFilter = this.positionsToSkipFilter.and((EventFilter)this.createEventFilter(this.containers));
        LOG.debug("Set event filter for exporters: {}", (Object)this.eventFilter);
    }

    private void recoverFromSnapshot() {
        this.state = new ExportersState((ZeebeDb<ZbColumnFamilies>)this.zeebeDb, this.zeebeDb.createContext());
        long snapshotPosition = this.state.getLowestPosition();
        LOG.debug("Recovered exporter '{}' from snapshot at lastExportedPosition {}", (Object)this.getName(), (Object)snapshotPosition);
    }

    private ExporterEventFilter createEventFilter(List<ExporterContainer> containers) {
        List recordFilters = containers.stream().map(c -> c.getContext().getFilter()).collect(Collectors.toList());
        Map<RecordType, Boolean> acceptRecordTypes = Arrays.stream(RecordType.values()).collect(Collectors.toMap(Function.identity(), type -> recordFilters.stream().anyMatch(f -> f.acceptType(type))));
        Map<ValueType, Boolean> acceptValueTypes = Arrays.stream(ValueType.values()).collect(Collectors.toMap(Function.identity(), type -> recordFilters.stream().anyMatch(f -> f.acceptValue(type))));
        return new ExporterEventFilter(acceptRecordTypes, acceptValueTypes);
    }

    private void onFailure() {
        this.isOpened.set(false);
        this.actor.close();
    }

    private void startActiveExportingMode() {
        this.logStream.registerRecordAvailableListener((LogRecordAwaiter)this);
        for (ExporterContainer container : this.containers) {
            container.initPosition();
            container.openExporter();
        }
        if (this.state.hasExporters()) {
            boolean failedToRecoverReader;
            long snapshotPosition = this.state.getLowestPosition();
            boolean bl = failedToRecoverReader = !this.logStreamReader.seekToNextEvent(snapshotPosition);
            if (failedToRecoverReader) {
                throw new IllegalStateException(String.format(ERROR_MESSAGE_RECOVER_FROM_SNAPSHOT_FAILED, snapshotPosition, this.getName()));
            }
            if (!this.isPaused) {
                this.exporterPhase = ExporterPhase.EXPORTING;
                this.actor.submit(this::readNextEvent);
            } else {
                this.exporterPhase = ExporterPhase.PAUSED;
            }
            this.actor.runAtFixedRate(this.distributionInterval, this::distributeExporterState);
        } else {
            this.actor.close();
        }
    }

    private void startPassiveExportingMode() {
        for (ExporterContainer container : this.containers) {
            container.initPosition();
        }
        if (this.state.hasExporters()) {
            this.exporterDistributionService.subscribeForExporterState(arg_0 -> ((ActorControl)this.actor).run(arg_0));
        } else {
            this.actor.close();
        }
    }

    private void distributeExporterState() {
        ExporterStateDistributeMessage exporterStateMessage = new ExporterStateDistributeMessage();
        this.state.visitExporterState((exporterId, exporterStateEntry) -> exporterStateMessage.putExporter((String)exporterId, exporterStateEntry.getPosition(), exporterStateEntry.getMetadata()));
        this.exporterDistributionService.distributeExporterState(exporterStateMessage);
    }

    private void skipRecord(LoggedEvent currentEvent) {
        RecordMetadata metadata = new RecordMetadata();
        long eventPosition = currentEvent.getPosition();
        currentEvent.readMetadata((BufferReader)metadata);
        this.metrics.eventSkipped(metadata.getValueType());
        for (ExporterContainer container : this.containers) {
            container.updatePositionOnSkipIfUpToDate(eventPosition);
        }
        this.actor.submit(this::readNextEvent);
    }

    private void readNextEvent() {
        if (this.shouldExport()) {
            LoggedEvent currentEvent = (LoggedEvent)this.logStreamReader.next();
            if (this.eventFilter == null || this.eventFilter.applies(currentEvent)) {
                this.inExportingPhase = true;
                this.exportEvent(currentEvent);
            } else {
                this.skipRecord(currentEvent);
            }
        }
    }

    private boolean shouldExport() {
        return this.isOpened.get() && this.logStreamReader.hasNext() && !this.inExportingPhase && !this.isPaused;
    }

    private void exportEvent(LoggedEvent event) {
        ActorFuture wrapRetryFuture = this.recordWrapStrategy.runWithRetry(() -> {
            this.recordExporter.wrap(event);
            return true;
        }, this::isClosed);
        this.actor.runOnCompletion(wrapRetryFuture, (b, t) -> {
            assert (t == null) : "Throwable must be null";
            ActorFuture retryFuture = this.exportingRetryStrategy.runWithRetry(this.recordExporter::export, this::isClosed);
            this.actor.runOnCompletion(retryFuture, (bool, throwable) -> {
                if (throwable != null) {
                    LOG.error(ERROR_MESSAGE_EXPORTING_ABORTED, (Object)event, throwable);
                    this.onFailure();
                } else {
                    this.metrics.eventExported(this.recordExporter.getTypedEvent().getValueType());
                    this.inExportingPhase = false;
                    this.actor.submit(this::readNextEvent);
                }
            });
        });
    }

    private void clearExporterState() {
        List exporterIds = this.containers.stream().map(ExporterContainer::getId).collect(Collectors.toList());
        this.state.visitExporterState((exporterId, exporterStateEntry) -> {
            if (!exporterIds.contains(exporterId)) {
                this.state.removeExporterState((String)exporterId);
                LOG.info("The exporter '{}' is not configured anymore. Its lastExportedPosition is removed from the state.", exporterId);
            }
        });
    }

    private boolean isClosed() {
        return !this.isOpened.get();
    }

    public HealthReport getHealthReport() {
        return this.healthReport;
    }

    public void addFailureListener(FailureListener listener) {
        this.actor.run(() -> this.listeners.add(listener));
    }

    public void removeFailureListener(FailureListener failureListener) {
        this.actor.run(() -> this.listeners.remove(failureListener));
    }

    public void onRecordAvailable() {
        this.actor.run(this::readNextEvent);
    }

    public ActorFuture<Long> getLowestPosition() {
        if (this.actor.isClosed()) {
            return CompletableActorFuture.completed((Object)-1L);
        }
        return this.actor.call(() -> this.state.getLowestPosition());
    }

    private static class RecordExporter {
        private final RecordValues recordValues = new RecordValues();
        private final RecordMetadata rawMetadata = new RecordMetadata();
        private final List<ExporterContainer> containers;
        private final TypedRecordImpl typedEvent;
        private final ExporterMetrics exporterMetrics;
        private boolean shouldExport;
        private int exporterIndex;

        RecordExporter(ExporterMetrics exporterMetrics, List<ExporterContainer> containers, int partitionId) {
            this.containers = containers;
            this.typedEvent = new TypedRecordImpl(partitionId);
            this.exporterMetrics = exporterMetrics;
        }

        void wrap(LoggedEvent rawEvent) {
            rawEvent.readMetadata((BufferReader)this.rawMetadata);
            UnifiedRecordValue recordValue = this.recordValues.readRecordValue(rawEvent, this.rawMetadata.getValueType());
            boolean bl = this.shouldExport = recordValue != null;
            if (this.shouldExport) {
                this.typedEvent.wrap(rawEvent, this.rawMetadata, recordValue);
                this.exporterIndex = 0;
            }
        }

        public boolean export() {
            if (!this.shouldExport) {
                return true;
            }
            int exportersCount = this.containers.size();
            while (this.exporterIndex < exportersCount) {
                ExporterContainer container = this.containers.get(this.exporterIndex);
                if (container.exportRecord(this.rawMetadata, (TypedRecord)this.typedEvent)) {
                    ++this.exporterIndex;
                    this.exporterMetrics.setLastExportedPosition(container.getId(), this.typedEvent.getPosition());
                    continue;
                }
                return false;
            }
            return true;
        }

        TypedRecordImpl getTypedEvent() {
            return this.typedEvent;
        }
    }

    private static class ExporterEventFilter
    implements EventFilter {
        private final RecordMetadata metadata = new RecordMetadata();
        private final Map<RecordType, Boolean> acceptRecordTypes;
        private final Map<ValueType, Boolean> acceptValueTypes;

        ExporterEventFilter(Map<RecordType, Boolean> acceptRecordTypes, Map<ValueType, Boolean> acceptValueTypes) {
            this.acceptRecordTypes = acceptRecordTypes;
            this.acceptValueTypes = acceptValueTypes;
        }

        public boolean applies(LoggedEvent event) {
            event.readMetadata((BufferReader)this.metadata);
            RecordType recordType = this.metadata.getRecordType();
            ValueType valueType = this.metadata.getValueType();
            return this.acceptRecordTypes.get(recordType) != false && this.acceptValueTypes.get(valueType) != false;
        }

        public String toString() {
            return "ExporterEventFilter{acceptRecordTypes=" + this.acceptRecordTypes + ", acceptValueTypes=" + this.acceptValueTypes + "}";
        }
    }
}

