/*
 * Copyright Camunda Services GmbH and/or licensed to Camunda Services GmbH under
 * one or more contributor license agreements. Licensed under a proprietary license. See the
 * License.txt file for more information. You may not use this file except in compliance with the
 * proprietary license.
 */

package io.camunda.identity.sdk.authentication;

import static org.apache.commons.lang3.StringUtils.isNoneBlank;

import com.auth0.jwk.InvalidPublicKeyException;
import com.auth0.jwk.Jwk;
import com.auth0.jwk.JwkException;
import com.auth0.jwk.JwkProvider;
import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTVerifier;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.exceptions.AlgorithmMismatchException;
import com.auth0.jwt.exceptions.SignatureVerificationException;
import com.auth0.jwt.interfaces.DecodedJWT;
import io.camunda.identity.sdk.IdentityConfiguration;
import io.camunda.identity.sdk.authentication.exception.InvalidClaimException;
import io.camunda.identity.sdk.authentication.exception.InvalidSignatureException;
import io.camunda.identity.sdk.authentication.exception.JsonWebKeyException;
import io.camunda.identity.sdk.authentication.exception.TokenDecodeException;
import io.camunda.identity.sdk.authentication.exception.TokenExpiredException;
import io.camunda.identity.sdk.authentication.exception.TokenVerificationException;
import io.camunda.identity.sdk.cache.ClientTokenCache;
import io.camunda.identity.sdk.exception.IdentityException;
import io.camunda.identity.sdk.impl.dto.WellKnownConfiguration;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.ehcache.Cache;

/**
 * The Authentication class provides functionality to authenticate a user with Identity
 * and verify access tokens.
 */
public abstract class AbstractAuthentication implements Authentication {
  public static final long JWKS_CACHE_SIZE = 5L;
  public static final long JWKS_CACHE_LIFETIME_DAYS = 7L;
  public static final String WELL_KNOWN_PATH = "/.well-known/openid-configuration";
  protected final IdentityConfiguration configuration;
  protected final Cache<String, Tokens> tokenCache = new ClientTokenCache().getCache();

  protected AbstractAuthentication(final IdentityConfiguration configuration) {
    this.configuration = configuration;
  }

  @Override
  public boolean isAvailable() {
    return isNoneBlank(
        configuration.getIssuer(),
        configuration.getIssuerBackendUrl(),
        configuration.getClientId(),
        configuration.getClientSecret()
    );
  }

  /**
   * Requests a client token from the cache if available. If
   * no token is found with the required audience, a new token
   * will be requested from the authentication provider and stored.
   *
   * @param audience the audience of the resource server
   * @return the tokens
   * @throws IdentityException if case of a failure
   */
  @Override
  public Tokens requestToken(final String audience) {
    if (!tokenCache.containsKey(audience)) {
      tokenCache.put(audience, requestFreshToken(audience));
    }

    return tokenCache.get(audience);
  }

  /**
   * Decodes a token. Can be used to access tokens data without validation
   *
   * @param token token in JWT format
   * @return decoded token
   * @throws TokenDecodeException the token can not be decoded
   */
  @Override
  public DecodedJWT decodeJWT(final String token) {
    try {
      return JWT.decode(token);
    } catch (final com.auth0.jwt.exceptions.JWTDecodeException e) {
      throw new TokenDecodeException(e);
    }
  }

  @Override
  public AccessToken verifyTokenIgnoringAudience(final String token) {
    return verifyToken(token, null);
  }

  /**
   * Verifies the validity of the passed token. Following checks will be performed:
   * <ul>
   *   <li>The token is correctly signed</li>
   *   <li>The token has not expired</li>
   *   <li>Token's audience (<b>aud</b> claim) matches application's audience</li>
   * </ul>
   *
   * @param token the token
   * @return the decoded jwt
   * @throws TokenDecodeException      the token can not be decoded
   * @throws InvalidSignatureException the token's signature is invalid
   * @throws TokenExpiredException     the token has expired
   * @throws InvalidClaimException     the provided claim is invalid
   * @throws JsonWebKeyException       the JWK needed to verify token's signature can not be
   *                                   retrieved
   */
  @Override
  public AccessToken verifyToken(final String token) {
    return verifyToken(token, configuration.getAudience());
  }

  /**
   * Verifies the validity of the passed token. Following checks will be performed:
   * <ul>
   *   <li>The token is correctly signed</li>
   *   <li>The token has not expired</li>
   *   <li>Token's audience (<b>aud</b> claim) matches provided audience</li>
   * </ul>
   *
   * @param token    the token
   * @param audience token's aud claim must match provided audience
   * @return the decoded jwt
   * @throws TokenDecodeException      the token can not be decoded
   * @throws InvalidSignatureException the token's signature is invalid
   * @throws TokenExpiredException     the token has expired
   * @throws InvalidClaimException     the provided claim is invalid
   * @throws JsonWebKeyException       the JWK needed to verify token's signature can not be
   *                                   retrieved
   */
  protected AccessToken verifyToken(final String token, final String audience) {
    try {
      final DecodedJWT jwt = verify(decodeJWT(token), audience);
      return new AccessToken(
          jwt,
          getPermissions(jwt, audience),
          getAssignedOrganizations(jwt),
          getUserDetails(jwt));
    } catch (final SignatureVerificationException | AlgorithmMismatchException e) {
      throw new InvalidSignatureException(e);
    } catch (final com.auth0.jwt.exceptions.TokenExpiredException e) {
      throw new TokenExpiredException(e);
    } catch (final com.auth0.jwt.exceptions.InvalidClaimException e) {
      throw new InvalidClaimException(e);
    }
  }

  protected UserDetails getUserDetails(final DecodedJWT token) {
    return new UserDetails(
        token.getSubject(),
        token.getClaim("email").asString(),
        token.getClaim("preferred_username").asString(),
        token.getClaim("name").asString(),
        getGroups(token)
    );
  }

  private DecodedJWT verify(final DecodedJWT token, final String audience) {
    try {
      final Jwk jwk = jwkProvider().get(token.getKeyId());
      verifyJwk(token, jwk);

      final Algorithm algorithm = signatureValidationAlgorithm(jwk);
      JWTVerifier tokenVerifier;
      if (audience != null) {
        tokenVerifier = JWT.require(algorithm).withAudience(audience).build();
      } else {
        tokenVerifier = JWT.require(algorithm).build();
      }

      return tokenVerifier.verify(token);
    } catch (final JwkException e) {
      throw new JsonWebKeyException("JWKS error", e);
    }
  }

  private void verifyJwk(final DecodedJWT token, final Jwk jwk) {
    if (jwk.getUsage() != null && !jwk.getUsage().equals("sig")) {
      throw new TokenVerificationException(
          "Token is signed with a JWK, that can not be used for signing");
    }

    if (jwk.getAlgorithm() != null && !jwk.getAlgorithm().equals(token.getAlgorithm())) {
      throw new TokenVerificationException("JWT algorithm does not match JWK algorithm");
    }
  }

  private Algorithm signatureValidationAlgorithm(final Jwk jwk)
      throws InvalidPublicKeyException {
    if (jwk.getAlgorithm() == null || jwk.getAlgorithm().equals("RS256")) {
      return Algorithm.RSA256((RSAPublicKey) jwk.getPublicKey(), null);
    }

    switch (jwk.getAlgorithm()) {
      case "RS384":
        return Algorithm.RSA384((RSAPublicKey) jwk.getPublicKey(), null);
      case "RS512":
        return Algorithm.RSA512((RSAPublicKey) jwk.getPublicKey(), null);
      case "ES256":
        return Algorithm.ECDSA256((ECPublicKey) jwk.getPublicKey(), null);
      case "ES384":
        return Algorithm.ECDSA384((ECPublicKey) jwk.getPublicKey(), null);
      case "ES512":
        return Algorithm.ECDSA512((ECPublicKey) jwk.getPublicKey(), null);
      default:
        throw new TokenVerificationException(
            String.format("Signing algorithm '%s' is not supported", jwk.getAlgorithm()));
    }
  }

  protected abstract List<String> getPermissions(final DecodedJWT token, final String audience);

  protected abstract List<String> getGroups(final DecodedJWT token);

  protected abstract Map<String, Set<String>> getAssignedOrganizations(final DecodedJWT token);

  protected abstract JwkProvider jwkProvider();

  protected abstract WellKnownConfiguration wellKnownConfiguration();

  protected abstract Tokens requestFreshToken(final String audience);
}
