/*
 * Copyright Camunda Services GmbH and/or licensed to Camunda Services GmbH under
 * one or more contributor license agreements. Licensed under a proprietary license. See the
 * License.txt file for more information. You may not use this file except in compliance with the
 * proprietary license.
 */

package io.camunda.identity.sdk.authentication;

import com.auth0.jwt.interfaces.DecodedJWT;
import io.camunda.identity.sdk.IdentityConfiguration;
import io.camunda.identity.sdk.annotation.AnnotationProcessor;
import io.camunda.identity.sdk.annotation.RequiresOAuthCredentials;
import io.camunda.identity.sdk.authentication.dto.AuthCodeDto;
import io.camunda.identity.sdk.authentication.exception.CodeExchangeException;
import io.camunda.identity.sdk.authentication.exception.InvalidClaimException;
import io.camunda.identity.sdk.authentication.exception.InvalidSignatureException;
import io.camunda.identity.sdk.authentication.exception.JsonWebKeyException;
import io.camunda.identity.sdk.authentication.exception.TokenDecodeException;
import io.camunda.identity.sdk.authentication.exception.TokenExpiredException;
import io.camunda.identity.sdk.exception.IdentityException;
import io.camunda.identity.sdk.impl.auth0.authentication.Auth0Authentication;
import io.camunda.identity.sdk.impl.generic.GenericAuthentication;
import io.camunda.identity.sdk.impl.keycloak.KeycloakAuthentication;
import io.camunda.identity.sdk.impl.microsoft.MicrosoftAuthentication;
import io.camunda.identity.sdk.impl.rest.RestClient;

/**
 * The Authentication class provides functionality to authenticate a user with Identity
 * and verify access tokens.
 */
public interface Authentication {
  /**
   * Factory function to create a new authentication instance
   *
   * @param configuration Identity configuration
   * @return Authentication instance
   */
  static Authentication create(final IdentityConfiguration configuration) {
    final Authentication authentication;
    switch (configuration.getType()) {
      case AUTH0:
        authentication = new Auth0Authentication(configuration);
        break;
      case KEYCLOAK:
        authentication = new KeycloakAuthentication(configuration, new RestClient());
        break;
      case MICROSOFT:
        authentication = new MicrosoftAuthentication(configuration, new RestClient());
        break;
      case GENERIC:
        authentication = new GenericAuthentication(configuration, new RestClient());
        break;
      default:
        throw new RuntimeException("not implemented");
    }

    return AnnotationProcessor.apply(configuration, Authentication.class, authentication);
  }

  /**
   * Returns a true/false indicating if authentication is available or not.
   *
   * @return the availability of authentication
   */
  boolean isAvailable();

  /**
   * Returns a new Authorize uri builder. It is used to build the authorize uri
   * for user authentication.
   *
   * @param redirectUri the redirect uri
   * @return the authorize uri builder
   */
  @RequiresOAuthCredentials
  AuthorizeUriBuilder authorizeUriBuilder(final String redirectUri);

  /**
   * Exchanges provided auth code with access tokens. The provided redirect uri argument must match
   * the redirect uri that was used to build the authorize uri.
   *
   * @param authCodeDto the auth code dto
   * @param redirectUri the redirect uri
   * @return the tokens
   * @throws CodeExchangeException if code exchange fails
   */
  @RequiresOAuthCredentials
  Tokens exchangeAuthCode(final AuthCodeDto authCodeDto, final String redirectUri)
      throws CodeExchangeException;

  /**
   * Exchanges provided refresh token with an access tokens.
   *
   * @param refreshToken refresh token used for the request
   * @return the tokens
   * @throws IdentityException if token renewal has failed
   */
  @RequiresOAuthCredentials
  Tokens renewToken(final String refreshToken);

  /**
   * Logs out from Identity backend. This will invalidate the given refresh token
   * and refresh tokens issued for other applications during the same session. The user's session
   * or access token still has to be deleted by the client application.
   *
   * @param refreshToken refresh token used for the request
   * @throws IdentityException if token revocation has failed
   */
  @RequiresOAuthCredentials
  void revokeToken(final String refreshToken);

  /**
   * Requests a client token from the cache if available. If
   * no token is found with the required audience, a new token
   * will be requested from the authentication provider and stored.
   *
   * @param audience the audience of the resource server
   * @return the tokens
   * @throws IdentityException if case of a failure
   */
  @RequiresOAuthCredentials
  Tokens requestToken(final String audience);

  /**
   * Decodes a token. Can be used to access tokens data without validation
   *
   * @param token token in JWT format
   * @return decoded token
   * @throws TokenDecodeException the token can not be decoded
   */
  DecodedJWT decodeJWT(final String token);

  /**
   * Verifies the validity of the passed token. Following checks will be performed:
   * <ul>
   *   <li>The token is correctly signed</li>
   *   <li>The token has not expired</li>
   *   <li>Token's audience (<b>aud</b> claim) matches application's audience</li>
   * </ul>
   *
   * @param token the token
   * @return the decoded jwt
   * @throws TokenDecodeException      the token can not be decoded
   * @throws InvalidSignatureException the token's signature is invalid
   * @throws TokenExpiredException     the token has expired
   * @throws InvalidClaimException     the provided claim is invalid
   * @throws JsonWebKeyException       the JWK needed to verify token's signature can not be
   *                                   retrieved
   */
  AccessToken verifyToken(final String token);

  /**
   * Verifies the validity of the passed token. Following checks will be performed:
   * <ul>
   *   <li>The token is correctly signed</li>
   *   <li>The token has not expired</li>
   * </ul>
   *
   * @param token the token
   * @return the decoded jwt
   * @throws TokenDecodeException      the token can not be decoded
   * @throws InvalidSignatureException the token's signature is invalid
   * @throws TokenExpiredException     the token has expired
   * @throws InvalidClaimException     the provided claim is invalid
   * @throws JsonWebKeyException       the JWK needed to verify token's signature can not be
   *                                   retrieved
   */
  AccessToken verifyTokenIgnoringAudience(final String token);

  /**
   * Checks if the token passed in is considered an M2M token.
   *
   * @param token the token
   * @return true if the token is a M2M token, else false
   */
  boolean isM2MToken(final String token);

  /**
   * Gets the client ID associated to the token
   *
   * @param token the token
   * @return the clientID
   */
  String getClientId(final String token);
}
