/*
 * Copyright Camunda Services GmbH and/or licensed to Camunda Services GmbH under
 * one or more contributor license agreements. Licensed under a proprietary license. See the
 * License.txt file for more information. You may not use this file except in compliance with the
 * proprietary license.
 */
package io.camunda.identity.sdk;

import io.camunda.identity.sdk.authentication.Authentication;
import io.camunda.identity.sdk.authorizations.Authorizations;
import io.camunda.identity.sdk.tenants.Tenants;
import io.camunda.identity.sdk.users.Users;

/**
 * Entry point for the Identity api
 */
public class Identity {
  private final Authentication authentication;
  private final Users users;
  private final Authorizations authorizations;
  private final Tenants tenants;

  /**
   * Instantiates a new Identity API instance.
   *
   * @param configuration the configuration
   */
  public Identity(
      final IdentityConfiguration configuration
  ) {
    this.authentication = Authentication.create(configuration);
    this.users = Users.create(configuration, this.authentication);
    this.authorizations = Authorizations.create(configuration);
    this.tenants = Tenants.create(configuration);
  }

  /**
   * Returns an initialized {@link Authentication} object.<br>
   * It can be used for authentication and verifying tokens issued by Camunda Identity.
   *
   * @return the authentication
   */
  public Authentication authentication() {
    return authentication;
  }

  /**
   * Returns an initialized {@link Users} object.<br>
   * It can be used for retrieving users from the Camunda Identity instance.
   *
   * @return the users
   */
  public Users users() {
    return users;
  }

  /**
   * Returns an initialized {@link Authorizations} object.<br>
   * It can be used for retrieving authorizations from the Camunda Identity instance.
   *
   * @return the authorizations
   */
  public Authorizations authorizations() {
    return authorizations;
  }

  /**
   * Returns an initialized {@link Tenants} object.<br>
   * It can be used for retrieving tenants from the Camunda Identity instance.
   *
   * @return the tenants
   */
  public Tenants tenants() {
    return tenants;
  }
}
