/*
 * Copyright Camunda Services GmbH and/or licensed to Camunda Services GmbH under
 * one or more contributor license agreements. Licensed under a proprietary license. See the
 * License.txt file for more information. You may not use this file except in compliance with the
 * proprietary license.
 */

package io.camunda.identity.sdk;

import org.apache.commons.lang3.Validate;

/**
 * Identity configuration.
 */
public class IdentityConfiguration {
  private final String baseUrl;
  private final String issuer;
  private final String issuerBackendUrl;
  private final String clientId;
  private final String clientSecret;
  private final String audience;
  private final Type type;

  /**
   * Instantiates a new Identity configuration.
   *
   * @param issuer           the issuer
   * @param issuerBackendUrl the issuer url for back-channel communication
   * @param clientId         the client id
   * @param clientSecret     the client secret
   * @param audience         the audience
   */
  public IdentityConfiguration(
      final String issuer,
      final String issuerBackendUrl,
      final String clientId,
      final String clientSecret,
      final String audience
  ) {
    this(null, issuer, issuerBackendUrl, clientId, clientSecret, audience, Type.KEYCLOAK);
  }

  /**
   * Instantiates a new Identity configuration.
   *
   * @param issuer           the issuer
   * @param issuerBackendUrl the issuer url for back-channel communication
   * @param clientId         the client id
   * @param clientSecret     the client secret
   * @param audience         the audience
   * @param type             the type
   */
  public IdentityConfiguration(
      final String issuer,
      final String issuerBackendUrl,
      final String clientId,
      final String clientSecret,
      final String audience,
      final String type
  ) {
    this(null, issuer, issuerBackendUrl, clientId, clientSecret, audience, Type.valueOf(type));
  }

  /**
   * Instantiates a new Identity configuration.
   *
   * @param baseUrl          the base url of the Camunda Identity instance
   * @param issuer           the issuer
   * @param issuerBackendUrl the issuer url for back-channel communication
   * @param clientId         the client id
   * @param clientSecret     the client secret
   * @param audience         the audience
   * @param type             the type
   */
  public IdentityConfiguration(
      final String baseUrl,
      final String issuer,
      final String issuerBackendUrl,
      final String clientId,
      final String clientSecret,
      final String audience,
      final String type
  ) {
    this(baseUrl, issuer, issuerBackendUrl, clientId, clientSecret, audience, Type.valueOf(type));
  }

  private IdentityConfiguration(
      final String baseUrl,
      final String issuer,
      final String issuerBackendUrl,
      final String clientId,
      final String clientSecret,
      final String audience,
      final Type type
  ) {
    Validate.notNull(type, "type must not be null");
    this.baseUrl = baseUrl;
    this.issuer = issuer;
    this.issuerBackendUrl = issuerBackendUrl;
    this.clientId = clientId;
    this.clientSecret = clientSecret;
    this.audience = audience;
    this.type = type;
  }

  /**
   * Gets base url
   *
   * @return the base url
   */
  public String getBaseUrl() {
    return baseUrl;
  }

  /**
   * Gets issuer.
   *
   * @return the issuer
   */
  public String getIssuer() {
    return issuer;
  }

  /**
   * Gets issuer backend url.
   *
   * @return the issuer backend url
   */
  public String getIssuerBackendUrl() {
    return issuerBackendUrl;
  }

  /**
   * Gets client id.
   *
   * @return the client id
   */
  public String getClientId() {
    return clientId;
  }

  /**
   * Gets audience.
   *
   * @return the audience
   */
  public String getAudience() {
    return audience;
  }

  /**
   * Gets client secret.
   *
   * @return the client secret
   */
  public String getClientSecret() {
    return clientSecret;
  }

  /**
   * Gets type.
   *
   * @return the type
   */
  public Type getType() {
    return type;
  }

  public enum Type {
    KEYCLOAK,
    AUTH0,
    MICROSOFT,
    GENERIC
  }

  /**
   * The type Builder.
   */
  public static class Builder {
    private String issuer;
    private String issuerBackendUrl;
    private String clientId;
    private String clientSecret;
    private String audience;
    private String type = Type.KEYCLOAK.name();
    private String baseUrl = null;

    /**
     * Instantiates a new Builder.
     */
    public Builder() {
    }

    /**
     * Instantiates a new Builder.
     *
     * @param issuer       the issuer
     * @param clientId     the client id
     * @param clientSecret the client secret
     */
    public Builder(
        final String issuer,
        final String issuerBackendUrl,
        final String clientId,
        final String clientSecret
    ) {
      this.issuer = issuer;
      this.issuerBackendUrl = issuerBackendUrl;
      this.clientId = clientId;
      this.clientSecret = clientSecret;
      this.audience = clientId;
    }

    /**
     * Sets the issuer
     *
     * @param issuer the issuer url
     * @return the builder
     */
    public Builder withIssuer(final String issuer) {
      this.issuer = issuer;
      return this;
    }

    /**
     * Sets the issuer backend url
     *
     * @param issuerBackendUrl the issuer backend url
     * @return the builder
     */
    public Builder withIssuerBackendUrl(final String issuerBackendUrl) {
      this.issuerBackendUrl = issuerBackendUrl;
      return this;
    }

    /**
     * Sets the client id
     *
     * @param clientId client id
     * @return the builder
     */
    public Builder withClientId(final String clientId) {
      this.clientId = clientId;
      return this;
    }

    /**
     * Sets the client secret
     *
     * @param clientSecret client secret
     * @return the builder
     */
    public Builder withClientSecret(final String clientSecret) {
      this.clientSecret = clientSecret;
      return this;
    }

    /**
     * With baseUrl builder.
     *
     * @param baseUrl the base url
     * @return the builder
     */
    public Builder withBaseUrl(final String baseUrl) {
      this.baseUrl = baseUrl;
      return this;
    }

    /**
     * With audience builder.
     *
     * @param audience the audience
     * @return the builder
     */
    public Builder withAudience(final String audience) {
      this.audience = audience;
      return this;
    }

    /**
     * With type builder.
     *
     * @param type the type
     * @return the builder
     */
    public Builder withType(final String type) {
      this.type = type;
      return this;
    }

    /**
     * Build an Identity configuration.
     *
     * @return the Identity configuration
     */
    public IdentityConfiguration build() {
      return new IdentityConfiguration(
          baseUrl,
          issuer,
          issuerBackendUrl,
          clientId,
          clientSecret,
          audience,
          type
      );
    }
  }
}
