/*
 * Copyright Camunda Services GmbH and/or licensed to Camunda Services GmbH under
 * one or more contributor license agreements. Licensed under a proprietary license. See the
 * License.txt file for more information. You may not use this file except in compliance with the
 * proprietary license.
 */
package io.camunda.identity.sdk.authentication;

import com.auth0.jwt.interfaces.DecodedJWT;
import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.lang3.Validate;

public class AccessToken implements Serializable {
  private static final long serialVersionUID = -1646956221165712612L;
  private final DecodedJWT token;
  private final List<String> permissions;
  private final Map<String, Set<String>> assignedOrganizations;
  private final UserDetails userDetails;

  AccessToken(
      final DecodedJWT token,
      final List<String> permissions,
      final Map<String, Set<String>> assignedOrganizations,
      final UserDetails userDetails
  ) {
    Validate.notNull(token, "token must not be null");
    Validate.notNull(permissions, "permissions must not be null");
    this.token = token;
    this.permissions = permissions;
    this.assignedOrganizations = assignedOrganizations;
    this.userDetails = userDetails;
  }

  /**
   * Returns the token
   *
   * @return token as DecodedJWT
   */
  public DecodedJWT getToken() {
    return token;
  }

  /**
   * Returns true if the token contains all given permissions
   *
   * @param permissions permissions that must be included in the token
   * @return true if token has correct permission
   */
  public boolean hasPermissions(final Set<String> permissions) {
    return this.permissions.containsAll(permissions);
  }

  /**
   * Returns permissions included in the token
   *
   * @return permissions included in the token
   */
  public List<String> getPermissions() {
    return permissions;
  }

  /**
   * Returns organizations included in the token
   *
   * @return organizations included in the token
   */
  public Map<String, Set<String>> getAssignedOrganizations() {
    return this.assignedOrganizations;
  }

  /**
   * Returns user details
   *
   * @return user details
   */
  public UserDetails getUserDetails() {
    return userDetails;
  }
}
