/*
 * Copyright Camunda Services GmbH and/or licensed to Camunda Services GmbH under
 * one or more contributor license agreements. Licensed under a proprietary license. See the
 * License.txt file for more information. You may not use this file except in compliance with the
 * proprietary license.
 */
package io.camunda.identity.sdk.authorizations;

import io.camunda.identity.sdk.IdentityConfiguration;
import io.camunda.identity.sdk.annotation.AnnotationProcessor;
import io.camunda.identity.sdk.annotation.RequiresBaseUrl;
import io.camunda.identity.sdk.authorizations.dto.Authorization;
import io.camunda.identity.sdk.authorizations.dto.UpdateAuthorizations;
import io.camunda.identity.sdk.impl.AuthorizationsImpl;
import java.util.List;
import java.util.Set;

/**
 * The Authorizations class provides functionality to retrieve the authorizations stored
 * in the Camunda Identity instance.
 */
public interface Authorizations {
  /**
   * Returns the list of authorizations which are assigned to the user linked to the token.
   *
   * @param token the token
   * @return the list of authorizations
   */
  @RequiresBaseUrl
  List<Authorization> forToken(final String token);

  /**
   * Returns the list of authorizations which are assigned to the user linked to the token
   * taking into account the organizationId passed in.
   *
   * @param token the token
   * @param organizationId the orgainzation to retrieve authorizations for
   * @return the list of authorizations
   */
  @RequiresBaseUrl
  List<Authorization> forToken(final String token, final String organizationId);

  /**
   * Assigns an authorization to the user belonging to the token passed in
   *
   * @param token the user token to authenticate the request with
   * @param entityId the ID of the entity to assign the authorization to
   * @param entityType the type of entity see:
   *    {@link io.camunda.identity.sdk.authorizations.dto.EntityType}
   * @param organizationId the organization ID to create the authorization for,
   *                       the user must have access to the organization.
   * @param resourceKey the key of the resource
   * @param resourceType the type of resource see:
   *    {@link io.camunda.identity.sdk.authorizations.dto.ResourceType}
   * @param requestedPermissions the permissions to assign to the authorization see:
   *    {@link io.camunda.identity.sdk.authorizations.dto.ResourcePermissions}
   */
  @RequiresBaseUrl
  void assignAuthorization(
      final String token,
      final String entityId,
      final String entityType,
      final String organizationId,
      final String resourceKey,
      final String resourceType,
      final Set<String> requestedPermissions
  );

  /**
   * Updates authorizations for the user belonging to the token passed in
   *
   * @param token the user token to authenticate the request with
   * @param updateAuthorizations the representation of the authorizations to update, see
   *    {@link io.camunda.identity.sdk.authorizations.dto.UpdateAuthorizations}
   */
  @RequiresBaseUrl
  void update(
      final String token,
      final UpdateAuthorizations updateAuthorizations
  );

  /**
   * Factory function to create a new authorizations instance
   *
   * @param configuration  Identity configuration
   * @return Authorizations instance
   */
  static Authorizations create(
      final IdentityConfiguration configuration
  ) {
    return AnnotationProcessor.apply(configuration, Authorizations.class,
        new AuthorizationsImpl(configuration));
  }
}
