/*
 * Copyright Camunda Services GmbH and/or licensed to Camunda Services GmbH under
 * one or more contributor license agreements. Licensed under a proprietary license. See the
 * License.txt file for more information. You may not use this file except in compliance with the
 * proprietary license.
 */

package io.camunda.identity.sdk.impl.keycloak;

import static io.camunda.identity.sdk.utility.UrlUtility.combinePaths;

import com.auth0.jwk.JwkProvider;
import com.auth0.jwk.JwkProviderBuilder;
import com.auth0.jwt.interfaces.Claim;
import com.auth0.jwt.interfaces.DecodedJWT;
import io.camunda.identity.sdk.IdentityConfiguration;
import io.camunda.identity.sdk.authentication.AuthorizeUriBuilder;
import io.camunda.identity.sdk.authentication.Tokens;
import io.camunda.identity.sdk.authentication.dto.AuthCodeDto;
import io.camunda.identity.sdk.authentication.exception.CodeExchangeException;
import io.camunda.identity.sdk.impl.dto.AccessTokenDto;
import io.camunda.identity.sdk.impl.generic.GenericAuthentication;
import io.camunda.identity.sdk.impl.rest.RestClient;
import io.camunda.identity.sdk.impl.rest.request.ClientTokenRequest;
import io.camunda.identity.sdk.impl.rest.request.ExchangeAuthCodeRequest;
import io.camunda.identity.sdk.impl.rest.request.RenewTokenRequest;
import io.camunda.identity.sdk.impl.rest.request.RevokeTokenRequest;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.Validate;

public class KeycloakAuthentication extends GenericAuthentication {
  public static final String AUTHORIZE_PATH = "/protocol/openid-connect/auth";
  public static final String TOKEN_PATH = "/protocol/openid-connect/token";
  public static final String LOGOUT_PATH = "/protocol/openid-connect/logout";
  public static final String JWKS_PATH = "/protocol/openid-connect/certs";

  public KeycloakAuthentication(
      final IdentityConfiguration configuration,
      final RestClient restClient
  ) {
    super(configuration, restClient);
  }

  @Override
  public AuthorizeUriBuilder authorizeUriBuilder(final String redirectUri) {
    return new KeycloakAuthorizeUriBuilder(
        configuration,
        combinePaths(configuration.getIssuer(), AUTHORIZE_PATH),
        redirectUri
    );
  }

  @Override
  public Tokens exchangeAuthCode(
      final AuthCodeDto authCodeDto,
      final String redirectUri
  ) throws CodeExchangeException {
    Validate.notNull(authCodeDto, "authCodeDto must not be null");
    Validate.notNull(redirectUri, "redirectUri must not be null");

    if (authCodeDto.getError() != null && !authCodeDto.getError().isBlank()) {
      throw new CodeExchangeException(authCodeDto.getError());
    }

    Validate.notEmpty(authCodeDto.getCode(), "code must not be null");

    final ExchangeAuthCodeRequest request = new ExchangeAuthCodeRequest(
        configuration,
        combinePaths(configuration.getIssuerBackendUrl(), TOKEN_PATH),
        redirectUri,
        authCodeDto.getCode()
    );
    final AccessTokenDto accessTokenDto = restClient.request(request);

    return this.fromAccessTokenDto(accessTokenDto);
  }

  @Override
  protected Tokens requestFreshToken(final String audience) {
    final ClientTokenRequest request = new ClientTokenRequest(
        configuration,
        combinePaths(configuration.getIssuerBackendUrl(), TOKEN_PATH),
        audience,
        null
    );
    final AccessTokenDto accessTokenDto = restClient.request(request);

    return fromAccessTokenDto(accessTokenDto);
  }

  @Override
  public Tokens renewToken(final String refreshToken) {
    Validate.notEmpty(refreshToken, "refreshToken can not be empty");

    final RenewTokenRequest request = new RenewTokenRequest(
        configuration,
        combinePaths(configuration.getIssuer(), TOKEN_PATH),
        refreshToken
    );
    final AccessTokenDto accessTokenDto = restClient.request(request);

    return this.fromAccessTokenDto(accessTokenDto);
  }

  @Override
  public void revokeToken(final String refreshToken) {
    Validate.notEmpty(refreshToken, "refreshToken can not be empty");

    final RevokeTokenRequest request = new RevokeTokenRequest(
        configuration,
        combinePaths(configuration.getIssuer(), LOGOUT_PATH),
        refreshToken
    );
    restClient.request(request);
  }

  @Override
  public List<String> getPermissions(final DecodedJWT token, final String audience) {
    if (audience == null) {
      return Collections.emptyList();
    }

    final Claim permissionClaim = token.getClaim("permissions");

    if (permissionClaim.isMissing()) {
      return Collections.emptyList();
    }

    return (List<String>) permissionClaim.asMap()
        .getOrDefault(audience, new ArrayList<>());
  }

  @Override
  protected List<String> getGroupsInOrganization(final DecodedJWT token,
                                                 final String organizationId) {
    final Claim groupsClaim = token.getClaim("groups");

    if (groupsClaim.isMissing()) {
      return Collections.emptyList();
    }

    return groupsClaim.asList(String.class);
  }

  @Override
  public boolean isM2MToken(final String token) {
    final DecodedJWT decodedJwt = decodeJWT(token);

    return !decodedJwt.getClaim("client_id").isMissing();
  }

  @Override
  public String getClientId(final String token) {
    final DecodedJWT decodedJwt = decodeJWT(token);

    return decodedJwt.getClaim("client_id").asString();
  }

  @Override
  protected JwkProvider jwkProvider() {
    if (jwkProvider == null) {
      try {
        final String jwksPath = combinePaths(configuration.getIssuerBackendUrl(), JWKS_PATH);
        jwkProvider = new JwkProviderBuilder(new URL(jwksPath))
            .cached(JWKS_CACHE_SIZE, JWKS_CACHE_LIFETIME_DAYS, TimeUnit.DAYS)
            .build();
      } catch (final MalformedURLException e) {
        throw new IllegalStateException("invalid issuer url", e);
      }
    }
    return jwkProvider;
  }

}
