/*
 * Copyright Camunda Services GmbH and/or licensed to Camunda Services GmbH under
 * one or more contributor license agreements. Licensed under a proprietary license. See the
 * License.txt file for more information. You may not use this file except in compliance with the
 * proprietary license.
 */

package io.camunda.identity.sdk.users;

import io.camunda.identity.sdk.IdentityConfiguration;
import io.camunda.identity.sdk.annotation.AnnotationProcessor;
import io.camunda.identity.sdk.annotation.RequiresBaseUrl;
import io.camunda.identity.sdk.authentication.Authentication;
import io.camunda.identity.sdk.impl.UsersImpl;
import io.camunda.identity.sdk.users.dto.User;
import java.util.List;

/**
 * The Users class provides functionality to retrieve the users stored
 * in the Camunda Identity instance.
 */
public interface Users {
  /**
   * All calls performed with the returned Users instance will be made using given access
   * token
   *
   * @param accessToken access token
   * @return Users instance
   */
  Users withAccessToken(String accessToken);

  /**
   * Returns the list of users which match the search string passed in.
   *
   * @param search the search string
   * @return the list of users
   */
  @RequiresBaseUrl
  List<User> search(final String search);

  /**
   * Returns the list of users which match the search string passed in, found on the page
   * requested.
   *
   * @param search the search string
   * @param page   the page of results
   * @return the list of users
   */
  @RequiresBaseUrl
  List<User> search(final String search, final Integer page);

  /**
   * Returns the list of users which match the search string passed in, found on the page
   * requested with the maximum number of results defined by result size.
   *
   * @param search     the search string
   * @param page       the page of results
   * @param resultSize the amount of results per page
   * @return the list of users
   */
  @RequiresBaseUrl
  List<User> search(
      final String search,
      final Integer page,
      final Integer resultSize
  );

  /**
   * Returns the list of users which match the userIds passed in.
   *
   * @param userIds the userIds
   * @return the list of users
   */
  @RequiresBaseUrl
  List<User> get(final List<String> userIds);


  /**
   * Returns a true/false indicating if the search is available or not.
   *
   * @return the availability of search
   */
  boolean isAvailable();

  /**
   * Factory function to create a new users instance
   *
   * @param configuration  Identity configuration
   * @param authentication Identity Authentication
   * @return Users instance
   */
  static Users create(
      final IdentityConfiguration configuration,
      final Authentication authentication
  ) {
    return AnnotationProcessor.apply(configuration, Users.class,
        new UsersImpl(configuration, authentication));
  }
}
