/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.backup.management;

import io.camunda.zeebe.backup.api.Backup;
import io.camunda.zeebe.backup.api.BackupStatus;
import io.camunda.zeebe.backup.api.BackupStatusCode;
import io.camunda.zeebe.backup.api.BackupStore;
import io.camunda.zeebe.backup.common.BackupIdentifierWildcardImpl;
import io.camunda.zeebe.backup.management.BackupAlreadyExistsException;
import io.camunda.zeebe.backup.management.InProgressBackup;
import io.camunda.zeebe.scheduler.ConcurrencyControl;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class BackupServiceImpl {
    private static final Logger LOG = LoggerFactory.getLogger(BackupServiceImpl.class);
    private final Set<InProgressBackup> backupsInProgress = new HashSet<InProgressBackup>();
    private final BackupStore backupStore;
    private ConcurrencyControl concurrencyControl;

    BackupServiceImpl(BackupStore backupStore) {
        this.backupStore = backupStore;
    }

    void close() {
        this.backupsInProgress.forEach(InProgressBackup::close);
    }

    ActorFuture<Void> takeBackup(InProgressBackup inProgressBackup, ConcurrencyControl concurrencyControl) {
        this.concurrencyControl = concurrencyControl;
        this.backupsInProgress.add(inProgressBackup);
        CompletableFuture<Collection<BackupStatus>> checkCurrentBackup = this.backupStore.list(new BackupIdentifierWildcardImpl(Optional.empty(), Optional.of(inProgressBackup.id().partitionId()), Optional.of(inProgressBackup.checkpointId())));
        ActorFuture backupSaved = concurrencyControl.createFuture();
        checkCurrentBackup.whenCompleteAsync((availableBackups, error) -> {
            if (error != null) {
                backupSaved.completeExceptionally(error);
            } else {
                this.takeBackupIfDoesNotExist((Collection<BackupStatus>)availableBackups, inProgressBackup, concurrencyControl, (ActorFuture<Void>)backupSaved);
            }
        }, arg_0 -> ((ConcurrencyControl)concurrencyControl).run(arg_0));
        backupSaved.onComplete((ignore, error) -> this.closeInProgressBackup(inProgressBackup));
        return backupSaved;
    }

    private void takeBackupIfDoesNotExist(Collection<BackupStatus> availableBackups, InProgressBackup inProgressBackup, ConcurrencyControl concurrencyControl, ActorFuture<Void> backupSaved) {
        BackupStatusCode existingBackupStatus = availableBackups.isEmpty() ? BackupStatusCode.DOES_NOT_EXIST : Collections.max(availableBackups, BackupStatusCode.BY_STATUS).statusCode();
        switch (existingBackupStatus) {
            case COMPLETED: {
                LOG.debug("Backup {} is already completed, will not take a new one", (Object)inProgressBackup.id());
                backupSaved.complete(null);
                break;
            }
            case FAILED: 
            case IN_PROGRESS: {
                LOG.error("Backup {} already exists with status {}, will not take a new one", (Object)inProgressBackup.id(), (Object)existingBackupStatus);
                backupSaved.completeExceptionally((Throwable)new BackupAlreadyExistsException(inProgressBackup.id(), existingBackupStatus));
                break;
            }
            case DOES_NOT_EXIST: {
                inProgressBackup.findValidSnapshot().andThen(inProgressBackup::findSegmentFiles, (Executor)concurrencyControl).andThen(ok -> inProgressBackup.reserveSnapshot(), (Executor)concurrencyControl).andThen(ok -> inProgressBackup.findSnapshotFiles(), (Executor)concurrencyControl).onComplete((result, error) -> {
                    if (error != null) {
                        this.failBackup(inProgressBackup, backupSaved, (Throwable)error);
                    } else {
                        this.saveBackup(inProgressBackup, backupSaved);
                    }
                });
                break;
            }
            default: {
                LOG.warn("Invalid case on BackupStatus {}", (Object)existingBackupStatus);
            }
        }
    }

    private void saveBackup(InProgressBackup inProgressBackup, ActorFuture<Void> backupSaved) {
        this.saveBackup(inProgressBackup).onComplete(this.proceed(error -> this.failBackup(inProgressBackup, backupSaved, (Throwable)error), () -> backupSaved.complete(null)));
    }

    private ActorFuture<Void> saveBackup(InProgressBackup inProgressBackup) {
        ActorFuture future = this.concurrencyControl.createFuture();
        Backup backup = inProgressBackup.createBackup();
        this.backupStore.save(backup).whenComplete((ignore, error) -> {
            if (error == null) {
                future.complete(null);
            } else {
                future.completeExceptionally("Failed to save backup", error);
            }
        });
        return future;
    }

    private void failBackup(InProgressBackup inProgressBackup, ActorFuture<Void> backupSaved, Throwable error) {
        backupSaved.completeExceptionally(error);
        this.backupStore.markFailed(inProgressBackup.id(), error.getMessage());
    }

    private void closeInProgressBackup(InProgressBackup inProgressBackup) {
        this.backupsInProgress.remove(inProgressBackup);
        inProgressBackup.close();
    }

    private BiConsumer<Void, Throwable> proceed(Consumer<Throwable> onError, Runnable nextStep) {
        return (ignore, error) -> {
            if (error != null) {
                onError.accept((Throwable)error);
            } else {
                nextStep.run();
            }
        };
    }

    ActorFuture<Optional<BackupStatus>> getBackupStatus(int partitionId, long checkpointId, ConcurrencyControl executor) {
        ActorFuture future = executor.createFuture();
        executor.run(() -> this.backupStore.list(new BackupIdentifierWildcardImpl(Optional.empty(), Optional.of(partitionId), Optional.of(checkpointId))).whenComplete((backupStatuses, throwable) -> {
            if (throwable != null) {
                future.completeExceptionally(throwable);
            } else {
                future.complete(backupStatuses.stream().max(BackupStatusCode.BY_STATUS));
            }
        }));
        return future;
    }

    void failInProgressBackups(int partitionId, long lastCheckpointId, ConcurrencyControl executor) {
        if (lastCheckpointId != -1L) {
            executor.run(() -> ((CompletableFuture)this.backupStore.list(new BackupIdentifierWildcardImpl(Optional.empty(), Optional.of(partitionId), Optional.of(lastCheckpointId))).thenAccept(backups -> backups.forEach(this::failInProgressBackup))).exceptionally(failure -> {
                LOG.warn("Failed to list backups that should be marked as failed", failure);
                return null;
            }));
        }
    }

    private void failInProgressBackup(BackupStatus backupStatus) {
        if (backupStatus.statusCode() != BackupStatusCode.IN_PROGRESS) {
            return;
        }
        LOG.info("The backup {} initiated by previous leader is still in progress. Marking it as failed.", (Object)backupStatus.id());
        ((CompletableFuture)this.backupStore.markFailed(backupStatus.id(), "Backup is cancelled due to leader change.").thenAccept(ignore -> LOG.trace("Marked backup {} as failed.", (Object)backupStatus.id()))).exceptionally(failed -> {
            LOG.warn("Failed to mark backup {} as failed", (Object)backupStatus.id(), failed);
            return null;
        });
    }

    ActorFuture<Void> deleteBackup(int partitionId, long checkpointId, ConcurrencyControl executor) {
        ActorFuture deleteCompleted = executor.createFuture();
        executor.run(() -> ((CompletableFuture)((CompletableFuture)this.backupStore.list(new BackupIdentifierWildcardImpl(Optional.empty(), Optional.of(partitionId), Optional.of(checkpointId))).thenCompose(backups -> CompletableFuture.allOf((CompletableFuture[])backups.stream().map(this::deleteBackupIfExists).toArray(CompletableFuture[]::new)))).thenAccept(ignore -> deleteCompleted.complete(null))).exceptionally(error -> {
            deleteCompleted.completeExceptionally(error);
            return null;
        }));
        return deleteCompleted;
    }

    private CompletableFuture<Void> deleteBackupIfExists(BackupStatus backupStatus) {
        LOG.debug("Deleting backup {}", (Object)backupStatus.id());
        if (backupStatus.statusCode() == BackupStatusCode.IN_PROGRESS) {
            return this.backupStore.markFailed(backupStatus.id(), "The backup is going to be deleted.").thenCompose(ignore -> this.backupStore.delete(backupStatus.id()));
        }
        return this.backupStore.delete(backupStatus.id());
    }

    ActorFuture<Collection<BackupStatus>> listBackups(int partitionId, ConcurrencyControl executor) {
        ActorFuture availableBackupsFuture = executor.createFuture();
        executor.run(() -> ((CompletableFuture)this.backupStore.list(new BackupIdentifierWildcardImpl(Optional.empty(), Optional.of(partitionId), Optional.empty())).thenAccept(arg_0 -> ((ActorFuture)availableBackupsFuture).complete(arg_0))).exceptionally(error -> {
            availableBackupsFuture.completeExceptionally(error);
            return null;
        }));
        return availableBackupsFuture;
    }
}

