/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.backup.management;

import io.camunda.zeebe.backup.api.Backup;
import io.camunda.zeebe.backup.api.BackupIdentifier;
import io.camunda.zeebe.backup.api.NamedFileSet;
import io.camunda.zeebe.backup.common.BackupDescriptorImpl;
import io.camunda.zeebe.backup.common.BackupImpl;
import io.camunda.zeebe.backup.common.NamedFileSetImpl;
import io.camunda.zeebe.backup.management.InProgressBackup;
import io.camunda.zeebe.backup.management.JournalInfoProvider;
import io.camunda.zeebe.scheduler.ConcurrencyControl;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import io.camunda.zeebe.snapshots.PersistedSnapshot;
import io.camunda.zeebe.snapshots.PersistedSnapshotStore;
import io.camunda.zeebe.snapshots.SnapshotException;
import io.camunda.zeebe.snapshots.SnapshotReservation;
import io.camunda.zeebe.util.Either;
import io.camunda.zeebe.util.VersionUtil;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class InProgressBackupImpl
implements InProgressBackup {
    private static final Logger LOG = LoggerFactory.getLogger(InProgressBackupImpl.class);
    private static final String ERROR_MSG_NO_VALID_SNAPSHOT = "Cannot find a snapshot that can be included in the backup %d. All available snapshots (%s) have processedPosition or lastFollowupEventPosition > checkpointPosition %d";
    private final PersistedSnapshotStore snapshotStore;
    private final BackupIdentifier backupId;
    private final long checkpointPosition;
    private final int numberOfPartitions;
    private final ConcurrencyControl concurrencyControl;
    private final Path segmentsDirectory;
    private final JournalInfoProvider journalInfoProvider;
    private boolean hasSnapshot = true;
    private Set<PersistedSnapshot> availableValidSnapshots;
    private SnapshotReservation snapshotReservation;
    private PersistedSnapshot reservedSnapshot;
    private NamedFileSet snapshotFileSet;
    private NamedFileSet segmentsFileSet;

    InProgressBackupImpl(PersistedSnapshotStore snapshotStore, BackupIdentifier backupId, long checkpointPosition, int numberOfPartitions, ConcurrencyControl concurrencyControl, Path segmentsDirectory, JournalInfoProvider journalInfoProvider) {
        this.snapshotStore = snapshotStore;
        this.backupId = backupId;
        this.checkpointPosition = checkpointPosition;
        this.numberOfPartitions = numberOfPartitions;
        this.concurrencyControl = concurrencyControl;
        this.segmentsDirectory = segmentsDirectory;
        this.journalInfoProvider = journalInfoProvider;
    }

    @Override
    public long checkpointId() {
        return this.backupId.checkpointId();
    }

    @Override
    public long checkpointPosition() {
        return this.checkpointPosition;
    }

    @Override
    public BackupIdentifier id() {
        return this.backupId;
    }

    @Override
    public ActorFuture<Void> findValidSnapshot() {
        ActorFuture result = this.concurrencyControl.createFuture();
        this.snapshotStore.getAvailableSnapshots().onComplete((snapshots, error) -> {
            if (error != null) {
                result.completeExceptionally(error);
            } else if (snapshots.isEmpty()) {
                this.hasSnapshot = false;
                this.availableValidSnapshots = Collections.emptySet();
                result.complete(null);
            } else {
                Either<String, Set<PersistedSnapshot>> eitherSnapshots = this.findValidSnapshot((Set<PersistedSnapshot>)snapshots);
                if (eitherSnapshots.isLeft()) {
                    result.completeExceptionally((Throwable)new SnapshotException.SnapshotNotFoundException((String)eitherSnapshots.getLeft()));
                } else {
                    this.availableValidSnapshots = (Set)eitherSnapshots.get();
                    result.complete(null);
                }
            }
        });
        return result;
    }

    @Override
    public ActorFuture<Void> reserveSnapshot() {
        ActorFuture future = this.concurrencyControl.createFuture();
        if (this.hasSnapshot) {
            Iterator<PersistedSnapshot> snapshotIterator = this.availableValidSnapshots.stream().sorted(Comparator.comparingLong(PersistedSnapshot::getCompactionBound).reversed()).iterator();
            this.tryReserveAnySnapshot(snapshotIterator, (ActorFuture<Void>)future);
        } else {
            future.complete(null);
        }
        return future;
    }

    @Override
    public ActorFuture<Void> findSnapshotFiles() {
        if (!this.hasSnapshot) {
            this.snapshotFileSet = new NamedFileSetImpl(Map.of());
            return this.concurrencyControl.createCompletedFuture();
        }
        ActorFuture filesCollected = this.concurrencyControl.createFuture();
        Path snapshotRoot = this.reservedSnapshot.getPath();
        try (Stream<Path> stream = Files.list(snapshotRoot);){
            Set<Path> snapshotFiles = stream.collect(Collectors.toSet());
            Path checksumFile = this.reservedSnapshot.getChecksumPath();
            HashMap<String, Path> fileSet = new HashMap<String, Path>();
            snapshotFiles.forEach(path -> {
                Path name = snapshotRoot.relativize((Path)path);
                fileSet.put(name.toString(), (Path)path);
            });
            fileSet.put(checksumFile.getFileName().toString(), checksumFile);
            this.snapshotFileSet = new NamedFileSetImpl(fileSet);
            filesCollected.complete(null);
        }
        catch (IOException e) {
            filesCollected.completeExceptionally((Throwable)e);
        }
        return filesCollected;
    }

    @Override
    public ActorFuture<Void> findSegmentFiles() {
        ActorFuture filesCollected = this.concurrencyControl.createFuture();
        try {
            long maxIndex = 0L;
            if (this.availableValidSnapshots != null) {
                maxIndex = this.availableValidSnapshots.stream().mapToLong(PersistedSnapshot::getIndex).max().orElse(0L);
            }
            CompletableFuture<Collection<Path>> segments = this.journalInfoProvider.getTailSegments(maxIndex);
            segments.whenComplete((journalMetadata, throwable) -> {
                if (throwable != null) {
                    filesCollected.completeExceptionally(throwable);
                } else if (journalMetadata.isEmpty()) {
                    filesCollected.completeExceptionally((Throwable)new IllegalStateException("Segments must not be empty"));
                } else {
                    Map<String, Path> map = journalMetadata.stream().collect(Collectors.toMap(path -> this.segmentsDirectory.relativize((Path)path).toString(), Function.identity()));
                    this.segmentsFileSet = new NamedFileSetImpl(map);
                    filesCollected.complete(null);
                }
            });
        }
        catch (Exception e) {
            filesCollected.completeExceptionally((Throwable)e);
        }
        return filesCollected;
    }

    @Override
    public Backup createBackup() {
        Optional<String> snapshotId = this.hasSnapshot ? Optional.of(this.reservedSnapshot.getId()) : Optional.empty();
        BackupDescriptorImpl backupDescriptor = new BackupDescriptorImpl(snapshotId, this.checkpointPosition, this.numberOfPartitions, VersionUtil.getVersion());
        return new BackupImpl(this.backupId, backupDescriptor, this.snapshotFileSet, this.segmentsFileSet);
    }

    @Override
    public void close() {
        if (this.snapshotReservation != null) {
            this.snapshotReservation.release();
            LOG.debug("Released reservation for snapshot {}", (Object)this.reservedSnapshot.getId());
        }
    }

    private Either<String, Set<PersistedSnapshot>> findValidSnapshot(Set<PersistedSnapshot> snapshots) {
        Set validSnapshots = snapshots.stream().filter(s -> s.getMetadata().processedPosition() < this.checkpointPosition).filter(s -> s.getMetadata().lastFollowupEventPosition() < this.checkpointPosition).collect(Collectors.toSet());
        if (validSnapshots.isEmpty()) {
            return Either.left((Object)String.format(ERROR_MSG_NO_VALID_SNAPSHOT, this.checkpointId(), snapshots, this.checkpointPosition));
        }
        return Either.right(validSnapshots);
    }

    private void tryReserveAnySnapshot(Iterator<PersistedSnapshot> snapshotIterator, ActorFuture<Void> future) {
        PersistedSnapshot snapshot = snapshotIterator.next();
        LOG.debug("Attempting to reserve snapshot {}", (Object)snapshot.getId());
        ActorFuture reservationFuture = snapshot.reserve();
        reservationFuture.onComplete((reservation, error) -> {
            if (error != null) {
                LOG.trace("Attempting to reserve snapshot {}, but failed", (Object)snapshot.getId(), error);
                if (snapshotIterator.hasNext()) {
                    this.tryReserveAnySnapshot(snapshotIterator, future);
                } else {
                    future.completeExceptionally(String.format("Attempted to reserve snapshots %s, but no snapshot could be reserved", this.availableValidSnapshots), error);
                }
            } else {
                this.snapshotReservation = reservation;
                this.reservedSnapshot = snapshot;
                LOG.debug("Reserved snapshot {}", (Object)snapshot.getId());
                future.complete(null);
            }
        });
    }
}

