/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.broker.client.impl;

import io.camunda.zeebe.broker.client.api.BrokerClusterState;
import io.camunda.zeebe.broker.client.api.BrokerErrorException;
import io.camunda.zeebe.broker.client.api.BrokerRejectionException;
import io.camunda.zeebe.broker.client.api.BrokerResponseException;
import io.camunda.zeebe.broker.client.api.BrokerTopologyManager;
import io.camunda.zeebe.broker.client.api.IllegalBrokerResponseException;
import io.camunda.zeebe.broker.client.api.NoTopologyAvailableException;
import io.camunda.zeebe.broker.client.api.PartitionInactiveException;
import io.camunda.zeebe.broker.client.api.PartitionNotFoundException;
import io.camunda.zeebe.broker.client.api.RequestDispatchStrategy;
import io.camunda.zeebe.broker.client.api.dto.BrokerRequest;
import io.camunda.zeebe.broker.client.api.dto.BrokerResponse;
import io.camunda.zeebe.broker.client.impl.BrokerClientMetrics;
import io.camunda.zeebe.broker.client.impl.ErrorResponseHandler;
import io.camunda.zeebe.protocol.record.ErrorCode;
import io.camunda.zeebe.protocol.record.MessageHeaderDecoder;
import io.camunda.zeebe.scheduler.Actor;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import io.camunda.zeebe.transport.ClientRequest;
import io.camunda.zeebe.transport.ClientTransport;
import java.time.Duration;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;
import java.util.function.ToIntFunction;
import org.agrona.DirectBuffer;

final class BrokerRequestManager
extends Actor {
    private static final TransportRequestSender SENDER_WITH_RETRY = (c, s, r, t) -> c.sendRequestWithRetry(s, BrokerRequestManager::responseValidation, r, t);
    private static final TransportRequestSender SENDER_WITHOUT_RETRY = ClientTransport::sendRequest;
    private final ClientTransport clientTransport;
    private final RequestDispatchStrategy dispatchStrategy;
    private final BrokerTopologyManager topologyManager;
    private final Duration requestTimeout;

    BrokerRequestManager(ClientTransport clientTransport, BrokerTopologyManager topologyManager, RequestDispatchStrategy dispatchStrategy, Duration requestTimeout) {
        this.clientTransport = clientTransport;
        this.dispatchStrategy = dispatchStrategy;
        this.topologyManager = topologyManager;
        this.requestTimeout = requestTimeout;
    }

    private static boolean responseValidation(DirectBuffer responseContent) {
        ErrorResponseHandler errorHandler = new ErrorResponseHandler();
        MessageHeaderDecoder headerDecoder = new MessageHeaderDecoder();
        headerDecoder.wrap(responseContent, 0);
        if (errorHandler.handlesResponse(headerDecoder)) {
            errorHandler.wrap(responseContent, headerDecoder.encodedLength(), headerDecoder.blockLength(), headerDecoder.version());
            ErrorCode errorCode = errorHandler.getErrorCode();
            return errorCode != ErrorCode.PARTITION_LEADER_MISMATCH;
        }
        return true;
    }

    <T> CompletableFuture<BrokerResponse<T>> sendRequestWithRetry(BrokerRequest<T> request) {
        return this.sendRequestWithRetry(request, this.requestTimeout);
    }

    <T> CompletableFuture<BrokerResponse<T>> sendRequest(BrokerRequest<T> request) {
        return this.sendRequest(request, this.requestTimeout);
    }

    <T> CompletableFuture<BrokerResponse<T>> sendRequest(BrokerRequest<T> request, Duration timeout) {
        return this.sendRequestInternal(request, SENDER_WITHOUT_RETRY, timeout);
    }

    <T> CompletableFuture<BrokerResponse<T>> sendRequestWithRetry(BrokerRequest<T> request, Duration requestTimeout) {
        return this.sendRequestInternal(request, SENDER_WITH_RETRY, requestTimeout);
    }

    private <T> CompletableFuture<BrokerResponse<T>> sendRequestInternal(BrokerRequest<T> request, TransportRequestSender sender, Duration requestTimeout) {
        CompletableFuture responseFuture = new CompletableFuture();
        request.serializeValue();
        this.actor.run(() -> this.sendRequestInternal(request, responseFuture, sender, requestTimeout));
        return responseFuture;
    }

    private <T> void sendRequestInternal(BrokerRequest<T> request, CompletableFuture<BrokerResponse<T>> returnFuture, TransportRequestSender sender, Duration requestTimeout) {
        BrokerAddressProvider nodeIdProvider;
        try {
            nodeIdProvider = this.determineBrokerNodeIdProvider(request);
        }
        catch (PartitionNotFoundException e) {
            returnFuture.completeExceptionally(e);
            BrokerClientMetrics.registerFailedRequest(request.getPartitionId(), request.getType(), "PARTITION_NOT_FOUND");
            return;
        }
        catch (NoTopologyAvailableException e) {
            returnFuture.completeExceptionally(e);
            BrokerClientMetrics.registerFailedRequest(request.getPartitionId(), request.getType(), "NO_TOPOLOGY");
            return;
        }
        catch (PartitionInactiveException e) {
            returnFuture.completeExceptionally(e);
            BrokerClientMetrics.registerFailedRequest(request.getPartitionId(), request.getType(), "PARTITION_INACTIVE");
            return;
        }
        ActorFuture<DirectBuffer> responseFuture = sender.send(this.clientTransport, nodeIdProvider, request, requestTimeout);
        long startTime = System.currentTimeMillis();
        this.actor.runOnCompletion(responseFuture, (clientResponse, error) -> {
            RequestResult result = null;
            try {
                if (error == null) {
                    BrokerResponse response = request.getResponse((DirectBuffer)clientResponse);
                    result = this.handleResponse(response, returnFuture);
                    if (result.wasProcessed()) {
                        long elapsedTime = System.currentTimeMillis() - startTime;
                        BrokerClientMetrics.registerSuccessfulRequest(request.getPartitionId(), request.getType(), elapsedTime);
                        return;
                    }
                } else {
                    returnFuture.completeExceptionally((Throwable)error);
                }
            }
            catch (RuntimeException e) {
                returnFuture.completeExceptionally(new BrokerResponseException(e));
            }
            this.registerFailure(request, result, (Throwable)error);
        });
    }

    private <T> void registerFailure(BrokerRequest<T> request, RequestResult result, Throwable error) {
        if (result != null && result.getErrorCode() == ErrorCode.RESOURCE_EXHAUSTED) {
            return;
        }
        String code = result != null && result.getErrorCode() != ErrorCode.NULL_VAL ? result.getErrorCode().toString() : (error != null && error.getClass().equals(TimeoutException.class) ? "TIMEOUT" : "UNKNOWN");
        BrokerClientMetrics.registerFailedRequest(request.getPartitionId(), request.getType(), code);
    }

    private <T> RequestResult handleResponse(BrokerResponse<T> response, CompletableFuture<BrokerResponse<T>> responseFuture) {
        try {
            if (response.isResponse()) {
                responseFuture.complete(response);
                return RequestResult.processed();
            }
            if (response.isRejection()) {
                responseFuture.completeExceptionally(new BrokerRejectionException(response.getRejection()));
                return RequestResult.processed();
            }
            if (response.isError()) {
                responseFuture.completeExceptionally(new BrokerErrorException(response.getError()));
                return RequestResult.failed(response.getError().getCode());
            }
            responseFuture.completeExceptionally(new IllegalBrokerResponseException("Expected broker response to be either response, rejection, or error, but is neither of them"));
        }
        catch (RuntimeException e) {
            responseFuture.completeExceptionally(new BrokerResponseException(e));
        }
        return RequestResult.failed(ErrorCode.NULL_VAL);
    }

    private BrokerAddressProvider determineBrokerNodeIdProvider(BrokerRequest<?> request) {
        if (request.getBrokerId().isPresent()) {
            return new BrokerAddressProvider(clusterState -> request.getBrokerId().orElseThrow());
        }
        if (request.addressesSpecificPartition()) {
            BrokerClusterState topology = this.topologyManager.getTopology();
            if (topology != null && !topology.getPartitions().contains(request.getPartitionId())) {
                throw new PartitionNotFoundException(request.getPartitionId());
            }
            this.throwIfPartitionInactive(request.getPartitionId());
            return new BrokerAddressProvider(request.getPartitionId());
        }
        if (request.requiresPartitionId()) {
            RequestDispatchStrategy strategy = request.requestDispatchStrategy().orElse(this.dispatchStrategy);
            int partitionId = strategy.determinePartition(this.topologyManager);
            if (partitionId == -3) {
                partitionId = 1;
            }
            request.setPartitionId(partitionId);
            this.throwIfPartitionInactive(partitionId);
            return new BrokerAddressProvider(request.getPartitionId());
        }
        return new BrokerAddressProvider();
    }

    private void throwIfPartitionInactive(int partitionId) {
        boolean noPartitionLeader;
        BrokerClusterState topology = this.topologyManager.getTopology();
        if (topology == null) {
            throw new NoTopologyAvailableException();
        }
        Set<Integer> inactiveNodes = topology.getInactiveNodesForPartition(partitionId);
        boolean someNodesInactive = inactiveNodes != null && !inactiveNodes.isEmpty();
        boolean bl = noPartitionLeader = topology.getLeaderForPartition(partitionId) == -2;
        if (someNodesInactive && noPartitionLeader) {
            throw new PartitionInactiveException(partitionId);
        }
    }

    private static interface TransportRequestSender {
        public ActorFuture<DirectBuffer> send(ClientTransport var1, Supplier<String> var2, ClientRequest var3, Duration var4);
    }

    private class BrokerAddressProvider
    implements Supplier<String> {
        private final ToIntFunction<BrokerClusterState> nodeIdSelector;

        BrokerAddressProvider() {
            this(BrokerClusterState::getRandomBroker);
        }

        BrokerAddressProvider(int partitionId) {
            this(state -> state.getLeaderForPartition(partitionId));
        }

        BrokerAddressProvider(ToIntFunction<BrokerClusterState> nodeIdSelector) {
            this.nodeIdSelector = nodeIdSelector;
        }

        @Override
        public String get() {
            BrokerClusterState topology = BrokerRequestManager.this.topologyManager.getTopology();
            if (topology != null) {
                return topology.getBrokerAddress(this.nodeIdSelector.applyAsInt(topology));
            }
            return null;
        }
    }

    private static class RequestResult {
        private final boolean processed;
        private final ErrorCode errorCode;

        RequestResult(boolean processed, ErrorCode errorCode) {
            this.processed = processed;
            this.errorCode = errorCode;
        }

        boolean wasProcessed() {
            return this.processed;
        }

        public ErrorCode getErrorCode() {
            return this.errorCode;
        }

        static RequestResult processed() {
            return new RequestResult(true, null);
        }

        static RequestResult failed(ErrorCode code) {
            return new RequestResult(false, code);
        }
    }
}

