/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.broker.system.partitions.impl;

import io.atomix.raft.storage.log.IndexedRaftLogEntry;
import io.camunda.zeebe.broker.system.partitions.AtomixRecordEntrySupplier;
import io.camunda.zeebe.broker.system.partitions.StateController;
import io.camunda.zeebe.db.ZeebeDb;
import io.camunda.zeebe.db.ZeebeDbFactory;
import io.camunda.zeebe.logstreams.impl.Loggers;
import io.camunda.zeebe.snapshots.ConstructableSnapshotStore;
import io.camunda.zeebe.snapshots.PersistedSnapshot;
import io.camunda.zeebe.snapshots.TransientSnapshot;
import io.camunda.zeebe.util.FileUtil;
import io.camunda.zeebe.util.sched.ConcurrencyControl;
import io.camunda.zeebe.util.sched.future.ActorFuture;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Optional;
import java.util.function.ToLongFunction;
import org.slf4j.Logger;

public class StateControllerImpl
implements StateController {
    private static final Logger LOG = Loggers.SNAPSHOT_LOGGER;
    private final Path runtimeDirectory;
    private final ZeebeDbFactory zeebeDbFactory;
    private final ToLongFunction<ZeebeDb> exporterPositionSupplier;
    private final AtomixRecordEntrySupplier entrySupplier;
    private ZeebeDb db;
    private final ConstructableSnapshotStore constructableSnapshotStore;
    private final ConcurrencyControl concurrencyControl;

    public StateControllerImpl(ZeebeDbFactory zeebeDbFactory, ConstructableSnapshotStore constructableSnapshotStore, Path runtimeDirectory, AtomixRecordEntrySupplier entrySupplier, ToLongFunction<ZeebeDb> exporterPositionSupplier, ConcurrencyControl concurrencyControl) {
        this.constructableSnapshotStore = constructableSnapshotStore;
        this.runtimeDirectory = runtimeDirectory;
        this.zeebeDbFactory = zeebeDbFactory;
        this.exporterPositionSupplier = exporterPositionSupplier;
        this.entrySupplier = entrySupplier;
        this.concurrencyControl = concurrencyControl;
    }

    @Override
    public ActorFuture<Optional<TransientSnapshot>> takeTransientSnapshot(long lowerBoundSnapshotPosition) {
        ActorFuture future = this.concurrencyControl.createFuture();
        this.concurrencyControl.run(() -> this.takeTransientSnapshotInternal(lowerBoundSnapshotPosition, (ActorFuture<Optional<TransientSnapshot>>)future));
        return future;
    }

    @Override
    public ActorFuture<ZeebeDb> recover() {
        ActorFuture future = this.concurrencyControl.createFuture();
        this.concurrencyControl.run(() -> this.recoverInternal((ActorFuture<ZeebeDb>)future));
        return future;
    }

    @Override
    public ActorFuture<Void> closeDb() {
        ActorFuture future = this.concurrencyControl.createFuture();
        this.concurrencyControl.run(() -> this.closeDbInternal((ActorFuture<Void>)future));
        return future;
    }

    private void closeDbInternal(ActorFuture<Void> future) {
        try {
            if (this.db != null) {
                ZeebeDb dbToClose = this.db;
                this.db = null;
                dbToClose.close();
                LOG.debug("Closed database from '{}'.", (Object)this.runtimeDirectory);
            }
            this.tryDeletingRuntimeDirectory();
            future.complete(null);
        }
        catch (Exception e) {
            future.completeExceptionally((Throwable)e);
        }
    }

    private void recoverInternal(ActorFuture<ZeebeDb> future) {
        try {
            FileUtil.deleteFolderIfExists((Path)this.runtimeDirectory);
        }
        catch (IOException e) {
            future.completeExceptionally((Throwable)new RuntimeException("Failed to delete runtime folder. Cannot recover from snapshot.", e));
        }
        Optional optLatestSnapshot = this.constructableSnapshotStore.getLatestSnapshot();
        if (optLatestSnapshot.isPresent()) {
            PersistedSnapshot snapshot = (PersistedSnapshot)optLatestSnapshot.get();
            LOG.debug("Recovering state from available snapshot: {}", (Object)snapshot);
            this.constructableSnapshotStore.copySnapshot(snapshot, this.runtimeDirectory).onComplete((ok, error) -> {
                if (error != null) {
                    future.completeExceptionally((Throwable)new RuntimeException(String.format("Failed to recover from snapshot %s", snapshot.getId()), (Throwable)error));
                } else {
                    this.openDb(future);
                }
            });
        } else {
            this.openDb(future);
        }
    }

    private void takeTransientSnapshotInternal(long lowerBoundSnapshotPosition, ActorFuture<Optional<TransientSnapshot>> future) {
        if (!this.isDbOpened()) {
            LOG.warn("Expected to take snapshot for last processed position {}, but database was closed.", (Object)lowerBoundSnapshotPosition);
            future.complete(Optional.empty());
            return;
        }
        long exportedPosition = this.exporterPositionSupplier.applyAsLong(this.db);
        long snapshotPosition = this.determineSnapshotPosition(lowerBoundSnapshotPosition, exportedPosition);
        Optional<IndexedRaftLogEntry> optionalIndexed = this.entrySupplier.getPreviousIndexedEntry(snapshotPosition);
        if (optionalIndexed.isEmpty()) {
            future.completeExceptionally((Throwable)new IllegalStateException(String.format("Failed to take snapshot. Expected to find an indexed entry for determined snapshot position %d (processedPosition = %d, exportedPosition=%d), but found no matching indexed entry which contains this position.", snapshotPosition, lowerBoundSnapshotPosition, exportedPosition)));
            return;
        }
        IndexedRaftLogEntry snapshotIndexedEntry = optionalIndexed.get();
        Optional transientSnapshot = this.constructableSnapshotStore.newTransientSnapshot(snapshotIndexedEntry.index(), snapshotIndexedEntry.term(), lowerBoundSnapshotPosition, exportedPosition);
        transientSnapshot.ifPresentOrElse(snapshot -> this.takeSnapshot((TransientSnapshot)snapshot, future), () -> future.complete(Optional.empty()));
    }

    private void openDb(ActorFuture<ZeebeDb> future) {
        try {
            if (this.db == null) {
                this.db = this.zeebeDbFactory.createDb(this.runtimeDirectory.toFile());
                LOG.debug("Opened database from '{}'.", (Object)this.runtimeDirectory);
                future.complete((Object)this.db);
            }
        }
        catch (Exception error) {
            future.completeExceptionally((Throwable)new RuntimeException("Failed to open database", error));
        }
    }

    private void tryDeletingRuntimeDirectory() {
        try {
            FileUtil.deleteFolderIfExists((Path)this.runtimeDirectory);
        }
        catch (Exception e) {
            LOG.debug("Failed to delete runtime directory when closing", (Throwable)e);
        }
    }

    @Override
    public void close() throws Exception {
        this.closeDb();
    }

    boolean isDbOpened() {
        return this.db != null;
    }

    private void takeSnapshot(TransientSnapshot snapshot, ActorFuture<Optional<TransientSnapshot>> transientSnapshotFuture) {
        ActorFuture snapshotTaken = snapshot.take(snapshotDir -> {
            if (this.db == null) {
                LOG.error("Expected to take a snapshot, but no database was opened");
                return false;
            }
            LOG.debug("Taking temporary snapshot into {}.", snapshotDir);
            try {
                this.db.createSnapshot(snapshotDir.toFile());
            }
            catch (Exception e) {
                LOG.error("Failed to create snapshot of runtime database", (Throwable)e);
                return false;
            }
            return true;
        });
        snapshotTaken.onComplete((taken, error) -> {
            if (error != null) {
                transientSnapshotFuture.completeExceptionally(error);
            } else if (taken.booleanValue()) {
                transientSnapshotFuture.complete(Optional.of(snapshot));
            } else {
                transientSnapshotFuture.complete(Optional.empty());
            }
        });
    }

    private long determineSnapshotPosition(long lowerBoundSnapshotPosition, long exportedPosition) {
        long snapshotPosition = Math.min(exportedPosition, lowerBoundSnapshotPosition);
        LOG.trace("Based on lowest exporter position '{}' and last processed position '{}', determined '{}' as snapshot position.", new Object[]{exportedPosition, lowerBoundSnapshotPosition, snapshotPosition});
        return snapshotPosition;
    }
}

