/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.broker.system;

import io.atomix.cluster.AtomixCluster;
import io.camunda.zeebe.backup.gcs.GcsBackupConfig;
import io.camunda.zeebe.backup.gcs.GcsBackupStore;
import io.camunda.zeebe.backup.s3.S3BackupConfig;
import io.camunda.zeebe.backup.s3.S3BackupStore;
import io.camunda.zeebe.broker.Loggers;
import io.camunda.zeebe.broker.system.InvalidConfigurationException;
import io.camunda.zeebe.broker.system.configuration.BrokerCfg;
import io.camunda.zeebe.broker.system.configuration.ClusterCfg;
import io.camunda.zeebe.broker.system.configuration.DataCfg;
import io.camunda.zeebe.broker.system.configuration.DiskCfg;
import io.camunda.zeebe.broker.system.configuration.ExperimentalCfg;
import io.camunda.zeebe.broker.system.configuration.PartitioningCfg;
import io.camunda.zeebe.broker.system.configuration.SecurityCfg;
import io.camunda.zeebe.broker.system.configuration.backup.BackupStoreCfg;
import io.camunda.zeebe.broker.system.configuration.backup.GcsBackupStoreConfig;
import io.camunda.zeebe.broker.system.configuration.backup.S3BackupStoreConfig;
import io.camunda.zeebe.broker.system.configuration.partitioning.FixedPartitionCfg;
import io.camunda.zeebe.broker.system.configuration.partitioning.Scheme;
import io.camunda.zeebe.broker.system.partitions.impl.AsyncSnapshotDirector;
import io.camunda.zeebe.scheduler.ActorScheduler;
import java.io.File;
import java.time.Duration;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;
import org.springframework.util.unit.DataSize;

public final class SystemContext {
    public static final Logger LOG = Loggers.SYSTEM_LOGGER;
    private static final String BROKER_ID_LOG_PROPERTY = "broker-id";
    private static final String NODE_ID_ERROR_MSG = "Node id %s needs to be non negative and smaller then cluster size %s.";
    private static final String SNAPSHOT_PERIOD_ERROR_MSG = "Snapshot period %s needs to be larger then or equals to one minute.";
    private static final String MAX_BATCH_SIZE_ERROR_MSG = "Expected to have an append batch size maximum which is non negative and smaller then '%d', but was '%s'.";
    private final BrokerCfg brokerCfg;
    private Map<String, String> diagnosticContext;
    private final ActorScheduler scheduler;
    private final AtomixCluster cluster;

    public SystemContext(BrokerCfg brokerCfg, ActorScheduler scheduler, AtomixCluster cluster) {
        this.brokerCfg = brokerCfg;
        this.scheduler = scheduler;
        this.cluster = cluster;
        this.initSystemContext();
    }

    private void initSystemContext() {
        this.validateConfiguration();
        ClusterCfg cluster = this.brokerCfg.getCluster();
        String brokerId = String.format("Broker-%d", cluster.getNodeId());
        this.diagnosticContext = Collections.singletonMap(BROKER_ID_LOG_PROPERTY, brokerId);
    }

    private void validateConfiguration() {
        ClusterCfg cluster = this.brokerCfg.getCluster();
        this.validateDataConfig(this.brokerCfg.getData());
        this.validateExperimentalConfigs(cluster, this.brokerCfg.getExperimental());
        SecurityCfg security = this.brokerCfg.getNetwork().getSecurity();
        if (security.isEnabled()) {
            this.validateNetworkSecurityConfig(security);
        }
    }

    private void validateExperimentalConfigs(ClusterCfg cluster, ExperimentalCfg experimental) {
        DataSize maxAppendBatchSize = experimental.getMaxAppendBatchSize();
        if (maxAppendBatchSize.isNegative() || maxAppendBatchSize.toBytes() >= Integer.MAX_VALUE) {
            throw new IllegalArgumentException(String.format(MAX_BATCH_SIZE_ERROR_MSG, Integer.MAX_VALUE, maxAppendBatchSize));
        }
        PartitioningCfg partitioningConfig = experimental.getPartitioning();
        if (partitioningConfig.getScheme() == Scheme.FIXED) {
            this.validateFixedPartitioningScheme(cluster, experimental);
        }
    }

    private void validateDataConfig(DataCfg dataCfg) {
        Duration snapshotPeriod = dataCfg.getSnapshotPeriod();
        if (snapshotPeriod.isNegative() || snapshotPeriod.minus(AsyncSnapshotDirector.MINIMUM_SNAPSHOT_PERIOD).isNegative()) {
            throw new IllegalArgumentException(String.format(SNAPSHOT_PERIOD_ERROR_MSG, snapshotPeriod));
        }
        if (dataCfg.getDisk().isEnableMonitoring()) {
            try {
                DiskCfg.FreeSpaceCfg freeSpaceCfg = dataCfg.getDisk().getFreeSpace();
                long processingFreeSpace = freeSpaceCfg.getProcessing().toBytes();
                long replicationFreeSpace = freeSpaceCfg.getReplication().toBytes();
                if (processingFreeSpace <= replicationFreeSpace) {
                    throw new IllegalArgumentException("Minimum free space for processing (%d) must be greater than minimum free space for replication (%d). Configured values are %s".formatted(processingFreeSpace, replicationFreeSpace, freeSpaceCfg));
                }
            }
            catch (Exception e) {
                throw new InvalidConfigurationException("Failed to parse disk monitoring configuration", e);
            }
        }
        this.validateBackupCfg(dataCfg.getBackup());
    }

    private void validateBackupCfg(BackupStoreCfg backup) {
        try {
            switch (backup.getStore()) {
                case NONE: {
                    LOG.warn("No backup store is configured. Backups will not be taken");
                    break;
                }
                case S3: {
                    S3BackupStore.validateConfig((S3BackupConfig)S3BackupStoreConfig.toStoreConfig(backup.getS3()));
                    break;
                }
                case GCS: {
                    GcsBackupStore.validateConfig((GcsBackupConfig)GcsBackupStoreConfig.toStoreConfig(backup.getGcs()));
                    break;
                }
                default: {
                    throw new UnsupportedOperationException("Does not support validating configuration of backup store %s".formatted(new Object[]{backup.getStore()}));
                }
            }
        }
        catch (Exception e) {
            throw new InvalidConfigurationException("Failed configuring backup store %s".formatted(new Object[]{backup.getStore()}), e);
        }
    }

    private void validateFixedPartitioningScheme(ClusterCfg cluster, ExperimentalCfg experimental) {
        PartitioningCfg partitioning = experimental.getPartitioning();
        List<FixedPartitionCfg> partitions = partitioning.getFixed();
        int replicationFactor = cluster.getReplicationFactor();
        int partitionsCount = cluster.getPartitionsCount();
        HashMap<Integer, Set<Integer>> partitionMembers = new HashMap<Integer, Set<Integer>>();
        for (FixedPartitionCfg partition : partitions) {
            Set<Integer> members = this.validateFixedPartitionMembers(cluster, partition, cluster.getRaft().isEnablePriorityElection());
            partitionMembers.put(partition.getPartitionId(), members);
        }
        for (int partitionId = 1; partitionId <= partitionsCount; ++partitionId) {
            Set members = partitionMembers.getOrDefault(partitionId, Collections.emptySet());
            if (members.size() >= replicationFactor) continue;
            throw new IllegalArgumentException(String.format("Expected fixed partition scheme to define configurations for all partitions such that they have %d replicas, but partition %d has %d configured replicas: %s", replicationFactor, partitionId, members.size(), members));
        }
    }

    private Set<Integer> validateFixedPartitionMembers(ClusterCfg cluster, FixedPartitionCfg partitionConfig, boolean isPriorityElectionEnabled) {
        HashSet<Integer> members = new HashSet<Integer>();
        int clusterSize = cluster.getClusterSize();
        int partitionsCount = cluster.getPartitionsCount();
        int partitionId = partitionConfig.getPartitionId();
        if (partitionId < 1 || partitionId > partitionsCount) {
            throw new IllegalArgumentException(String.format("Expected fixed partition scheme to define entries with a valid partitionId between 1 and %d, but %d was given", partitionsCount, partitionId));
        }
        HashSet<Integer> observedPriorities = new HashSet<Integer>();
        for (FixedPartitionCfg.NodeCfg node : partitionConfig.getNodes()) {
            int nodeId = node.getNodeId();
            if (nodeId < 0 || nodeId >= clusterSize) {
                throw new IllegalArgumentException(String.format("Expected fixed partition scheme for partition %d to define nodes with a nodeId between 0 and %d, but it was %d", partitionId, clusterSize - 1, nodeId));
            }
            if (isPriorityElectionEnabled && !observedPriorities.add(node.getPriority())) {
                throw new IllegalArgumentException(String.format("Expected each node for a partition %d to have a different priority, but at least two of them have the same priorities: %s", partitionId, partitionConfig.getNodes()));
            }
            members.add(nodeId);
        }
        return members;
    }

    private void validateNetworkSecurityConfig(SecurityCfg security) {
        File certificateChainPath = security.getCertificateChainPath();
        File privateKeyPath = security.getPrivateKeyPath();
        if (certificateChainPath == null) {
            throw new IllegalArgumentException("Expected to have a valid certificate chain path for network security, but none configured");
        }
        if (privateKeyPath == null) {
            throw new IllegalArgumentException("Expected to have a valid private key path for network security, but none configured");
        }
        if (!certificateChainPath.canRead()) {
            throw new IllegalArgumentException(String.format("Expected the configured network security certificate chain path '%s' to point to a readable file, but it does not", certificateChainPath));
        }
        if (!privateKeyPath.canRead()) {
            throw new IllegalArgumentException(String.format("Expected the configured network security private key path '%s' to point to a readable file, but it does not", privateKeyPath));
        }
    }

    public ActorScheduler getScheduler() {
        return this.scheduler;
    }

    public BrokerCfg getBrokerConfiguration() {
        return this.brokerCfg;
    }

    public AtomixCluster getCluster() {
        return this.cluster;
    }

    public Map<String, String> getDiagnosticContext() {
        return this.diagnosticContext;
    }
}

