/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.broker.system.partitions.impl;

import io.atomix.raft.storage.log.IndexedRaftLogEntry;
import io.camunda.zeebe.broker.system.partitions.AtomixRecordEntrySupplier;
import io.camunda.zeebe.broker.system.partitions.NoEntryAtSnapshotPosition;
import io.camunda.zeebe.broker.system.partitions.StateController;
import io.camunda.zeebe.db.ZeebeDb;
import io.camunda.zeebe.db.ZeebeDbFactory;
import io.camunda.zeebe.logstreams.impl.Loggers;
import io.camunda.zeebe.scheduler.ConcurrencyControl;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import io.camunda.zeebe.snapshots.ConstructableSnapshotStore;
import io.camunda.zeebe.snapshots.PersistedSnapshot;
import io.camunda.zeebe.snapshots.SnapshotException;
import io.camunda.zeebe.snapshots.TransientSnapshot;
import io.camunda.zeebe.util.Either;
import io.camunda.zeebe.util.FileUtil;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Optional;
import java.util.function.ToLongFunction;
import org.slf4j.Logger;

public class StateControllerImpl
implements StateController {
    private static final Logger LOG = Loggers.SNAPSHOT_LOGGER;
    private final Path runtimeDirectory;
    private final ZeebeDbFactory zeebeDbFactory;
    private final ToLongFunction<ZeebeDb> exporterPositionSupplier;
    private final AtomixRecordEntrySupplier entrySupplier;
    private ZeebeDb db;
    private final ConstructableSnapshotStore constructableSnapshotStore;
    private final ConcurrencyControl concurrencyControl;

    public StateControllerImpl(ZeebeDbFactory zeebeDbFactory, ConstructableSnapshotStore constructableSnapshotStore, Path runtimeDirectory, AtomixRecordEntrySupplier entrySupplier, ToLongFunction<ZeebeDb> exporterPositionSupplier, ConcurrencyControl concurrencyControl) {
        this.constructableSnapshotStore = constructableSnapshotStore;
        this.runtimeDirectory = runtimeDirectory;
        this.zeebeDbFactory = zeebeDbFactory;
        this.exporterPositionSupplier = exporterPositionSupplier;
        this.entrySupplier = entrySupplier;
        this.concurrencyControl = concurrencyControl;
    }

    @Override
    public ActorFuture<TransientSnapshot> takeTransientSnapshot(long lowerBoundSnapshotPosition) {
        ActorFuture future = this.concurrencyControl.createFuture();
        this.concurrencyControl.run(() -> this.takeTransientSnapshotInternal(lowerBoundSnapshotPosition, (ActorFuture<TransientSnapshot>)future));
        return future;
    }

    @Override
    public ActorFuture<ZeebeDb> recover() {
        ActorFuture future = this.concurrencyControl.createFuture();
        this.concurrencyControl.run(() -> this.recoverInternal((ActorFuture<ZeebeDb>)future));
        return future;
    }

    @Override
    public ActorFuture<Void> closeDb() {
        ActorFuture future = this.concurrencyControl.createFuture();
        this.concurrencyControl.run(() -> this.closeDbInternal((ActorFuture<Void>)future));
        return future;
    }

    private void closeDbInternal(ActorFuture<Void> future) {
        try {
            if (this.db != null) {
                ZeebeDb dbToClose = this.db;
                this.db = null;
                dbToClose.close();
                LOG.debug("Closed database from '{}'.", (Object)this.runtimeDirectory);
            }
            this.tryDeletingRuntimeDirectory();
            future.complete(null);
        }
        catch (Exception e) {
            future.completeExceptionally((Throwable)e);
        }
    }

    private void recoverInternal(ActorFuture<ZeebeDb> future) {
        try {
            FileUtil.deleteFolderIfExists((Path)this.runtimeDirectory);
        }
        catch (IOException e) {
            future.completeExceptionally((Throwable)new RuntimeException("Failed to delete runtime folder. Cannot recover from snapshot.", e));
        }
        Optional optLatestSnapshot = this.constructableSnapshotStore.getLatestSnapshot();
        if (optLatestSnapshot.isPresent()) {
            PersistedSnapshot snapshot = (PersistedSnapshot)optLatestSnapshot.get();
            LOG.debug("Recovering state from available snapshot: {}", (Object)snapshot);
            this.constructableSnapshotStore.copySnapshot(snapshot, this.runtimeDirectory).onComplete((ok, error) -> {
                if (error != null) {
                    future.completeExceptionally((Throwable)new RuntimeException(String.format("Failed to recover from snapshot %s", snapshot.getId()), (Throwable)error));
                } else {
                    this.openDb(future);
                }
            });
        } else {
            this.openDb(future);
        }
    }

    private void takeTransientSnapshotInternal(long lowerBoundSnapshotPosition, ActorFuture<TransientSnapshot> future) {
        NextSnapshotId nextSnapshotId;
        if (!this.isDbOpened()) {
            String error = String.format("Expected to take snapshot for last processed position %d, but database was closed.", lowerBoundSnapshotPosition);
            future.completeExceptionally((Throwable)new SnapshotException.StateClosedException(error));
            return;
        }
        try {
            nextSnapshotId = this.tryFindNextSnapshotId(lowerBoundSnapshotPosition);
        }
        catch (NoEntryAtSnapshotPosition e) {
            future.completeExceptionally((Throwable)e);
            return;
        }
        Either transientSnapshot = this.constructableSnapshotStore.newTransientSnapshot(nextSnapshotId.index, nextSnapshotId.term, nextSnapshotId.processedPosition, nextSnapshotId.exportedPosition);
        if (transientSnapshot.isLeft()) {
            future.completeExceptionally((Throwable)transientSnapshot.getLeft());
        } else {
            this.takeSnapshot((TransientSnapshot)transientSnapshot.get(), future);
        }
    }

    private NextSnapshotId tryFindNextSnapshotId(long lastProcessedPosition) throws NoEntryAtSnapshotPosition {
        long exportedPosition = this.exporterPositionSupplier.applyAsLong(this.db);
        if (exportedPosition == -1L) {
            Optional latestSnapshot = this.constructableSnapshotStore.getLatestSnapshot();
            if (latestSnapshot.isPresent()) {
                PersistedSnapshot persistedSnapshot = (PersistedSnapshot)latestSnapshot.get();
                return new NextSnapshotId(persistedSnapshot.getIndex(), persistedSnapshot.getTerm(), lastProcessedPosition, 0L);
            }
            return new NextSnapshotId(0L, 0L, lastProcessedPosition, 0L);
        }
        long snapshotPosition = Math.min(exportedPosition, lastProcessedPosition);
        Optional<IndexedRaftLogEntry> logEntry = this.entrySupplier.getPreviousIndexedEntry(snapshotPosition);
        if (logEntry.isPresent()) {
            return new NextSnapshotId(logEntry.get().index(), logEntry.get().term(), lastProcessedPosition, exportedPosition);
        }
        Optional latestSnapshot = this.constructableSnapshotStore.getLatestSnapshot();
        if (latestSnapshot.isPresent()) {
            LOG.warn("No log entry for next snapshot position {}, using index and term from previous snapshot", (Object)snapshotPosition);
            return new NextSnapshotId(((PersistedSnapshot)latestSnapshot.get()).getIndex(), ((PersistedSnapshot)latestSnapshot.get()).getTerm(), lastProcessedPosition, exportedPosition);
        }
        throw new NoEntryAtSnapshotPosition(String.format("Failed to take snapshot. Expected to find an indexed entry for determined snapshot position %d (processedPosition = %d, exportedPosition=%d) or previous snapshot, but found neither.", snapshotPosition, lastProcessedPosition, exportedPosition));
    }

    private void openDb(ActorFuture<ZeebeDb> future) {
        try {
            if (this.db == null) {
                this.db = this.zeebeDbFactory.createDb(this.runtimeDirectory.toFile());
                LOG.debug("Opened database from '{}'.", (Object)this.runtimeDirectory);
                future.complete((Object)this.db);
            }
        }
        catch (Exception error) {
            future.completeExceptionally((Throwable)new RuntimeException("Failed to open database", error));
        }
    }

    private void tryDeletingRuntimeDirectory() {
        try {
            FileUtil.deleteFolderIfExists((Path)this.runtimeDirectory);
        }
        catch (Exception e) {
            LOG.debug("Failed to delete runtime directory when closing", (Throwable)e);
        }
    }

    @Override
    public void close() throws Exception {
        this.closeDb();
    }

    boolean isDbOpened() {
        return this.db != null;
    }

    private void takeSnapshot(TransientSnapshot snapshot, ActorFuture<TransientSnapshot> transientSnapshotFuture) {
        ActorFuture snapshotTaken = snapshot.take(snapshotDir -> {
            if (this.db == null) {
                throw new SnapshotException.StateClosedException("Expected to take a snapshot, but no database was opened");
            }
            LOG.debug("Taking temporary snapshot into {}.", snapshotDir);
            this.db.createSnapshot(snapshotDir.toFile());
        });
        snapshotTaken.onComplete((ok, error) -> {
            if (error != null) {
                transientSnapshotFuture.completeExceptionally(error);
            } else {
                transientSnapshotFuture.complete((Object)snapshot);
            }
        });
    }

    private record NextSnapshotId(long index, long term, long processedPosition, long exportedPosition) {
    }
}

