/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.dynamic.config;

import io.camunda.zeebe.dynamic.config.serializer.ClusterConfigurationSerializer;
import io.camunda.zeebe.dynamic.config.state.ClusterConfiguration;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.zip.CRC32C;

final class PersistedClusterConfiguration {
    public static final int HEADER_LENGTH = 9;
    private static final byte VERSION = 1;
    private final Path topologyFile;
    private final ClusterConfigurationSerializer serializer;
    private ClusterConfiguration clusterConfiguration;

    private PersistedClusterConfiguration(Path topologyFile, ClusterConfigurationSerializer serializer, ClusterConfiguration clusterConfiguration) {
        this.topologyFile = topologyFile;
        this.serializer = serializer;
        this.clusterConfiguration = clusterConfiguration;
    }

    static PersistedClusterConfiguration ofFile(Path topologyFile, ClusterConfigurationSerializer serializer) {
        ClusterConfiguration currentlyPersisted;
        try {
            currentlyPersisted = PersistedClusterConfiguration.readFromFile(topologyFile, serializer);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return new PersistedClusterConfiguration(topologyFile, serializer, currentlyPersisted);
    }

    ClusterConfiguration getConfiguration() {
        return this.clusterConfiguration;
    }

    void update(ClusterConfiguration clusterConfiguration) throws IOException {
        if (this.clusterConfiguration.equals(clusterConfiguration)) {
            return;
        }
        this.writeToFile(clusterConfiguration);
        this.clusterConfiguration = clusterConfiguration;
    }

    public boolean isUninitialized() {
        return this.clusterConfiguration.isUninitialized();
    }

    private static ClusterConfiguration readFromFile(Path topologyFile, ClusterConfigurationSerializer serializer) throws IOException {
        if (!Files.exists(topologyFile, new LinkOption[0])) {
            return ClusterConfiguration.uninitialized();
        }
        byte[] content = Files.readAllBytes(topologyFile);
        if (content.length < 9) {
            throw new MissingHeader(topologyFile, content.length);
        }
        ByteBuffer header = ByteBuffer.wrap(content, 0, 9).order(ByteOrder.LITTLE_ENDIAN);
        byte version = header.get();
        long expectedChecksum = header.getLong();
        if (version != 1) {
            throw new UnexpectedVersion(topologyFile, version);
        }
        long actualChecksum = PersistedClusterConfiguration.checksum(content, 9, content.length - 9);
        if (expectedChecksum != actualChecksum) {
            throw new ChecksumMismatch(topologyFile, expectedChecksum, actualChecksum);
        }
        return serializer.decodeClusterTopology(content, 9, content.length - 9);
    }

    private void writeToFile(ClusterConfiguration clusterConfiguration) throws IOException {
        byte[] body = this.serializer.encode(clusterConfiguration);
        long checksum = PersistedClusterConfiguration.checksum(body, 0, body.length);
        ByteBuffer buffer = ByteBuffer.allocate(9 + body.length).order(ByteOrder.LITTLE_ENDIAN).put((byte)1).putLong(checksum).put(body);
        Files.write(this.topologyFile, buffer.array(), StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.DSYNC);
    }

    private static long checksum(byte[] bytes, int offset, int length) {
        CRC32C checksum = new CRC32C();
        checksum.update(bytes, offset, length);
        return checksum.getValue();
    }

    public static final class MissingHeader
    extends RuntimeException {
        private MissingHeader(Path topologyFile, Object fileSize) {
            super("Topology file %s is too small to contain the expected header: %s bytes".formatted(topologyFile, fileSize));
        }
    }

    public static final class UnexpectedVersion
    extends RuntimeException {
        private UnexpectedVersion(Path topologyFile, byte version) {
            super("Topology file %s had version '%s', but expected version '%s'".formatted(topologyFile, version, (byte)1));
        }
    }

    public static final class ChecksumMismatch
    extends RuntimeException {
        private ChecksumMismatch(Path topologyFile, long expectedChecksum, long actualChecksum) {
            super("Corrupted topology file: %s. Expected checksum: '%d', actual checksum: '%d'".formatted(topologyFile, expectedChecksum, actualChecksum));
        }
    }
}

