/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.topology;

import io.camunda.zeebe.topology.serializer.ClusterTopologySerializer;
import io.camunda.zeebe.topology.state.ClusterTopology;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.zip.CRC32C;

final class PersistedClusterTopology {
    public static final int HEADER_LENGTH = 9;
    private static final byte VERSION = 1;
    private final Path topologyFile;
    private final ClusterTopologySerializer serializer;
    private ClusterTopology clusterTopology;

    private PersistedClusterTopology(Path topologyFile, ClusterTopologySerializer serializer, ClusterTopology clusterTopology) {
        this.topologyFile = topologyFile;
        this.serializer = serializer;
        this.clusterTopology = clusterTopology;
    }

    static PersistedClusterTopology ofFile(Path topologyFile, ClusterTopologySerializer serializer) {
        ClusterTopology currentlyPersisted;
        try {
            currentlyPersisted = PersistedClusterTopology.readFromFile(topologyFile, serializer);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return new PersistedClusterTopology(topologyFile, serializer, currentlyPersisted);
    }

    ClusterTopology getTopology() {
        return this.clusterTopology;
    }

    void update(ClusterTopology clusterTopology) throws IOException {
        if (this.clusterTopology.equals(clusterTopology)) {
            return;
        }
        this.writeToFile(clusterTopology);
        this.clusterTopology = clusterTopology;
    }

    public boolean isUninitialized() {
        return this.clusterTopology.isUninitialized();
    }

    private static ClusterTopology readFromFile(Path topologyFile, ClusterTopologySerializer serializer) throws IOException {
        if (!Files.exists(topologyFile, new LinkOption[0])) {
            return ClusterTopology.uninitialized();
        }
        byte[] content = Files.readAllBytes(topologyFile);
        if (content.length < 9) {
            throw new MissingHeader(topologyFile, content.length);
        }
        ByteBuffer header = ByteBuffer.wrap(content, 0, 9).order(ByteOrder.LITTLE_ENDIAN);
        byte version = header.get();
        long expectedChecksum = header.getLong();
        if (version != 1) {
            throw new UnexpectedVersion(topologyFile, version);
        }
        long actualChecksum = PersistedClusterTopology.checksum(content, 9, content.length - 9);
        if (expectedChecksum != actualChecksum) {
            throw new ChecksumMismatch(topologyFile, expectedChecksum, actualChecksum);
        }
        return serializer.decodeClusterTopology(content, 9, content.length - 9);
    }

    private void writeToFile(ClusterTopology clusterTopology) throws IOException {
        byte[] body = this.serializer.encode(clusterTopology);
        long checksum = PersistedClusterTopology.checksum(body, 0, body.length);
        ByteBuffer buffer = ByteBuffer.allocate(9 + body.length).order(ByteOrder.LITTLE_ENDIAN).put((byte)1).putLong(checksum).put(body);
        Files.write(this.topologyFile, buffer.array(), StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.DSYNC);
    }

    private static long checksum(byte[] bytes, int offset, int length) {
        CRC32C checksum = new CRC32C();
        checksum.update(bytes, offset, length);
        return checksum.getValue();
    }

    public static final class MissingHeader
    extends RuntimeException {
        private MissingHeader(Path topologyFile, Object fileSize) {
            super("Topology file %s is too small to contain the expected header: %s bytes".formatted(topologyFile, fileSize));
        }
    }

    public static final class UnexpectedVersion
    extends RuntimeException {
        private UnexpectedVersion(Path topologyFile, byte version) {
            super("Topology file %s had version '%s', but expected version '%s'".formatted(topologyFile, version, (byte)1));
        }
    }

    public static final class ChecksumMismatch
    extends RuntimeException {
        private ChecksumMismatch(Path topologyFile, long expectedChecksum, long actualChecksum) {
            super("Corrupted topology file: %s. Expected checksum: '%d', actual checksum: '%d'".formatted(topologyFile, expectedChecksum, actualChecksum));
        }
    }
}

