/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.journal.file;

import io.camunda.zeebe.journal.CorruptedJournalException;
import io.camunda.zeebe.journal.JournalException;
import io.camunda.zeebe.journal.JournalRecord;
import io.camunda.zeebe.journal.file.FrameUtil;
import io.camunda.zeebe.journal.file.JournalIndex;
import io.camunda.zeebe.journal.file.JournalMetrics;
import io.camunda.zeebe.journal.file.Segment;
import io.camunda.zeebe.journal.record.JournalRecordReaderUtil;
import io.camunda.zeebe.journal.record.JournalRecordSerializer;
import io.camunda.zeebe.journal.record.PersistedJournalRecord;
import io.camunda.zeebe.journal.record.RecordData;
import io.camunda.zeebe.journal.record.RecordMetadata;
import io.camunda.zeebe.journal.record.SBESerializer;
import io.camunda.zeebe.journal.util.ChecksumGenerator;
import io.camunda.zeebe.util.Either;
import io.camunda.zeebe.util.buffer.BufferWriter;
import io.camunda.zeebe.util.buffer.DirectBufferWriter;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import org.agrona.DirectBuffer;
import org.agrona.MutableDirectBuffer;
import org.agrona.concurrent.UnsafeBuffer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class SegmentWriter {
    private static final Logger LOG = LoggerFactory.getLogger(SegmentWriter.class);
    private final MappedByteBuffer buffer;
    private final Segment segment;
    private final JournalIndex index;
    private final long firstIndex;
    private final long firstAsqn;
    private long lastAsqn;
    private JournalRecord lastEntry;
    private int lastEntryPosition;
    private final JournalRecordReaderUtil recordUtil;
    private final ChecksumGenerator checksumGenerator = new ChecksumGenerator();
    private final JournalRecordSerializer serializer = new SBESerializer();
    private final MutableDirectBuffer writeBuffer = new UnsafeBuffer();
    private final int descriptorLength;
    private final JournalMetrics metrics;

    SegmentWriter(MappedByteBuffer buffer, Segment segment, JournalIndex index, long lastWrittenAsqn, JournalMetrics metrics) {
        this.segment = segment;
        this.descriptorLength = segment.descriptor().length();
        this.recordUtil = new JournalRecordReaderUtil(this.serializer);
        this.index = index;
        this.firstIndex = segment.index();
        this.buffer = buffer;
        this.writeBuffer.wrap((ByteBuffer)buffer);
        this.firstAsqn = lastWrittenAsqn + 1L;
        this.lastAsqn = lastWrittenAsqn;
        this.lastEntryPosition = segment.descriptor().lastPosition();
        this.metrics = metrics;
        if (this.lastEntryPosition > 0) {
            LOG.trace("Found lastEntryPosition {} and lastIndex {} in descriptor.", (Object)this.lastEntryPosition, (Object)segment.descriptor().lastIndex());
            this.jumpToLastEntry(this.lastEntryPosition, segment.descriptor().lastIndex());
        } else {
            LOG.trace("Found not info about last entry in descriptor. Scanning the segment to reset the writer.");
            this.reset(0L, false);
        }
    }

    long getLastIndex() {
        return this.lastEntry != null ? this.lastEntry.index() : this.segment.index() - 1L;
    }

    int getLastEntryPosition() {
        return this.lastEntryPosition;
    }

    long getNextIndex() {
        if (this.lastEntry != null) {
            return this.lastEntry.index() + 1L;
        }
        return this.firstIndex;
    }

    public long getLastAsqn() {
        return this.lastAsqn;
    }

    Either<JournalException.SegmentFull, JournalRecord> append(JournalRecord record) {
        long entryIndex = record.index();
        long asqn = record.asqn();
        DirectBufferWriter recordDataWriter = new DirectBufferWriter().wrap(record.data());
        long expectedChecksum = record.checksum();
        this.verifyAsqnIsIncreasing(asqn);
        int startPosition = this.buffer.position();
        int frameLength = FrameUtil.getLength();
        int metadataLength = this.serializer.getMetadataLength();
        Either<JournalException.SegmentFull, Integer> writeResult = this.writeRecordAtOldVersion(entryIndex, asqn, startPosition + frameLength + metadataLength, (BufferWriter)recordDataWriter);
        return this.tryFinalizeAppend(expectedChecksum, startPosition, frameLength, metadataLength, writeResult);
    }

    Either<JournalException.SegmentFull, JournalRecord> append(long asqn, BufferWriter recordDataWriter) {
        return this.append(this.getNextIndex(), asqn, recordDataWriter, null);
    }

    private Either<JournalException.SegmentFull, JournalRecord> append(Long entryIndex, long asqn, BufferWriter recordDataWriter, Long expectedChecksum) {
        this.verifyAsqnIsIncreasing(asqn);
        int startPosition = this.buffer.position();
        int frameLength = FrameUtil.getLength();
        int metadataLength = this.serializer.getMetadataLength();
        Either<JournalException.SegmentFull, Integer> writeResult = this.writeRecord(entryIndex, asqn, startPosition + frameLength + metadataLength, recordDataWriter);
        return this.tryFinalizeAppend(expectedChecksum, startPosition, frameLength, metadataLength, writeResult);
    }

    Either<JournalException.SegmentFull, JournalRecord> append(long expectedChecksum, byte[] serializedRecord) {
        int startPosition = this.buffer.position();
        int frameLength = FrameUtil.getLength();
        int recordLength = serializedRecord.length;
        int metadataLength = this.serializer.getMetadataLength();
        if (startPosition + frameLength + metadataLength + recordLength > this.buffer.capacity()) {
            return Either.left((Object)new JournalException.SegmentFull("Not enough space to write record"));
        }
        this.writeBuffer.putBytes(startPosition + frameLength + metadataLength, serializedRecord);
        this.finalizeAppend(expectedChecksum, startPosition, frameLength, metadataLength, recordLength);
        return Either.right((Object)this.lastEntry);
    }

    private void verifyAsqnIsIncreasing(long asqn) {
        if (asqn != -1L && asqn <= this.lastAsqn) {
            throw new JournalException.InvalidAsqn(String.format("The records asqn is not big enough. Expected it to be bigger than %d but was %d", this.lastAsqn, asqn));
        }
    }

    private static void verifyNoIndexGap(Long entryIndex, long nextIndex) {
        if (entryIndex != nextIndex) {
            throw new JournalException.InvalidIndex(String.format("The record index is not sequential. Expected the next index to be %d, but the record to append has index %d", nextIndex, entryIndex));
        }
    }

    private Either<JournalException.SegmentFull, JournalRecord> tryFinalizeAppend(Long expectedChecksum, int startPosition, int frameLength, int metadataLength, Either<JournalException.SegmentFull, Integer> writeResult) {
        return writeResult.map(recordLength -> {
            this.finalizeAppend(expectedChecksum, startPosition, frameLength, metadataLength, (int)recordLength);
            return this.lastEntry;
        }).mapLeft(segmentFull -> {
            this.buffer.position(startPosition);
            return segmentFull;
        });
    }

    private void finalizeAppend(Long expectedChecksum, int startPosition, int frameLength, int metadataLength, int recordLength) {
        long checksum = this.checksumGenerator.compute(this.buffer, startPosition + frameLength + metadataLength, recordLength);
        if (expectedChecksum != null && expectedChecksum != checksum) {
            this.buffer.position(startPosition);
            throw new JournalException.InvalidChecksum(String.format("Failed to append record. Checksum %d does not match the expected %d.", checksum, expectedChecksum));
        }
        this.writeMetadata(startPosition, frameLength, recordLength, checksum);
        int nextEntryOffset = startPosition + frameLength + metadataLength + recordLength;
        this.invalidateNextEntry(nextEntryOffset);
        this.updateLastWrittenEntry(startPosition, frameLength, metadataLength, recordLength);
        FrameUtil.writeVersion(this.buffer, startPosition);
        int appendedBytes = frameLength + metadataLength + recordLength;
        this.buffer.position(startPosition + appendedBytes);
        this.metrics.observeAppend(appendedBytes);
    }

    private void updateLastWrittenEntry(int startPosition, int frameLength, int metadataLength, int recordLength) {
        RecordMetadata metadata = this.serializer.readMetadata((DirectBuffer)this.writeBuffer, startPosition + frameLength);
        RecordData data = this.serializer.readData((DirectBuffer)this.writeBuffer, startPosition + frameLength + metadataLength);
        SegmentWriter.verifyNoIndexGap(data.index(), this.getNextIndex());
        this.lastEntry = new PersistedJournalRecord(metadata, data, (DirectBuffer)new UnsafeBuffer((DirectBuffer)this.writeBuffer, startPosition + frameLength + metadataLength, recordLength));
        this.updateLastAsqn(this.lastEntry.asqn());
        this.index.index(this.lastEntry, startPosition);
        this.lastEntryPosition = startPosition;
    }

    private void updateLastAsqn(long asqn) {
        this.lastAsqn = asqn != -1L ? asqn : this.lastAsqn;
    }

    private void writeMetadata(int startPosition, int frameLength, int recordLength, long checksum) {
        RecordMetadata recordMetadata = new RecordMetadata(checksum, recordLength);
        this.serializer.writeMetadata(recordMetadata, this.writeBuffer, startPosition + frameLength);
    }

    private Either<JournalException.SegmentFull, Integer> writeRecord(long index, long asqn, int offset, BufferWriter recordDataWriter) {
        return this.serializer.writeData(index, asqn, recordDataWriter, this.writeBuffer, offset).mapLeft(e -> new JournalException.SegmentFull("Not enough space to write record"));
    }

    private Either<JournalException.SegmentFull, Integer> writeRecordAtOldVersion(long index, long asqn, int offset, BufferWriter recordDataWriter) {
        return this.serializer.writeDataAtVersion(1, index, asqn, recordDataWriter, this.writeBuffer, offset).mapLeft(e -> new JournalException.SegmentFull("Not enough space to write record"));
    }

    private void invalidateNextEntry(int position) {
        if (position >= this.buffer.capacity()) {
            return;
        }
        FrameUtil.markAsIgnored(this.buffer, position);
    }

    private void jumpToLastEntry(int lastPosition, long lastIndex) {
        try {
            this.buffer.position(lastPosition);
            this.buffer.mark();
            if (!FrameUtil.hasValidVersion(this.buffer)) {
                this.reset(0L, false);
            } else {
                long nextIndex = lastIndex;
                while (FrameUtil.hasValidVersion(this.buffer)) {
                    this.advanceToNextEntry(nextIndex);
                    ++nextIndex;
                }
            }
        }
        catch (Exception e) {
            LOG.trace("Failed to read last entry from the given lastEntryPosition {}.Scanning the segment to reset the writer.", (Object)lastPosition, (Object)e);
            this.reset(0L, false);
        }
    }

    private void advanceToNextEntry(long nextIndex) {
        int position = this.buffer.position();
        FrameUtil.readVersion(this.buffer);
        this.lastEntry = this.recordUtil.read(this.buffer, nextIndex);
        this.updateLastAsqn(this.lastEntry.asqn());
        this.lastEntryPosition = position;
        this.index.index(this.lastEntry, position);
        this.buffer.mark();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void reset(long index, boolean detectCorruption) {
        this.buffer.position(this.descriptorLength);
        this.buffer.mark();
        int position = this.buffer.position();
        try {
            for (long nextIndex = this.firstIndex; (index == 0L || nextIndex <= index) && FrameUtil.hasValidVersion(this.buffer); ++nextIndex) {
                this.advanceToNextEntry(nextIndex);
                position = this.buffer.position();
            }
        }
        catch (BufferUnderflowException bufferUnderflowException) {
        }
        catch (CorruptedJournalException e) {
            if (detectCorruption) {
                throw e;
            }
            this.resetPartiallyWrittenEntry(e, position);
        }
        finally {
            this.buffer.reset();
        }
    }

    private void resetPartiallyWrittenEntry(CorruptedJournalException e, int position) {
        LOG.debug("{} Found a corrupted or partially written entry at position {}. Considering it as a partially written entry and resetting the position.", (Object)e.getMessage(), (Object)position);
        FrameUtil.markAsIgnored(this.buffer, position);
        this.buffer.position(position);
        this.buffer.mark();
    }

    public void truncate(long index) {
        if (index >= this.getLastIndex()) {
            return;
        }
        this.lastEntry = null;
        this.index.deleteAfter(index);
        this.lastAsqn = this.firstAsqn - 1L;
        if (index < this.segment.index()) {
            this.buffer.position(this.descriptorLength);
            this.invalidateNextEntry(this.descriptorLength);
        } else {
            if (this.lastEntryPosition > 0) {
                this.invalidateNextEntry(this.lastEntryPosition);
            }
            this.reset(index, true);
            this.invalidateNextEntry(this.buffer.position());
        }
    }
}

