/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.scheduler.startup;

import io.camunda.zeebe.scheduler.ConcurrencyControl;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import io.camunda.zeebe.scheduler.startup.StartupProcessException;
import io.camunda.zeebe.scheduler.startup.StartupProcessStepException;
import io.camunda.zeebe.scheduler.startup.StartupStep;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.List;
import java.util.Objects;
import java.util.Queue;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class StartupProcess<CONTEXT> {
    private final Logger logger;
    private final Queue<StartupStep<CONTEXT>> steps;
    private final Deque<StartupStep<CONTEXT>> startedSteps = new ArrayDeque<StartupStep<CONTEXT>>();
    private boolean startupCalled = false;
    private ActorFuture<CONTEXT> shutdownFuture;
    private ActorFuture<CONTEXT> startupFuture;

    public StartupProcess(List<StartupStep<CONTEXT>> steps) {
        this(LoggerFactory.getLogger(StartupProcess.class), steps);
    }

    public StartupProcess(Logger logger, List<? extends StartupStep<CONTEXT>> steps) {
        this.steps = new ArrayDeque<StartupStep<CONTEXT>>((Collection)Objects.requireNonNull(steps));
        this.logger = Objects.requireNonNull(logger);
    }

    public ActorFuture<CONTEXT> startup(ConcurrencyControl concurrencyControl, CONTEXT context) {
        ActorFuture result = concurrencyControl.createFuture();
        concurrencyControl.run(() -> this.startupSynchronized(concurrencyControl, context, result));
        return result;
    }

    public ActorFuture<CONTEXT> shutdown(ConcurrencyControl concurrencyControl, CONTEXT context) {
        ActorFuture result = concurrencyControl.createFuture();
        concurrencyControl.run(() -> this.shutdownSynchronized(concurrencyControl, context, result));
        return result;
    }

    private void startupSynchronized(ConcurrencyControl concurrencyControl, CONTEXT context, ActorFuture<CONTEXT> startupFuture) {
        this.logger.debug("Startup was called with context: {}", (Object)context);
        if (this.startupCalled) {
            throw new IllegalStateException("startup(...) must only be called once");
        }
        this.startupCalled = true;
        this.startupFuture = startupFuture;
        concurrencyControl.runOnCompletion(startupFuture, (result, error2) -> {
            this.startupFuture = null;
        });
        ArrayDeque<StartupStep<CONTEXT>> stepsToStart = new ArrayDeque<StartupStep<CONTEXT>>(this.steps);
        this.proceedWithStartupSynchronized(concurrencyControl, stepsToStart, context, startupFuture);
    }

    private void proceedWithStartupSynchronized(ConcurrencyControl concurrencyControl, Queue<StartupStep<CONTEXT>> stepsToStart, CONTEXT context, ActorFuture<CONTEXT> startupFuture) {
        if (stepsToStart.isEmpty()) {
            startupFuture.complete(context);
            this.logger.debug("Finished startup process");
        } else if (this.shutdownFuture != null) {
            this.logger.info("Aborting startup process because shutdown was called");
            startupFuture.completeExceptionally(new StartupProcessException("Aborting startup process because shutdown was called"));
        } else {
            StartupStep<CONTEXT> stepToStart = stepsToStart.poll();
            this.startedSteps.push(stepToStart);
            this.logCurrentStepSynchronized("Startup", stepToStart);
            ActorFuture<CONTEXT> stepStartupFuture = stepToStart.startup(context);
            concurrencyControl.runOnCompletion(stepStartupFuture, (contextReturnedByStep, error2) -> {
                if (error2 != null) {
                    this.completeStartupFutureExceptionallySynchronized(startupFuture, stepToStart, (Throwable)error2);
                } else {
                    this.proceedWithStartupSynchronized(concurrencyControl, stepsToStart, contextReturnedByStep, startupFuture);
                }
            });
        }
    }

    private void completeStartupFutureExceptionallySynchronized(ActorFuture<CONTEXT> startupFuture, StartupStep<CONTEXT> stepToStart, Throwable error2) {
        this.logger.warn("Aborting startup process due to exception during step " + stepToStart.getName(), error2);
        startupFuture.completeExceptionally(this.aggregateExceptionsSynchronized("Startup", Collections.singletonList(new StartupProcessStepException(stepToStart.getName(), error2))));
    }

    private void shutdownSynchronized(ConcurrencyControl concurrencyControl, CONTEXT context, ActorFuture<CONTEXT> resultFuture) {
        this.logger.debug("Shutdown was called with context: {}", (Object)context);
        if (this.shutdownFuture == null) {
            this.shutdownFuture = resultFuture;
            if (this.startupFuture != null) {
                concurrencyControl.runOnCompletion(this.startupFuture, (contextReturnedByStartup, error2) -> {
                    Object contextForShutdown = error2 == null ? contextReturnedByStartup : context;
                    this.proceedWithShutdownSynchronized(concurrencyControl, contextForShutdown, this.shutdownFuture, new ArrayList<StartupProcessStepException>());
                });
            } else {
                this.proceedWithShutdownSynchronized(concurrencyControl, context, this.shutdownFuture, new ArrayList<StartupProcessStepException>());
            }
        } else {
            this.logger.info("Shutdown already in progress");
            concurrencyControl.runOnCompletion(this.shutdownFuture, (contextReturnedByShutdown, error2) -> {
                if (error2 != null) {
                    resultFuture.completeExceptionally((Throwable)error2);
                } else {
                    resultFuture.complete(contextReturnedByShutdown);
                }
            });
        }
    }

    private void proceedWithShutdownSynchronized(ConcurrencyControl concurrencyControl, CONTEXT context, ActorFuture<CONTEXT> shutdownFuture, List<StartupProcessStepException> collectedExceptions) {
        if (this.startedSteps.isEmpty()) {
            this.completeShutdownFutureSynchronized(context, shutdownFuture, collectedExceptions);
        } else {
            StartupStep<CONTEXT> stepToShutdown = this.startedSteps.pop();
            this.logCurrentStepSynchronized("Shutdown", stepToShutdown);
            ActorFuture<CONTEXT> shutdownStepFuture = stepToShutdown.shutdown(context);
            concurrencyControl.runOnCompletion(shutdownStepFuture, (contextReturnedByShutdown, error2) -> {
                Object contextToUse;
                if (error2 != null) {
                    collectedExceptions.add(new StartupProcessStepException(stepToShutdown.getName(), (Throwable)error2));
                    contextToUse = context;
                } else {
                    contextToUse = contextReturnedByShutdown;
                }
                this.proceedWithShutdownSynchronized(concurrencyControl, contextToUse, shutdownFuture, collectedExceptions);
            });
        }
    }

    private void completeShutdownFutureSynchronized(CONTEXT context, ActorFuture<CONTEXT> shutdownFuture, List<StartupProcessStepException> collectedExceptions) {
        if (collectedExceptions.isEmpty()) {
            shutdownFuture.complete(context);
            this.logger.debug("Finished shutdown process");
        } else {
            Throwable umbrellaException = this.aggregateExceptionsSynchronized("Shutdown", collectedExceptions);
            shutdownFuture.completeExceptionally(umbrellaException);
            this.logger.warn(umbrellaException.getMessage(), umbrellaException);
        }
    }

    private Throwable aggregateExceptionsSynchronized(String operation, List<StartupProcessStepException> exceptions) {
        List failedSteps = exceptions.stream().map(StartupProcessStepException::getStepName).collect(Collectors.toList());
        String message = String.format("%s failed in the following steps: %s. See suppressed exceptions for details.", operation, failedSteps);
        StartupProcessException exception = new StartupProcessException(message);
        exceptions.forEach(exception::addSuppressed);
        return exception;
    }

    private void logCurrentStepSynchronized(String process, StartupStep<CONTEXT> step) {
        this.logger.info(process + " " + step.getName());
    }
}

