/*
 * Copyright Camunda Services GmbH and/or licensed to Camunda Services GmbH under
 * one or more contributor license agreements. See the NOTICE file distributed
 * with this work for additional information regarding copyright ownership.
 * Licensed under the Zeebe Community License 1.1. You may not use this file
 * except in compliance with the Zeebe Community License 1.1.
 */
package io.camunda.zeebe.protocol.impl.record.value.processinstance;

import com.fasterxml.jackson.annotation.JsonIgnore;
import io.camunda.zeebe.msgpack.property.ArrayProperty;
import io.camunda.zeebe.msgpack.property.LongProperty;
import io.camunda.zeebe.msgpack.value.LongValue;
import io.camunda.zeebe.msgpack.value.ObjectValue;
import io.camunda.zeebe.protocol.impl.record.UnifiedRecordValue;
import io.camunda.zeebe.protocol.record.value.ProcessInstanceModificationRecordValue;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

public final class ProcessInstanceModificationRecord extends UnifiedRecordValue
    implements ProcessInstanceModificationRecordValue {

  private final LongProperty processInstanceKeyProperty = new LongProperty("processInstanceKey");
  private final ArrayProperty<ProcessInstanceModificationTerminateInstruction>
      terminateInstructionsProperty =
          new ArrayProperty<>(
              "terminateInstructions", new ProcessInstanceModificationTerminateInstruction());
  private final ArrayProperty<ProcessInstanceModificationActivateInstruction>
      activateInstructionsProperty =
          new ArrayProperty<>(
              "activateInstructions", new ProcessInstanceModificationActivateInstruction());

  @Deprecated(since = "8.1.3")
  private final ArrayProperty<LongValue> activatedElementInstanceKeys =
      new ArrayProperty<>("activatedElementInstanceKeys", new LongValue());

  public ProcessInstanceModificationRecord() {
    declareProperty(processInstanceKeyProperty)
        .declareProperty(terminateInstructionsProperty)
        .declareProperty(activateInstructionsProperty)
        .declareProperty(activatedElementInstanceKeys);
  }

  /**
   * This method is expensive because it copies each element before returning it. It is recommended
   * to use {@link #hasTerminateInstructions()} before calling this.
   *
   * <p>{@inheritDoc}
   */
  @Override
  public List<ProcessInstanceModificationTerminateInstructionValue> getTerminateInstructions() {
    // we need to make a copy of each element in the ArrayProperty while iterating it because the
    // inner values are updated during the iteration
    return terminateInstructionsProperty.stream()
        .map(
            element -> {
              final var elementCopy = new ProcessInstanceModificationTerminateInstruction();
              elementCopy.copy(element);
              return (ProcessInstanceModificationTerminateInstructionValue) elementCopy;
            })
        .toList();
  }

  /**
   * This method is expensive because it copies each element before returning it. It is recommended
   * to use {@link #hasActivateInstructions()} before calling this.
   *
   * <p>{@inheritDoc}
   */
  @Override
  public List<ProcessInstanceModificationActivateInstructionValue> getActivateInstructions() {
    // we need to make a copy of each element in the ArrayProperty while iterating it because the
    // inner values are updated during the iteration
    return activateInstructionsProperty.stream()
        .map(
            element -> {
              final var elementCopy = new ProcessInstanceModificationActivateInstruction();
              elementCopy.copy(element);
              return (ProcessInstanceModificationActivateInstructionValue) elementCopy;
            })
        .toList();
  }

  @Override
  public Set<Long> getAncestorScopeKeys() {
    final Set<Long> activatedElementInstanceKeys =
        this.activatedElementInstanceKeys.stream()
            .map(LongValue::getValue)
            .collect(Collectors.toSet());
    // For backwards compatibility's sake we have to add the ancestor scope keys of all activate
    // instructions, as from version 8.1.3 on the activatedElementInstanceKeys property is no longer
    // filled.
    activatedElementInstanceKeys.addAll(
        getActivateInstructions().stream()
            .map(ProcessInstanceModificationActivateInstructionValue::getAncestorScopeKeys)
            .flatMap(Set::stream)
            .collect(Collectors.toSet()));
    return activatedElementInstanceKeys;
  }

  /** Returns true if this record has terminate instructions, otherwise false. */
  @JsonIgnore
  public boolean hasTerminateInstructions() {
    return !terminateInstructionsProperty.isEmpty();
  }

  public ProcessInstanceModificationRecord addTerminateInstruction(
      final ProcessInstanceModificationTerminateInstructionValue terminateInstruction) {
    terminateInstructionsProperty.add().copy(terminateInstruction);
    return this;
  }

  /** Returns true if this record has activate instructions, otherwise false. */
  @JsonIgnore
  public boolean hasActivateInstructions() {
    return !activateInstructionsProperty.isEmpty();
  }

  public ProcessInstanceModificationRecord addActivateInstruction(
      final ProcessInstanceModificationActivateInstruction activateInstruction) {
    activateInstructionsProperty.add().copy(activateInstruction);
    return this;
  }

  @Override
  public long getProcessInstanceKey() {
    return processInstanceKeyProperty.getValue();
  }

  public ProcessInstanceModificationRecord setProcessInstanceKey(final long processInstanceKey) {
    processInstanceKeyProperty.setValue(processInstanceKey);
    return this;
  }

  /** hashCode relies on implementation provided by {@link ObjectValue#hashCode()} */
  @Override
  public int hashCode() {
    return super.hashCode();
  }

  /** equals relies on implementation provided by {@link ObjectValue#equals(Object)} */
  @Override
  public boolean equals(final Object o) {
    return super.equals(o);
  }
}
