/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.util.health;

import io.camunda.zeebe.util.health.FailureListener;
import io.camunda.zeebe.util.health.HealthMonitor;
import io.camunda.zeebe.util.health.HealthMonitorable;
import io.camunda.zeebe.util.health.HealthReport;
import io.camunda.zeebe.util.health.HealthStatus;
import io.camunda.zeebe.util.sched.ActorControl;
import java.time.Duration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.slf4j.Logger;

public class CriticalComponentsHealthMonitor
implements HealthMonitor {
    private static final Duration HEALTH_MONITORING_PERIOD = Duration.ofSeconds(60L);
    private final Map<String, MonitoredComponent> monitoredComponents = new HashMap<String, MonitoredComponent>();
    private final Map<String, HealthReport> componentHealth = new HashMap<String, HealthReport>();
    private final Set<FailureListener> failureListeners = new HashSet<FailureListener>();
    private final ActorControl actor;
    private final Logger log;
    private volatile HealthReport healthReport = HealthReport.unhealthy(this).withMessage("Components are not yet initialized");
    private final String name;

    public CriticalComponentsHealthMonitor(String name, ActorControl actor, Logger log) {
        this.name = name;
        this.actor = actor;
        this.log = log;
    }

    @Override
    public void startMonitoring() {
        this.actor.runAtFixedRate(HEALTH_MONITORING_PERIOD, this::updateHealth);
    }

    @Override
    public void monitorComponent(String componentName) {
        this.actor.run(() -> this.componentHealth.put(componentName, HealthReport.unknown(componentName)));
    }

    @Override
    public void removeComponent(String componentName) {
        this.actor.run(() -> {
            MonitoredComponent monitoredComponent = this.monitoredComponents.remove(componentName);
            if (monitoredComponent != null) {
                this.componentHealth.remove(componentName);
                monitoredComponent.component.removeFailureListener(monitoredComponent);
            }
        });
    }

    @Override
    public void registerComponent(String componentName, HealthMonitorable component) {
        this.actor.run(() -> {
            MonitoredComponent monitoredComponent = new MonitoredComponent(componentName, component);
            this.monitoredComponents.put(componentName, monitoredComponent);
            this.componentHealth.put(componentName, component.getHealthReport());
            component.addFailureListener(monitoredComponent);
            this.calculateHealth();
        });
    }

    @Override
    public String getName() {
        return this.name;
    }

    @Override
    public HealthReport getHealthReport() {
        return this.healthReport;
    }

    @Override
    public void addFailureListener(FailureListener failureListener) {
        this.actor.run(() -> this.failureListeners.add(failureListener));
    }

    @Override
    public void removeFailureListener(FailureListener failureListener) {
        this.actor.run(() -> this.failureListeners.remove(failureListener));
    }

    private void updateHealth() {
        this.componentHealth.keySet().forEach(component -> this.componentHealth.put((String)component, this.getHealth((String)component)));
        this.calculateHealth();
    }

    private void calculateHealth() {
        HealthReport previousReport = this.healthReport;
        this.healthReport = this.calculateStatus();
        if (previousReport == this.healthReport) {
            return;
        }
        switch (this.healthReport.getStatus()) {
            case HEALTHY: {
                this.failureListeners.forEach(FailureListener::onRecovered);
                break;
            }
            case UNHEALTHY: {
                this.failureListeners.forEach(l -> l.onFailure(this.healthReport));
                break;
            }
            case DEAD: {
                this.failureListeners.forEach(l -> l.onUnrecoverableFailure(this.healthReport));
                break;
            }
            default: {
                this.log.warn("Unknown health status {}", (Object)this.healthReport);
            }
        }
        this.logComponentStatus(this.healthReport);
    }

    private void logComponentStatus(HealthReport status) {
        this.log.debug("Detected '{}' components. The current health status of components: {}", (Object)status.getStatus(), this.componentHealth.values());
    }

    private HealthReport calculateStatus() {
        Map componentByStatus = this.componentHealth.values().stream().collect(Collectors.toMap(HealthReport::getStatus, Function.identity(), (l, r) -> l));
        HealthReport deadReport = (HealthReport)componentByStatus.get((Object)HealthStatus.DEAD);
        HealthReport unhealthyReport = (HealthReport)componentByStatus.get((Object)HealthStatus.UNHEALTHY);
        if (deadReport != null) {
            return HealthReport.dead(this).withIssue(deadReport);
        }
        if (unhealthyReport != null) {
            return HealthReport.unhealthy(this).withIssue(unhealthyReport);
        }
        return HealthReport.healthy(this);
    }

    private HealthReport getHealth(String componentName) {
        MonitoredComponent monitoredComponent = this.monitoredComponents.get(componentName);
        if (monitoredComponent != null) {
            return monitoredComponent.component.getHealthReport();
        }
        return HealthReport.unknown(componentName);
    }

    private final class MonitoredComponent
    implements FailureListener {
        private final String componentName;
        private final HealthMonitorable component;

        private MonitoredComponent(String componentName, HealthMonitorable component) {
            this.componentName = componentName;
            this.component = component;
        }

        @Override
        public void onFailure(HealthReport report) {
            CriticalComponentsHealthMonitor.this.actor.run(() -> this.onComponentFailure(report));
        }

        @Override
        public void onRecovered() {
            CriticalComponentsHealthMonitor.this.actor.run(this::onComponentRecovered);
        }

        @Override
        public void onUnrecoverableFailure(HealthReport report) {
            CriticalComponentsHealthMonitor.this.actor.run(() -> this.onComponentDied(report));
        }

        private void onComponentFailure(HealthReport report) {
            if (!CriticalComponentsHealthMonitor.this.monitoredComponents.containsKey(this.componentName)) {
                return;
            }
            CriticalComponentsHealthMonitor.this.log.warn("{} failed, marking it as unhealthy: {}", (Object)this.componentName, (Object)CriticalComponentsHealthMonitor.this.healthReport);
            CriticalComponentsHealthMonitor.this.componentHealth.put(this.componentName, report);
            CriticalComponentsHealthMonitor.this.calculateHealth();
        }

        private void onComponentRecovered() {
            if (!CriticalComponentsHealthMonitor.this.monitoredComponents.containsKey(this.componentName)) {
                return;
            }
            CriticalComponentsHealthMonitor.this.log.info("{} recovered, marking it as healthy", (Object)this.componentName);
            CriticalComponentsHealthMonitor.this.componentHealth.put(this.componentName, HealthReport.healthy(this.component));
            CriticalComponentsHealthMonitor.this.calculateHealth();
        }

        private void onComponentDied(HealthReport report) {
            if (!CriticalComponentsHealthMonitor.this.monitoredComponents.containsKey(this.componentName)) {
                return;
            }
            CriticalComponentsHealthMonitor.this.log.error("{} failed, marking it as dead: {}", (Object)this.componentName, (Object)report);
            CriticalComponentsHealthMonitor.this.componentHealth.put(this.componentName, report);
            CriticalComponentsHealthMonitor.this.calculateHealth();
        }
    }
}

