/*
 * Decompiled with CFR 0.152.
 */
package io.cdap.plugin.gcp.bigquery.action;

import com.google.auth.Credentials;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.cloud.RetryOption;
import com.google.cloud.bigquery.BigQuery;
import com.google.cloud.bigquery.BigQueryException;
import com.google.cloud.bigquery.DatasetId;
import com.google.cloud.bigquery.EncryptionConfiguration;
import com.google.cloud.bigquery.Field;
import com.google.cloud.bigquery.FieldValueList;
import com.google.cloud.bigquery.Job;
import com.google.cloud.bigquery.JobConfiguration;
import com.google.cloud.bigquery.JobId;
import com.google.cloud.bigquery.JobInfo;
import com.google.cloud.bigquery.JobStatistics;
import com.google.cloud.bigquery.LegacySQLTypeName;
import com.google.cloud.bigquery.QueryJobConfiguration;
import com.google.cloud.bigquery.Schema;
import com.google.cloud.bigquery.TableId;
import com.google.cloud.bigquery.TableResult;
import com.google.cloud.kms.v1.CryptoKeyName;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import io.cdap.cdap.api.annotation.Description;
import io.cdap.cdap.api.annotation.Macro;
import io.cdap.cdap.api.annotation.Name;
import io.cdap.cdap.api.annotation.Plugin;
import io.cdap.cdap.etl.api.FailureCollector;
import io.cdap.cdap.etl.api.action.ActionContext;
import io.cdap.plugin.gcp.bigquery.action.AbstractBigQueryAction;
import io.cdap.plugin.gcp.bigquery.action.AbstractBigQueryActionConfig;
import io.cdap.plugin.gcp.bigquery.sink.BigQuerySinkUtils;
import io.cdap.plugin.gcp.bigquery.util.BigQueryUtil;
import io.cdap.plugin.gcp.common.CmekUtils;
import io.cdap.plugin.gcp.common.GCPUtils;
import java.io.IOException;
import java.util.Collections;
import java.util.Map;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Plugin(type="action")
@Name(value="BigQueryExecute")
@Description(value="Execute a Google BigQuery SQL.")
public final class BigQueryExecute
extends AbstractBigQueryAction {
    private static final Logger LOG = LoggerFactory.getLogger(BigQueryExecute.class);
    public static final String NAME = "BigQueryExecute";
    private static final String RECORDS_PROCESSED = "records.processed";
    private Config config;

    public void run(ActionContext context) throws Exception {
        FailureCollector collector = context.getFailureCollector();
        this.config.validate(collector, context.getArguments().asMap());
        QueryJobConfiguration.Builder builder = QueryJobConfiguration.newBuilder((String)this.config.getSql());
        if (this.config.getMode().equals((Object)QueryJobConfiguration.Priority.BATCH)) {
            builder.setPriority(QueryJobConfiguration.Priority.BATCH);
        } else {
            builder.setPriority(QueryJobConfiguration.Priority.INTERACTIVE);
        }
        CryptoKeyName cmekKeyName = CmekUtils.getCmekKey(this.config.cmekKey, context.getArguments().asMap(), collector);
        collector.getOrThrowException();
        String datasetName = this.config.getDataset();
        String tableName = this.config.getTable();
        String datasetProjectId = this.config.getDatasetProject();
        if (this.config.getStoreResults().booleanValue() && datasetProjectId != null && datasetName != null && tableName != null) {
            builder.setDestinationTable(TableId.of((String)datasetProjectId, (String)datasetName, (String)tableName));
        }
        if (this.config.shouldUseCache()) {
            builder.setUseQueryCache(Boolean.valueOf(true));
        }
        builder.setUseLegacySql(Boolean.valueOf(this.config.isLegacySQL()));
        JobId jobId = JobId.newBuilder().setRandomJob().setLocation(this.config.getLocation()).build();
        GoogleCredentials credentials = this.config.getServiceAccount() == null ? null : GCPUtils.loadServiceAccountCredentials(this.config.getServiceAccount(), this.config.isServiceAccountFilePath());
        BigQuery bigQuery = GCPUtils.getBigQuery(this.config.getProject(), credentials);
        if (this.config.getStoreResults().booleanValue() && !Strings.isNullOrEmpty((String)datasetName) && !Strings.isNullOrEmpty((String)tableName)) {
            BigQuerySinkUtils.createDatasetIfNotExists(bigQuery, DatasetId.of((String)datasetProjectId, (String)datasetName), this.config.getLocation(), cmekKeyName, () -> String.format("Unable to create BigQuery dataset '%s.%s'", datasetProjectId, datasetName));
            if (cmekKeyName != null) {
                builder.setDestinationEncryptionConfiguration(EncryptionConfiguration.newBuilder().setKmsKeyName(cmekKeyName.toString()).build());
            }
        }
        builder.setLabels(BigQueryUtil.getJobTags("bq_execute_plugin"));
        QueryJobConfiguration queryConfig = builder.build();
        Job queryJob = bigQuery.create(JobInfo.newBuilder((JobConfiguration)queryConfig).setJobId(jobId).build(), new BigQuery.JobOption[0]);
        LOG.info("Executing SQL as job {}.", (Object)jobId.getJob());
        LOG.debug("The BigQuery SQL is {}", (Object)this.config.getSql());
        queryJob = queryJob.waitFor(new RetryOption[0]);
        if (queryJob.getStatus().getError() != null) {
            throw new RuntimeException(queryJob.getStatus().getExecutionErrors().toString());
        }
        TableResult queryResults = queryJob.getQueryResults(new BigQuery.QueryResultsOption[0]);
        long rows = queryResults.getTotalRows();
        if (this.config.shouldSetAsArguments()) {
            if (rows == 0L || queryResults.getSchema() == null) {
                LOG.warn("The query result does not contain any row or schema, will not save the results in the arguments");
            } else {
                Schema schema = queryResults.getSchema();
                FieldValueList firstRow = (FieldValueList)queryResults.iterateAll().iterator().next();
                for (int i = 0; i < schema.getFields().size(); ++i) {
                    Field field = schema.getFields().get(i);
                    String name = field.getName();
                    if (field.getMode().equals((Object)Field.Mode.REPEATED)) {
                        LOG.warn("Field {} is an array, will not save the value in the argument", (Object)name);
                        continue;
                    }
                    if (field.getType().equals((Object)LegacySQLTypeName.RECORD)) {
                        LOG.warn("Field {} is a record type with nested schema, will not save the value in the argument", (Object)name);
                        continue;
                    }
                    context.getArguments().set(name, firstRow.get(name).getStringValue());
                }
            }
        }
        long processedBytes = ((JobStatistics.QueryStatistics)queryJob.getStatistics()).getTotalBytesProcessed();
        LOG.info("Job {} processed {} bytes", (Object)queryJob.getJobId(), (Object)processedBytes);
        ImmutableMap tags = new ImmutableMap.Builder().put((Object)"aet", (Object)"action").put((Object)"tpe", (Object)NAME).build();
        context.getMetrics().gauge(RECORDS_PROCESSED, rows);
        context.getMetrics().child((Map)tags).countLong("bytes.processed", processedBytes);
    }

    @Override
    public AbstractBigQueryActionConfig getConfig() {
        return this.config;
    }

    public static final class Config
    extends AbstractBigQueryActionConfig {
        private static final String MODE = "mode";
        private static final String SQL = "sql";
        private static final String DATASET = "dataset";
        private static final String TABLE = "table";
        private static final String NAME_LOCATION = "location";
        private static final int ERROR_CODE_NOT_FOUND = 404;
        private static final String STORE_RESULTS = "storeResults";
        @Description(value="Dialect of the SQL command. The value must be 'legacy' or 'standard'. If set to 'standard', the query will use BigQuery's standard SQL: https://cloud.google.com/bigquery/sql-reference/. If set to 'legacy', BigQuery's legacy SQL dialect will be used for this query.")
        @Macro
        private String dialect;
        @Name(value="sql")
        @Description(value="SQL command to execute.")
        @Macro
        private String sql;
        @Name(value="mode")
        @Description(value="Mode to execute the query in. The value must be 'batch' or 'interactive'. An interactive query is executed as soon as possible and counts towards the concurrent rate limit and the daily rate limit. A batch query is queued and started as soon as idle resources are available, usually within a few minutes. If the query hasn't started within 3 hours, its priority is changed to 'interactive'")
        @Macro
        private String mode;
        @Description(value="Use the cache when executing the query.")
        @Macro
        private String useCache;
        @Name(value="location")
        @Description(value="Location of the job. Must match the location of the dataset specified in the query. Defaults to 'US'")
        @Macro
        private String location;
        @Name(value="dataset")
        @Description(value="Dataset to store the query results in. If not specified, the results will not be stored.")
        @Macro
        @Nullable
        private String dataset;
        @Name(value="table")
        @Description(value="Table to store the query results in. If not specified, the results will not be stored.")
        @Macro
        @Nullable
        private String table;
        @Name(value="cmekKey")
        @Macro
        @Nullable
        @Description(value="The GCP customer managed encryption key (CMEK) name used to encrypt data written to the dataset or table created by the plugin to store the query results. It is only applicable when users choose to store the query results in a BigQuery table. More information can be found at https://cloud.google.com/data-fusion/docs/how-to/customer-managed-encryption-keys")
        private String cmekKey;
        @Description(value="Row as arguments. For example, if the query is 'select min(id) as min_id, max(id) as max_id from my_dataset.my_table',an arguments for 'min_id' and 'max_id' will be set based on the query results. Plugins further down the pipeline can thenreference these values with macros ${min_id} and ${max_id}.")
        @Macro
        private String rowAsArguments;
        @Name(value="storeResults")
        @Nullable
        @Description(value="Whether to store results in a BigQuery Table.")
        private Boolean storeResults;

        private Config(@Nullable String project, @Nullable String serviceAccountType, @Nullable String serviceFilePath, @Nullable String serviceAccountJson, @Nullable String dataset, @Nullable String table, @Nullable String location, @Nullable String cmekKey, @Nullable String dialect, @Nullable String sql, @Nullable String mode, @Nullable Boolean storeResults) {
            this.project = project;
            this.serviceAccountType = serviceAccountType;
            this.serviceFilePath = serviceFilePath;
            this.serviceAccountJson = serviceAccountJson;
            this.dataset = dataset;
            this.table = table;
            this.location = location;
            this.cmekKey = cmekKey;
            this.dialect = dialect;
            this.sql = sql;
            this.mode = mode;
            this.storeResults = storeResults;
        }

        public boolean isLegacySQL() {
            return this.dialect.equalsIgnoreCase("legacy");
        }

        public boolean shouldUseCache() {
            return this.useCache.equalsIgnoreCase("true");
        }

        public boolean shouldSetAsArguments() {
            return this.rowAsArguments.equalsIgnoreCase("true");
        }

        public String getLocation() {
            return this.location;
        }

        public String getSql() {
            return this.sql;
        }

        public Boolean getStoreResults() {
            return this.storeResults == null || this.storeResults != false;
        }

        public QueryJobConfiguration.Priority getMode() {
            return QueryJobConfiguration.Priority.valueOf((String)this.mode.toUpperCase());
        }

        @Nullable
        public String getDataset() {
            return this.dataset;
        }

        @Nullable
        public String getTable() {
            return this.table;
        }

        @Override
        public void validate(FailureCollector failureCollector) {
            this.validate(failureCollector, Collections.emptyMap());
        }

        public void validate(FailureCollector failureCollector, Map<String, String> arguments) {
            if (!this.containsMacro(MODE)) {
                try {
                    this.getMode();
                }
                catch (IllegalArgumentException e) {
                    failureCollector.addFailure(e.getMessage(), "The mode must be 'batch' or 'interactive'.").withConfigProperty(MODE);
                }
            }
            if (!this.containsMacro(SQL)) {
                if (Strings.isNullOrEmpty((String)this.sql)) {
                    failureCollector.addFailure("SQL not specified.", "Please specify a SQL to execute").withConfigProperty(SQL);
                } else if (this.tryGetProject() != null && !this.containsMacro("serviceFilePath") && !this.containsMacro("serviceAccountJSON")) {
                    BigQuery bigquery = this.getBigQuery(failureCollector);
                    this.validateSQLSyntax(failureCollector, bigquery);
                }
            }
            if (!this.containsMacro(DATASET) && !this.containsMacro(TABLE) && Strings.isNullOrEmpty((String)this.dataset) != Strings.isNullOrEmpty((String)this.table)) {
                failureCollector.addFailure("Dataset and table must be specified together.", null).withConfigProperty(TABLE).withConfigProperty(DATASET);
            }
            if (!this.containsMacro(DATASET)) {
                BigQueryUtil.validateDataset(this.dataset, DATASET, failureCollector);
            }
            if (!this.containsMacro(TABLE)) {
                BigQueryUtil.validateTable(this.table, TABLE, failureCollector);
            }
            if (!this.containsMacro("cmekKey")) {
                this.validateCmekKey(failureCollector, arguments);
            }
            failureCollector.getOrThrowException();
        }

        void validateCmekKey(FailureCollector failureCollector, Map<String, String> arguments) {
            CryptoKeyName cmekKeyName = CmekUtils.getCmekKey(this.cmekKey, arguments, failureCollector);
            if (cmekKeyName == null || this.containsMacro(DATASET) || this.containsMacro(NAME_LOCATION) || this.containsMacro(TABLE) || this.projectOrServiceAccountContainsMacro() || Strings.isNullOrEmpty((String)this.dataset) || Strings.isNullOrEmpty((String)this.table) || this.containsMacro("datasetProject")) {
                return;
            }
            String datasetProjectId = this.getDatasetProject();
            String datasetName = this.getDataset();
            DatasetId datasetId = DatasetId.of((String)datasetProjectId, (String)datasetName);
            TableId tableId = TableId.of((String)datasetProjectId, (String)datasetName, (String)this.getTable());
            BigQuery bigQuery = this.getBigQuery(failureCollector);
            if (bigQuery == null) {
                return;
            }
            CmekUtils.validateCmekKeyAndDatasetOrTableLocation(bigQuery, datasetId, tableId, cmekKeyName, this.location, failureCollector);
        }

        public void validateSQLSyntax(FailureCollector failureCollector, BigQuery bigQuery) {
            QueryJobConfiguration queryJobConfiguration = QueryJobConfiguration.newBuilder((String)this.sql).setDryRun(Boolean.valueOf(true)).build();
            try {
                bigQuery.create(JobInfo.of((JobConfiguration)queryJobConfiguration), new BigQuery.JobOption[0]);
            }
            catch (BigQueryException e) {
                String errorMessage = e.getCode() == 404 ? String.format("Resource was not found. Please verify the resource name. If the resource will be created at runtime, then update to use a macro for the resource name. Error message received was: %s", e.getMessage()) : e.getMessage();
                failureCollector.addFailure(String.format("%s.", errorMessage), "Please specify a valid query.").withConfigProperty(SQL);
            }
        }

        private BigQuery getBigQuery(FailureCollector failureCollector) {
            GoogleCredentials credentials = null;
            try {
                credentials = this.getServiceAccount() == null ? null : GCPUtils.loadServiceAccountCredentials(this.getServiceAccount(), this.isServiceAccountFilePath());
            }
            catch (IOException e) {
                failureCollector.addFailure(e.getMessage(), null);
                failureCollector.getOrThrowException();
            }
            return GCPUtils.getBigQuery(this.getProject(), (Credentials)credentials);
        }

        public static Builder builder() {
            return new Builder();
        }

        public static class Builder {
            private String serviceAccountType;
            private String serviceFilePath;
            private String serviceAccountJson;
            private String project;
            private String dataset;
            private String table;
            private String cmekKey;
            private String location;
            private String dialect;
            private String sql;
            private String mode;
            private Boolean storeResults;

            public Builder setProject(@Nullable String project) {
                this.project = project;
                return this;
            }

            public Builder setServiceAccountType(@Nullable String serviceAccountType) {
                this.serviceAccountType = serviceAccountType;
                return this;
            }

            public Builder setServiceFilePath(@Nullable String serviceFilePath) {
                this.serviceFilePath = serviceFilePath;
                return this;
            }

            public Builder setServiceAccountJson(@Nullable String serviceAccountJson) {
                this.serviceAccountJson = serviceAccountJson;
                return this;
            }

            public Builder setDataset(@Nullable String dataset) {
                this.dataset = dataset;
                return this;
            }

            public Builder setTable(@Nullable String table) {
                this.table = table;
                return this;
            }

            public Builder setCmekKey(@Nullable String cmekKey) {
                this.cmekKey = cmekKey;
                return this;
            }

            public Builder setLocation(@Nullable String location) {
                this.location = location;
                return this;
            }

            public Builder setDialect(@Nullable String dialect) {
                this.dialect = dialect;
                return this;
            }

            public Builder setMode(@Nullable String mode) {
                this.mode = mode;
                return this;
            }

            public Builder setSql(@Nullable String sql) {
                this.sql = sql;
                return this;
            }

            public Config build() {
                return new Config(this.project, this.serviceAccountType, this.serviceFilePath, this.serviceAccountJson, this.dataset, this.table, this.location, this.cmekKey, this.dialect, this.sql, this.mode, this.storeResults);
            }
        }
    }
}

