/*
 * Decompiled with CFR 0.152.
 */
package io.cdap.plugin.gcp.bigquery.connector;

import com.google.api.gax.paging.Page;
import com.google.auth.Credentials;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.cloud.RetryOption;
import com.google.cloud.bigquery.BigQuery;
import com.google.cloud.bigquery.BigQueryException;
import com.google.cloud.bigquery.Dataset;
import com.google.cloud.bigquery.DatasetId;
import com.google.cloud.bigquery.Job;
import com.google.cloud.bigquery.JobConfiguration;
import com.google.cloud.bigquery.JobId;
import com.google.cloud.bigquery.JobInfo;
import com.google.cloud.bigquery.QueryJobConfiguration;
import com.google.cloud.bigquery.Table;
import com.google.cloud.bigquery.TableDefinition;
import com.google.cloud.bigquery.TableId;
import com.google.cloud.bigquery.TableResult;
import io.cdap.cdap.api.annotation.Category;
import io.cdap.cdap.api.annotation.Description;
import io.cdap.cdap.api.annotation.Name;
import io.cdap.cdap.api.annotation.Plugin;
import io.cdap.cdap.api.data.format.StructuredRecord;
import io.cdap.cdap.api.data.schema.Schema;
import io.cdap.cdap.etl.api.FailureCollector;
import io.cdap.cdap.etl.api.connector.BrowseDetail;
import io.cdap.cdap.etl.api.connector.BrowseEntity;
import io.cdap.cdap.etl.api.connector.BrowseRequest;
import io.cdap.cdap.etl.api.connector.ConnectorContext;
import io.cdap.cdap.etl.api.connector.ConnectorSpec;
import io.cdap.cdap.etl.api.connector.ConnectorSpecRequest;
import io.cdap.cdap.etl.api.connector.DirectConnector;
import io.cdap.cdap.etl.api.connector.PluginSpec;
import io.cdap.cdap.etl.api.connector.SampleRequest;
import io.cdap.cdap.etl.api.connector.SampleType;
import io.cdap.cdap.etl.api.validation.ValidationException;
import io.cdap.plugin.gcp.bigquery.connector.BigQueryConnectorSpecificConfig;
import io.cdap.plugin.gcp.bigquery.connector.BigQueryPath;
import io.cdap.plugin.gcp.bigquery.util.BigQueryDataParser;
import io.cdap.plugin.gcp.bigquery.util.BigQueryUtil;
import io.cdap.plugin.gcp.common.GCPUtils;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.UUID;
import javax.annotation.Nullable;
import org.threeten.bp.Duration;

@Plugin(type="connector")
@Name(value="BigQuery")
@Category(value="Google Cloud Platform")
@Description(value="Connection to access data in BigQuery datasets and tables.")
public final class BigQueryConnector
implements DirectConnector {
    public static final String NAME = "BigQuery";
    public static final String ENTITY_TYPE_DATASET = "dataset";
    private static final int ERROR_CODE_NOT_FOUND = 404;
    private BigQueryConnectorSpecificConfig config;

    BigQueryConnector(BigQueryConnectorSpecificConfig config) {
        this.config = config;
    }

    public List<StructuredRecord> sample(ConnectorContext context, SampleRequest sampleRequest) throws IOException {
        BigQueryPath path = new BigQueryPath(sampleRequest.getPath());
        String table = path.getTable();
        if (table == null) {
            throw new IllegalArgumentException("Path should contain both dataset and table name.");
        }
        String dataset = path.getDataset();
        String query = this.getTableQuery(String.format("`%s.%s.%s`", this.config.getDatasetProject(), dataset, table), sampleRequest.getLimit(), SampleType.fromString((String)((String)sampleRequest.getProperties().get("sampleType"))), (String)sampleRequest.getProperties().get("strata"), UUID.randomUUID().toString().replace("-", "_"));
        String id = UUID.randomUUID().toString();
        return this.getQueryResult(this.waitForJob(this.getBigQuery(this.config.getProject()), query, sampleRequest.getTimeoutMs(), id), id);
    }

    public void test(ConnectorContext context) throws ValidationException {
        FailureCollector failureCollector = context.getFailureCollector();
        String project = this.config.tryGetProject();
        if (project == null) {
            failureCollector.addFailure("Could not detect Google Cloud project id from the environment.", "Please specify a project id.");
        }
        GoogleCredentials credentials = null;
        if (this.config.isServiceAccountJson().booleanValue() || this.config.getServiceAccountFilePath() != null) {
            try {
                credentials = GCPUtils.loadServiceAccountCredentials(this.config.getServiceAccount(), this.config.isServiceAccountFilePath());
            }
            catch (Exception e) {
                failureCollector.addFailure(String.format("Service account key provided is not valid: %s", e.getMessage()), "Please provide a valid service account key.");
            }
        }
        if (!failureCollector.getValidationFailures().isEmpty()) {
            return;
        }
        try {
            BigQuery bigQuery = GCPUtils.getBigQuery(this.config.getDatasetProject(), (Credentials)credentials);
            bigQuery.listDatasets(new BigQuery.DatasetListOption[]{BigQuery.DatasetListOption.pageSize((long)1L)});
        }
        catch (Exception e) {
            failureCollector.addFailure(String.format("Could not connect to BigQuery: %s", e.getMessage()), "Please specify correct connection properties.");
        }
    }

    public BrowseDetail browse(ConnectorContext context, BrowseRequest browseRequest) throws IOException {
        BigQueryPath path = new BigQueryPath(browseRequest.getPath());
        String dataset = path.getDataset();
        if (dataset == null) {
            return this.config.rootDataset == null ? this.listDatasets(this.getBigQuery(this.config.getDatasetProject()), browseRequest.getLimit()) : BrowseDetail.builder().setTotalCount(1).addEntity(BrowseEntity.builder((String)this.config.rootDataset, (String)("/" + this.config.rootDataset), (String)ENTITY_TYPE_DATASET).canBrowse(true).build()).build();
        }
        String table = path.getTable();
        if (table == null) {
            return this.listTables(this.getBigQuery(this.config.getProject()), this.config.getDatasetProject(), dataset, browseRequest.getLimit());
        }
        return this.getTableDetail(this.getBigQuery(this.config.getProject()), this.config.getDatasetProject(), dataset, table);
    }

    private BrowseDetail getTableDetail(BigQuery bigQuery, String datasetProject, String datasetName, String tableName) {
        Table table = this.getTable(bigQuery, datasetProject, datasetName, tableName);
        return BrowseDetail.builder().addEntity(BrowseEntity.builder((String)tableName, (String)("/" + datasetName + "/" + tableName), (String)table.getDefinition().getType().name().toLowerCase()).canSample(true).build()).setTotalCount(1).build();
    }

    private Table getTable(BigQuery bigQuery, String datasetProject, String datasetName, String tableName) {
        Table table = bigQuery.getTable(TableId.of((String)datasetProject, (String)datasetName, (String)tableName), new BigQuery.TableOption[0]);
        if (table == null) {
            throw new IllegalArgumentException(String.format("Cannot find tableName: %s.%s.", datasetName, tableName));
        }
        return table;
    }

    private BrowseDetail listTables(BigQuery bigQuery, String datasetProject, String dataset, @Nullable Integer limit) {
        int countLimit = limit == null || limit <= 0 ? Integer.MAX_VALUE : limit;
        int count = 0;
        BrowseDetail.Builder browseDetailBuilder = BrowseDetail.builder();
        DatasetId datasetId = DatasetId.of((String)datasetProject, (String)dataset);
        Page tablePage = null;
        try {
            tablePage = bigQuery.listTables(datasetId, new BigQuery.TableListOption[0]);
        }
        catch (BigQueryException e) {
            if (e.getCode() == 404) {
                throw new IllegalArgumentException(String.format("Cannot find dataset: %s.", dataset), e);
            }
            throw e;
        }
        String parentPath = "/" + dataset + "/";
        for (Table table : tablePage.iterateAll()) {
            if (count >= countLimit) break;
            String name = table.getTableId().getTable();
            browseDetailBuilder.addEntity(BrowseEntity.builder((String)name, (String)(parentPath + name), (String)table.getDefinition().getType().name().toLowerCase()).canSample(true).build());
            ++count;
        }
        return browseDetailBuilder.setTotalCount(count).build();
    }

    private BrowseDetail listDatasets(BigQuery bigQuery, Integer limit) {
        Page datasetPage = this.config.showHiddenDatasets() ? bigQuery.listDatasets(new BigQuery.DatasetListOption[]{BigQuery.DatasetListOption.all()}) : bigQuery.listDatasets(new BigQuery.DatasetListOption[0]);
        int countLimit = limit == null || limit <= 0 ? Integer.MAX_VALUE : limit;
        int count = 0;
        BrowseDetail.Builder browseDetailBuilder = BrowseDetail.builder();
        for (Dataset dataset : datasetPage.iterateAll()) {
            if (count >= countLimit) break;
            String name = dataset.getDatasetId().getDataset();
            browseDetailBuilder.addEntity(BrowseEntity.builder((String)name, (String)("/" + name), (String)ENTITY_TYPE_DATASET).canBrowse(true).build());
            ++count;
        }
        return browseDetailBuilder.setTotalCount(count).build();
    }

    private BigQuery getBigQuery(String project) throws IOException {
        GoogleCredentials credentials = null;
        if (this.config.isServiceAccountJson().booleanValue() || this.config.getServiceAccountFilePath() != null) {
            credentials = GCPUtils.loadServiceAccountCredentials(this.config.getServiceAccount(), this.config.isServiceAccountFilePath());
        }
        return GCPUtils.getBigQuery(project, credentials);
    }

    protected String getTableQuery(String tableName, int limit, SampleType sampleType, @Nullable String strata, String sessionID) {
        switch (sampleType) {
            case RANDOM: {
                return String.format("WITH table AS (\n  SELECT *, RAND() AS r_%s\n  FROM %s\n  WHERE RAND() < 2*%d/(SELECT COUNT(*) FROM %s)\n)\nSELECT * EXCEPT (r_%s)\nFROM table\nORDER BY r_%s\nLIMIT %d", sessionID, tableName, limit, tableName, sessionID, sessionID, limit);
            }
            case STRATIFIED: {
                if (strata == null) {
                    throw new IllegalArgumentException("No strata column given.");
                }
                return String.format("SELECT * EXCEPT (`sqn_%s`, `c_%s`)\nFROM (\nSELECT *, row_number() OVER (ORDER BY %s, RAND()) AS sqn_%s,\nCOUNT(*) OVER () as c_%s,\nFROM %s\n) %s\nWHERE MOD(sqn_%s, CAST(c_%s / %d AS INT64)) = 1\nORDER BY %s\nLIMIT %d", sessionID, sessionID, strata, sessionID, sessionID, tableName, tableName, sessionID, sessionID, limit, strata, limit);
            }
        }
        return String.format("SELECT * FROM %s LIMIT %d", tableName, limit);
    }

    private Job waitForJob(BigQuery bigQuery, String query, @Nullable Long timeoutMs, String id) throws IOException {
        QueryJobConfiguration queryConfig = QueryJobConfiguration.newBuilder((String)query).build();
        JobId jobId = JobId.of((String)id);
        Job queryJob = bigQuery.create(JobInfo.newBuilder((JobConfiguration)queryConfig).setJobId(jobId).build(), new BigQuery.JobOption[0]);
        try {
            if (timeoutMs == null) {
                return queryJob.waitFor(new RetryOption[0]);
            }
            return queryJob.waitFor(new RetryOption[]{RetryOption.totalTimeout((Duration)Duration.ofMillis((long)timeoutMs))});
        }
        catch (InterruptedException e) {
            throw new IOException(String.format("Query job %s interrupted.", id), e);
        }
    }

    protected List<StructuredRecord> getQueryResult(@Nullable Job queryJob, String id) throws IOException {
        TableResult result;
        if (queryJob == null) {
            throw new IOException(String.format("Job %s no longer exists.", id));
        }
        if (!queryJob.isDone()) {
            queryJob.cancel();
            throw new IOException(String.format("Job %s timed out.", id));
        }
        if (queryJob.getStatus().getError() != null) {
            throw new IOException(String.format("Failed to query table : %s", queryJob.getStatus().getError().toString()));
        }
        try {
            result = queryJob.getQueryResults(new BigQuery.QueryResultsOption[0]);
        }
        catch (InterruptedException e) {
            throw new IOException("Query results interrupted.", e);
        }
        return BigQueryDataParser.parse(result);
    }

    public ConnectorSpec generateSpec(ConnectorContext context, ConnectorSpecRequest connectorSpecRequest) throws IOException {
        String tableName;
        BigQueryPath path = new BigQueryPath(connectorSpecRequest.getPath());
        ConnectorSpec.Builder specBuilder = ConnectorSpec.builder();
        HashMap<String, String> properties = new HashMap<String, String>();
        properties.put("useConnection", "true");
        properties.put("connection", connectorSpecRequest.getConnectionWithMacro());
        String datasetName = path.getDataset();
        if (datasetName != null) {
            properties.put(ENTITY_TYPE_DATASET, datasetName);
        }
        if ((tableName = path.getTable()) != null) {
            properties.put("table", tableName);
            Table table = this.getTable(this.getBigQuery(this.config.getProject()), this.config.getDatasetProject(), datasetName, tableName);
            TableDefinition definition = table.getDefinition();
            Schema schema = BigQueryUtil.getTableSchema(definition.getSchema(), null);
            specBuilder.setSchema(schema);
            if (definition.getType() != TableDefinition.Type.TABLE) {
                properties.put("enableQueryingViews", "true");
            }
        }
        return specBuilder.addRelatedPlugin(new PluginSpec("BigQueryTable", "batchsource", properties)).addRelatedPlugin(new PluginSpec("BigQueryTable", "batchsink", properties)).addRelatedPlugin(new PluginSpec("BigQueryMultiTable", "batchsink", properties)).addRelatedPlugin(new PluginSpec("BigQueryPushdownEngine", "sqlengine", properties)).addSupportedSampleType(SampleType.RANDOM).addSupportedSampleType(SampleType.STRATIFIED).build();
    }
}

