/*
 * Decompiled with CFR 0.152.
 */
package io.cdap.plugin.gcp.bigquery.sqlengine;

import com.google.cloud.RetryOption;
import com.google.cloud.bigquery.BigQuery;
import com.google.cloud.bigquery.BigQueryException;
import com.google.cloud.bigquery.Dataset;
import com.google.cloud.bigquery.DatasetId;
import com.google.cloud.bigquery.Job;
import com.google.cloud.bigquery.JobConfiguration;
import com.google.cloud.bigquery.JobId;
import com.google.cloud.bigquery.JobInfo;
import com.google.cloud.bigquery.JobStatistics;
import com.google.cloud.bigquery.QueryJobConfiguration;
import com.google.cloud.bigquery.StandardTableDefinition;
import com.google.cloud.bigquery.Table;
import com.google.cloud.bigquery.TableDefinition;
import com.google.cloud.bigquery.TableId;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.google.common.reflect.TypeToken;
import com.google.gson.Gson;
import io.cdap.cdap.api.data.schema.Schema;
import io.cdap.cdap.api.metrics.Metrics;
import io.cdap.cdap.etl.api.engine.sql.dataset.SQLDataset;
import io.cdap.cdap.etl.api.engine.sql.request.SQLReadRequest;
import io.cdap.cdap.etl.api.engine.sql.request.SQLReadResult;
import io.cdap.plugin.gcp.bigquery.source.BigQuerySourceConfig;
import io.cdap.plugin.gcp.bigquery.sqlengine.BigQuerySQLDataset;
import io.cdap.plugin.gcp.bigquery.sqlengine.BigQuerySQLEngine;
import io.cdap.plugin.gcp.bigquery.sqlengine.BigQuerySQLEngineConfig;
import io.cdap.plugin.gcp.bigquery.sqlengine.util.BigQuerySQLEngineUtils;
import io.cdap.plugin.gcp.bigquery.util.BigQueryUtil;
import java.lang.reflect.Type;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BigQueryReadDataset
implements SQLDataset,
BigQuerySQLDataset {
    private static final Logger LOG = LoggerFactory.getLogger(BigQueryReadDataset.class);
    private static final Gson GSON = new Gson();
    public static final String SQL_INPUT_CONFIG = "config";
    public static final String SQL_INPUT_FIELDS = "fields";
    public static final String SQL_INPUT_SCHEMA = "schema";
    private static final Type LIST_OF_STRINGS_TYPE = new TypeToken<ArrayList<String>>(){}.getType();
    private static final String BQ_PUSHDOWN_OPERATION_TAG = "read";
    private final BigQuerySQLEngineConfig sqlEngineConfig;
    private final BigQuery bigQuery;
    private final String datasetName;
    private final SQLReadRequest readRequest;
    private final TableId destinationTableId;
    private final String jobId;
    private Schema schema;
    private Long numRows;
    private Metrics metrics;

    private BigQueryReadDataset(String datasetName, BigQuerySQLEngineConfig sqlEngineConfig, BigQuery bigQuery, SQLReadRequest readRequest, TableId destinationTableId, String jobId, Metrics metrics) {
        this.datasetName = datasetName;
        this.sqlEngineConfig = sqlEngineConfig;
        this.bigQuery = bigQuery;
        this.readRequest = readRequest;
        this.destinationTableId = destinationTableId;
        this.jobId = jobId;
        this.metrics = metrics;
    }

    public static BigQueryReadDataset getInstance(String datasetName, BigQuerySQLEngineConfig sqlEngineConfig, BigQuery bigQuery, SQLReadRequest readRequest, TableId destinationTableId, Metrics metrics) {
        String jobId = BigQuerySQLEngineUtils.newIdentifier();
        return new BigQueryReadDataset(datasetName, sqlEngineConfig, bigQuery, readRequest, destinationTableId, jobId, metrics);
    }

    public SQLReadResult read() {
        Object result = null;
        AtomicReference<Object> newSourceTable = new AtomicReference<Object>(null);
        try {
            return this.readInternal(this.readRequest, newSourceTable);
        }
        catch (InterruptedException e) {
            LOG.error("Interrupted exception during BigQuery read operation.", (Throwable)e);
        }
        catch (BigQueryException bqe) {
            LOG.error("BigQuery exception during BigQuery read operation", (Throwable)bqe);
        }
        catch (Exception e) {
            LOG.error("Exception during BigQuery read operation", (Throwable)e);
        }
        if (result == null || !result.isSuccessful()) {
            this.tryDeleteTable(this.destinationTableId);
        }
        return SQLReadResult.failure((String)this.readRequest.getDatasetName());
    }

    private SQLReadResult readInternal(SQLReadRequest readRequest, AtomicReference<TableId> newSourceTable) throws BigQueryException, InterruptedException {
        Table sourceTable;
        String datasetName = readRequest.getDatasetName();
        if (!BigQuerySQLEngine.class.getName().equals(readRequest.getInput().getSqlEngineClassName())) {
            LOG.debug("Got output for another SQL engine {}, skipping", (Object)readRequest.getInput().getSqlEngineClassName());
            return SQLReadResult.unsupported((String)datasetName);
        }
        Map arguments = readRequest.getInput().getArguments();
        BigQuerySourceConfig sourceConfig = (BigQuerySourceConfig)((Object)GSON.fromJson((String)arguments.get(SQL_INPUT_CONFIG), BigQuerySourceConfig.class));
        this.schema = (Schema)GSON.fromJson((String)arguments.get(SQL_INPUT_SCHEMA), Schema.class);
        List fields = (List)GSON.fromJson((String)arguments.get(SQL_INPUT_FIELDS), LIST_OF_STRINGS_TYPE);
        String sourceProject = sourceConfig.getDatasetProject();
        String sourceDataset = sourceConfig.getDataset();
        String sourceTableName = sourceConfig.getTable();
        TableId sourceTableId = TableId.of((String)sourceProject, (String)sourceDataset, (String)sourceTableName);
        DatasetId sourceDatasetId = DatasetId.of((String)sourceTableId.getProject(), (String)sourceTableId.getDataset());
        DatasetId destinationDatasetId = DatasetId.of((String)this.destinationTableId.getProject(), (String)this.destinationTableId.getDataset());
        Dataset srcDataset = this.bigQuery.getDataset(sourceDatasetId, new BigQuery.DatasetOption[0]);
        Dataset destDataset = this.bigQuery.getDataset(destinationDatasetId, new BigQuery.DatasetOption[0]);
        if (srcDataset == null || destDataset == null) {
            LOG.warn("Direct table read is not supported when the datasets are not created.");
            return SQLReadResult.unsupported((String)datasetName);
        }
        if (!Objects.equals(srcDataset.getLocation(), destDataset.getLocation())) {
            LOG.error("Direct table read is only supported if both datasets are in the same location. '{}' is '{}' , '{}' is '{}' .", new Object[]{sourceDatasetId.getDataset(), srcDataset.getLocation(), destinationDatasetId.getDataset(), destDataset.getLocation()});
            return SQLReadResult.unsupported((String)datasetName);
        }
        String jobLocation = srcDataset.getLocation();
        try {
            sourceTable = this.bigQuery.getTable(sourceTableId, new BigQuery.TableOption[0]);
        }
        catch (BigQueryException e) {
            throw new IllegalArgumentException("Unable to get details about the BigQuery table: " + e.getMessage(), e);
        }
        Long tableTTL = -1L;
        if (!this.sqlEngineConfig.shouldRetainTables().booleanValue() && this.sqlEngineConfig.getTempTableTTLHours() > 0) {
            long ttlMillis = TimeUnit.MILLISECONDS.convert(this.sqlEngineConfig.getTempTableTTLHours().intValue(), TimeUnit.HOURS);
            tableTTL = Instant.now().toEpochMilli() + ttlMillis;
        }
        JobConfiguration queryConfig = this.getBQQueryJobConfiguration(sourceTable, sourceTableId, sourceConfig.getFilter(), sourceConfig.getPartitionFrom(), sourceConfig.getPartitionTo(), tableTTL);
        return this.executeBigQueryJob(queryConfig, sourceTable, sourceTableId, BigQueryJobType.QUERY, jobLocation);
    }

    private SQLReadResult executeBigQueryJob(JobConfiguration jobConfiguration, Table sourceTable, TableId sourceTableId, BigQueryJobType bigQueryJobType, String jobLocation) throws InterruptedException {
        JobId bqJobId = JobId.newBuilder().setJob(this.jobId).setLocation(jobLocation).setProject(this.sqlEngineConfig.getProject()).build();
        Job bqJob = this.bigQuery.create(JobInfo.newBuilder((JobConfiguration)jobConfiguration).setJobId(bqJobId).build(), new BigQuery.JobOption[0]);
        if ((bqJob = bqJob.waitFor(new RetryOption[0])).getStatus().getError() != null) {
            BigQuerySQLEngineUtils.logJobMetrics(bqJob, this.metrics);
            LOG.error("Error executing BigQuery Job of type {} : '{}' in Project '{}', Dataset '{}': {}", new Object[]{bigQueryJobType, this.jobId, this.sqlEngineConfig.getProject(), this.sqlEngineConfig.getDatasetProject(), bqJob.getStatus().getError().toString()});
            return SQLReadResult.failure((String)this.datasetName);
        }
        long numRows = sourceTable.getNumRows().longValue();
        if (bigQueryJobType.equals((Object)BigQueryJobType.QUERY)) {
            JobStatistics.QueryStatistics queryJobStats = (JobStatistics.QueryStatistics)bqJob.getStatistics();
            numRows = queryJobStats != null && queryJobStats.getNumDmlAffectedRows() != null ? queryJobStats.getNumDmlAffectedRows() : numRows;
        }
        LOG.info("Executed read operation for {} records from {}.{}.{} into {}.{}.{}", new Object[]{numRows, sourceTableId.getProject(), sourceTableId.getDataset(), sourceTableId.getTable(), this.destinationTableId.getProject(), this.destinationTableId.getDataset(), this.destinationTableId.getTable()});
        BigQuerySQLEngineUtils.logJobMetrics(bqJob, this.metrics);
        return SQLReadResult.success((String)this.datasetName, (SQLDataset)this);
    }

    private JobConfiguration getBQQueryJobConfiguration(Table sourceTable, TableId sourceTableId, String filter, String partitionFromDate, String partitionToDate, Long tableTTL) {
        BigQuerySQLEngineUtils.createEmptyTableWithSourceConfig(this.bigQuery, this.destinationTableId.getProject(), this.destinationTableId.getDataset(), this.destinationTableId.getTable(), sourceTable, tableTTL);
        String query = String.format("SELECT * FROM `%s.%s.%s`", sourceTableId.getProject(), sourceTableId.getDataset(), sourceTableId.getTable());
        StringBuilder condition = new StringBuilder();
        StandardTableDefinition tableDefinition = (StandardTableDefinition)Objects.requireNonNull(sourceTable).getDefinition();
        TableDefinition.Type type = tableDefinition.getType();
        if (type != TableDefinition.Type.VIEW && type != TableDefinition.Type.MATERIALIZED_VIEW && type != TableDefinition.Type.EXTERNAL) {
            condition.append(BigQueryUtil.generateTimePartitionCondition(tableDefinition, partitionFromDate, partitionToDate));
        }
        if (!Strings.isNullOrEmpty((String)filter)) {
            if (condition.length() == 0) {
                condition.append(filter);
            } else {
                condition.append(" and (").append(filter).append(")");
            }
        }
        if (condition.length() > 0) {
            query = String.format("%s WHERE %s", query, condition);
        }
        LOG.info("Reading data from `{}.{}.{}` to `{}.{}.{}` using SQL statement: {} ", new Object[]{sourceTableId.getProject(), sourceTableId.getDataset(), sourceTableId.getTable(), this.destinationTableId.getProject(), this.destinationTableId.getDataset(), this.destinationTableId.getTable(), query});
        QueryJobConfiguration.Builder queryConfigBuilder = QueryJobConfiguration.newBuilder((String)query).setDestinationTable(this.destinationTableId).setCreateDisposition(JobInfo.CreateDisposition.CREATE_NEVER).setWriteDisposition(JobInfo.WriteDisposition.WRITE_APPEND).setPriority(this.sqlEngineConfig.getJobPriority()).setLabels(BigQuerySQLEngineUtils.getJobTags(BQ_PUSHDOWN_OPERATION_TAG));
        return queryConfigBuilder.build();
    }

    @VisibleForTesting
    QueryJobConfiguration.Builder getQueryBuilder(Table sourceTable, TableId sourceTableId, TableId destinationTableId, List<String> fields, String filter, String partitionFromDate, String partitionToDate) {
        String query = String.format("SELECT %s FROM `%s.%s.%s`", String.join((CharSequence)",", fields), sourceTableId.getProject(), sourceTableId.getDataset(), sourceTableId.getTable());
        StringBuilder condition = new StringBuilder();
        StandardTableDefinition tableDefinition = (StandardTableDefinition)Objects.requireNonNull(sourceTable).getDefinition();
        TableDefinition.Type type = tableDefinition.getType();
        if (type != TableDefinition.Type.VIEW && type != TableDefinition.Type.MATERIALIZED_VIEW && type != TableDefinition.Type.EXTERNAL) {
            condition.append(BigQueryUtil.generateTimePartitionCondition(tableDefinition, partitionFromDate, partitionToDate));
        }
        if (!Strings.isNullOrEmpty((String)filter)) {
            if (condition.length() == 0) {
                condition.append(filter);
            } else {
                condition.append(" and (").append(filter).append(")");
            }
        }
        if (condition.length() > 0) {
            query = String.format("%s WHERE %s", query, condition);
        }
        LOG.info("Reading data from `{}.{}.{}` to `{}.{}.{}` using SQL statement: {} ", new Object[]{sourceTableId.getProject(), sourceTableId.getDataset(), sourceTableId.getTable(), destinationTableId.getProject(), destinationTableId.getDataset(), destinationTableId.getTable(), query});
        return QueryJobConfiguration.newBuilder((String)query).setDestinationTable(destinationTableId).setCreateDisposition(JobInfo.CreateDisposition.CREATE_NEVER).setWriteDisposition(JobInfo.WriteDisposition.WRITE_APPEND).setPriority(this.sqlEngineConfig.getJobPriority()).setLabels(BigQuerySQLEngineUtils.getJobTags(BQ_PUSHDOWN_OPERATION_TAG));
    }

    protected void tryDeleteTable(TableId table) {
        try {
            this.bigQuery.delete(table);
        }
        catch (BigQueryException bqe) {
            LOG.error("Unable to delete table {}.{}.{}. This may cause the pipeline to fail", new Object[]{table.getProject(), table.getDataset(), table.getTable(), bqe});
        }
    }

    @Override
    public String getBigQueryProject() {
        return this.destinationTableId.getProject();
    }

    @Override
    public String getBigQueryDataset() {
        return this.destinationTableId.getDataset();
    }

    @Override
    public String getBigQueryTable() {
        return this.destinationTableId.getTable();
    }

    @Override
    @Nullable
    public String getJobId() {
        return this.jobId;
    }

    @Override
    @Nullable
    public String getGCSPath() {
        return null;
    }

    public long getNumRows() {
        if (this.numRows == null) {
            this.numRows = BigQuerySQLEngineUtils.getNumRows(this.bigQuery, DatasetId.of((String)this.destinationTableId.getProject(), (String)this.destinationTableId.getDataset()), this.destinationTableId.getTable());
        }
        return this.numRows;
    }

    public String getDatasetName() {
        return this.datasetName;
    }

    public Schema getSchema() {
        return this.schema;
    }

    private static enum BigQueryJobType {
        QUERY,
        COPY,
        COPY_SNAPSHOT;

    }
}

