/*
 * Decompiled with CFR 0.152.
 */
package io.cdap.plugin.gcp.bigquery.sqlengine.util;

import com.google.cloud.bigquery.BigQuery;
import com.google.cloud.bigquery.DatasetId;
import com.google.cloud.bigquery.Field;
import com.google.cloud.bigquery.Job;
import com.google.cloud.bigquery.JobStatistics;
import com.google.cloud.bigquery.QueryStage;
import com.google.cloud.bigquery.StandardTableDefinition;
import com.google.cloud.bigquery.Table;
import com.google.cloud.bigquery.TableDefinition;
import com.google.cloud.bigquery.TableId;
import com.google.cloud.bigquery.TableInfo;
import com.google.gson.Gson;
import io.cdap.cdap.api.data.schema.Schema;
import io.cdap.cdap.api.metrics.Metrics;
import io.cdap.cdap.etl.api.engine.sql.SQLEngineException;
import io.cdap.cdap.etl.api.join.JoinCondition;
import io.cdap.cdap.etl.api.join.JoinDefinition;
import io.cdap.cdap.etl.api.join.JoinStage;
import io.cdap.plugin.gcp.bigquery.sqlengine.BigQueryJobType;
import io.cdap.plugin.gcp.bigquery.sqlengine.BigQuerySQLEngineConfig;
import io.cdap.plugin.gcp.bigquery.sqlengine.util.BigQuerySchemaValidation;
import io.cdap.plugin.gcp.bigquery.util.BigQueryUtil;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BigQuerySQLEngineUtils {
    private static final Logger LOG = LoggerFactory.getLogger(BigQuerySQLEngineUtils.class);
    private static final Gson GSON = new Gson();
    public static final String GCS_PATH_FORMAT = "gs://%s/%s/%s";
    public static final String BQ_TABLE_NAME_FORMAT = "%s_%s";
    public static final String METRIC_BYTES_PROCESSED = "bytes.processed";
    public static final String METRIC_BYTES_BILLED = "bytes.billed";
    public static final String METRIC_SLOT_MS = "slot.ms";

    private BigQuerySQLEngineUtils() {
    }

    public static String getGCSPath(String bucket, String runId, String tableId) {
        return String.format(GCS_PATH_FORMAT, bucket, runId, tableId);
    }

    public static String newIdentifier() {
        return UUID.randomUUID().toString().replace("-", "");
    }

    public static String getNewTableName(String runId) {
        return String.format(BQ_TABLE_NAME_FORMAT, runId, BigQuerySQLEngineUtils.newIdentifier());
    }

    public static Long getNumRows(BigQuery bigQuery, DatasetId dataset, String table) {
        LOG.debug("Getting number of records stored in table {}", (Object)table);
        TableId tableId = TableId.of((String)dataset.getProject(), (String)dataset.getDataset(), (String)table);
        Table bgTable = bigQuery.getTable(tableId, new BigQuery.TableOption[0]);
        if (bgTable == null) {
            throw new SQLEngineException(String.format("Table '%s' could not be found on dataset '%s' and project `%s`", table, dataset.getDataset(), dataset.getProject()));
        }
        long numRows = bgTable.getNumRows().longValue();
        LOG.debug("Table {} contains {} records", (Object)table, (Object)numRows);
        return numRows;
    }

    public static void createEmptyTable(BigQuerySQLEngineConfig config, BigQuery bigQuery, String project, String dataset, String table) {
        LOG.debug("Creating empty table {} in dataset {} and project {}", new Object[]{table, dataset, project});
        TableId tableId = TableId.of((String)project, (String)dataset, (String)table);
        StandardTableDefinition tableDefinition = StandardTableDefinition.of((com.google.cloud.bigquery.Schema)com.google.cloud.bigquery.Schema.of((Field[])new Field[0]));
        TableInfo.Builder tableInfoBuilder = TableInfo.newBuilder((TableId)tableId, (TableDefinition)tableDefinition);
        if (!config.shouldRetainTables().booleanValue() && config.getTempTableTTLHours() > 0) {
            long ttlMillis = TimeUnit.MILLISECONDS.convert(config.getTempTableTTLHours().intValue(), TimeUnit.HOURS);
            long expirationTime = Instant.now().toEpochMilli() + ttlMillis;
            tableInfoBuilder.setExpirationTime(Long.valueOf(expirationTime));
        }
        bigQuery.create(tableInfoBuilder.build(), new BigQuery.TableOption[0]);
        LOG.debug("Created empty table {} in dataset {} and project {}", new Object[]{table, dataset, project});
    }

    public static void createEmptyTableWithSourceConfig(BigQuery bigQuery, String project, String dataset, String table, Table sourceTable, Long tableTTL) {
        LOG.debug("Creating empty table {} in dataset {} and project {} with configurations similar to {}", new Object[]{table, dataset, project, sourceTable.getTableId()});
        StandardTableDefinition tableDefinitionSource = (StandardTableDefinition)sourceTable.getDefinition();
        TableId tableId = TableId.of((String)project, (String)dataset, (String)table);
        StandardTableDefinition tableDefinition = StandardTableDefinition.newBuilder().setSchema(tableDefinitionSource.getSchema()).setTimePartitioning(tableDefinitionSource.getTimePartitioning()).setClustering(tableDefinitionSource.getClustering()).build();
        TableInfo.Builder tableInfoBuilder = TableInfo.newBuilder((TableId)tableId, (TableDefinition)tableDefinition);
        if (tableTTL > 0L) {
            tableInfoBuilder.setExpirationTime(tableTTL);
        }
        bigQuery.create(tableInfoBuilder.build(), new BigQuery.TableOption[0]);
        LOG.debug("Created empty table {} in dataset {} and project {} with configurations similar to {}", new Object[]{table, dataset, project, sourceTable.getTableId()});
    }

    public static void updateTableExpiration(BigQuery bigQuery, TableId tableId, @Nullable Long tableTTL) {
        if (tableTTL == null || tableTTL <= 0L) {
            return;
        }
        Table table = bigQuery.getTable(tableId, new BigQuery.TableOption[0]);
        bigQuery.update((TableInfo)table.toBuilder().setExpirationTime(tableTTL).build(), new BigQuery.TableOption[0]);
        LOG.debug("Updated {}'s Expiration time to {}", (Object)tableId, (Object)tableTTL);
    }

    public static void validateInputStage(JoinStage inputStage, List<String> validationProblems) {
        String stageName = inputStage.getStageName();
        if (inputStage.getSchema() == null) {
            validationProblems.add(String.format("Input schema from stage '%s' is null", stageName));
        } else {
            BigQuerySchemaValidation bigQuerySchemaValidation = BigQuerySchemaValidation.validateSchema(inputStage.getSchema());
            if (!bigQuerySchemaValidation.isSupported()) {
                validationProblems.add(String.format("Input schema from stage '%s' contains unsupported field types for the following fields: %s", stageName, String.join((CharSequence)", ", bigQuerySchemaValidation.getInvalidFields())));
            }
        }
        if (!BigQuerySQLEngineUtils.isValidIdentifier(stageName)) {
            validationProblems.add(String.format("Unsupported stage name '%s'. Stage names cannot contain backtick ` or backslash \\ ", stageName));
        }
    }

    public static void validateOutputSchema(@Nullable Schema outputSchema, List<String> validationProblems) {
        if (outputSchema == null) {
            validationProblems.add("Output Schema is null");
        } else {
            BigQuerySchemaValidation bigQuerySchemaValidation = BigQuerySchemaValidation.validateSchema(outputSchema);
            if (!bigQuerySchemaValidation.isSupported()) {
                validationProblems.add(String.format("Output schema contains unsupported field types for the following fields: %s", String.join((CharSequence)", ", bigQuerySchemaValidation.getInvalidFields())));
            }
        }
    }

    public static void validateOnExpressionJoinCondition(JoinCondition.OnExpression onExpression, List<String> validationProblems) {
        for (Map.Entry alias : onExpression.getDatasetAliases().entrySet()) {
            if (BigQuerySQLEngineUtils.isValidIdentifier((String)alias.getValue())) continue;
            validationProblems.add(String.format("Unsupported alias '%s' for stage '%s'", alias.getValue(), alias.getKey()));
        }
    }

    public static void validateJoinOnKeyStages(JoinDefinition joinDefinition, List<String> validationProblems) {
        if (joinDefinition.getStages().size() < 3) {
            return;
        }
        boolean isInnerJoin = true;
        for (JoinStage stage : joinDefinition.getStages()) {
            isInnerJoin &= stage.isRequired();
        }
        if (!isInnerJoin) {
            validationProblems.add(String.format("Only 2 input stages are supported for outer joins, %d stages supplied.", joinDefinition.getStages().size()));
        }
    }

    public static boolean isSupportedSchema(Schema schema) {
        return BigQuerySchemaValidation.validateSchema(schema).isSupported();
    }

    public static boolean isValidIdentifier(String identifier) {
        return identifier != null && !identifier.contains("\\") && !identifier.contains("`");
    }

    public static Map<String, String> getJobTags(BigQueryJobType operation) {
        return BigQuerySQLEngineUtils.getJobTags(operation.getType());
    }

    public static Map<String, String> getJobTags(String operation) {
        Map<String, String> labels = BigQueryUtil.getJobTags("bq_pushdown");
        labels.put("pushdown_operation", operation);
        return labels;
    }

    public static void logJobMetrics(Job job, Metrics metrics) {
        if (job.getStatistics() == null) {
            LOG.warn("No statistics were found for BigQuery job {}", (Object)job.getJobId());
        }
        String startTimeStr = BigQuerySQLEngineUtils.getISODateTimeString(job.getStatistics().getStartTime());
        String endTimeStr = BigQuerySQLEngineUtils.getISODateTimeString(job.getStatistics().getEndTime());
        String executionTimeStr = BigQuerySQLEngineUtils.getExecutionTimeString(job.getStatistics().getStartTime(), job.getStatistics().getEndTime());
        if (job.getStatistics() instanceof JobStatistics.QueryStatistics) {
            JobStatistics.QueryStatistics queryStatistics = (JobStatistics.QueryStatistics)job.getStatistics();
            LOG.info("Metrics for job {}:\n Start: {} ,\n End: {} ,\n Execution time: {} ,\n Processed Bytes: {} ,\n Billed Bytes: {} ,\n Total Slot ms: {} ,\n Records per stage (read/write): {}", new Object[]{job.getJobId().getJob(), startTimeStr, endTimeStr, executionTimeStr, queryStatistics.getTotalBytesProcessed(), queryStatistics.getTotalBytesBilled(), queryStatistics.getTotalSlotMs(), BigQuerySQLEngineUtils.getQueryStageRecordCounts(queryStatistics.getQueryPlan())});
            if (LOG.isTraceEnabled()) {
                LOG.trace("Additional Metrics for job {}:\n Query Plan: {} ,\n Query Timeline: {} \n", new Object[]{job.getJobId().getJob(), GSON.toJson((Object)queryStatistics.getQueryPlan()), GSON.toJson((Object)queryStatistics.getTimeline())});
            }
            if (queryStatistics.getTotalBytesProcessed() != null) {
                metrics.countLong(METRIC_BYTES_PROCESSED, queryStatistics.getTotalBytesProcessed().longValue());
            }
            if (queryStatistics.getTotalBytesBilled() != null) {
                metrics.countLong(METRIC_BYTES_BILLED, queryStatistics.getTotalBytesBilled().longValue());
            }
            if (queryStatistics.getTotalSlotMs() != null) {
                metrics.countLong(METRIC_SLOT_MS, queryStatistics.getTotalSlotMs().longValue());
            }
            return;
        }
        LOG.info("Metrics for job: {}\n Start: {} ,\n End: {} ,\n Execution time: {}", new Object[]{job.getJobId().getJob(), startTimeStr, endTimeStr, executionTimeStr});
    }

    private static String getISODateTimeString(Long epoch) {
        if (epoch == null) {
            return "N/A";
        }
        return Instant.ofEpochMilli(epoch).toString();
    }

    private static String getExecutionTimeString(Long startEpoch, Long endEpoch) {
        if (startEpoch == null || endEpoch == null) {
            return "N/A";
        }
        return endEpoch - startEpoch + " ms";
    }

    private static String getQueryStageRecordCounts(List<QueryStage> queryPlan) {
        if (queryPlan == null || queryPlan.isEmpty()) {
            return "N/A";
        }
        return queryPlan.stream().map(qs -> BigQuerySQLEngineUtils.formatRecordCount(qs.getRecordsRead()) + "/" + BigQuerySQLEngineUtils.formatRecordCount(qs.getRecordsWritten())).collect(Collectors.joining(" , ", "[ ", " ]"));
    }

    private static String formatRecordCount(Long val) {
        if (val == null) {
            return "N/A";
        }
        return val.toString();
    }
}

