/*
 * Decompiled with CFR 0.152.
 */
package io.cdap.plugin.gcp.bigquery.util;

import com.google.auth.oauth2.GoogleCredentials;
import com.google.cloud.bigquery.BigQuery;
import com.google.cloud.bigquery.BigQueryException;
import com.google.cloud.bigquery.Dataset;
import com.google.cloud.bigquery.Field;
import com.google.cloud.bigquery.FieldList;
import com.google.cloud.bigquery.LegacySQLTypeName;
import com.google.cloud.bigquery.Schema;
import com.google.cloud.bigquery.StandardSQLTypeName;
import com.google.cloud.bigquery.StandardTableDefinition;
import com.google.cloud.bigquery.Table;
import com.google.cloud.bigquery.TableId;
import com.google.cloud.bigquery.TimePartitioning;
import com.google.cloud.kms.v1.CryptoKeyName;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import io.cdap.cdap.api.data.schema.Schema;
import io.cdap.cdap.etl.api.FailureCollector;
import io.cdap.cdap.etl.api.validation.InvalidConfigPropertyException;
import io.cdap.cdap.etl.api.validation.InvalidStageException;
import io.cdap.cdap.etl.api.validation.ValidationFailure;
import io.cdap.plugin.gcp.common.GCPUtils;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import java.util.zip.CRC32;
import javax.annotation.Nullable;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.security.Credentials;
import org.apache.hadoop.security.UserGroupInformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class BigQueryUtil {
    private static final Logger LOG = LoggerFactory.getLogger(BigQueryUtil.class);
    private static final String DEFAULT_PARTITION_COLUMN_NAME = "_PARTITIONTIME";
    private static final String BIGQUERY_BUCKET_PREFIX_PROPERTY_NAME = "gcp.bigquery.bucket.prefix";
    public static final String BUCKET_PATTERN = "[a-z0-9._-]+";
    public static final String DATASET_PATTERN = "[A-Za-z0-9_]+";
    public static final String TABLE_PATTERN = "[A-Za-z0-9_-]+";
    public static final String BQ_JOB_TYPE_SOURCE_TAG = "bq_source_plugin";
    public static final String BQ_JOB_TYPE_EXECUTE_TAG = "bq_execute_plugin";
    public static final String BQ_JOB_TYPE_SINK_TAG = "bq_sink_plugin";
    public static final String BQ_JOB_TYPE_PUSHDOWN_TAG = "bq_pushdown";
    public static final Set<Schema.Type> UNSUPPORTED_ARRAY_TYPES = ImmutableSet.of((Object)Schema.Type.ARRAY, (Object)Schema.Type.MAP);
    public static final Map<LegacySQLTypeName, String> BQ_TYPE_MAP = ImmutableMap.builder().put((Object)LegacySQLTypeName.INTEGER, (Object)"long").put((Object)LegacySQLTypeName.FLOAT, (Object)"double").put((Object)LegacySQLTypeName.BOOLEAN, (Object)"boolean").put((Object)LegacySQLTypeName.BYTES, (Object)"bytes").put((Object)LegacySQLTypeName.RECORD, (Object)"record").put((Object)LegacySQLTypeName.STRING, (Object)"string or datetime").put((Object)LegacySQLTypeName.DATETIME, (Object)"datetime or string").put((Object)LegacySQLTypeName.DATE, (Object)"date").put((Object)LegacySQLTypeName.TIME, (Object)"time").put((Object)LegacySQLTypeName.TIMESTAMP, (Object)"timestamp").put((Object)LegacySQLTypeName.NUMERIC, (Object)"decimal").build();
    private static final Map<Schema.Type, Set<LegacySQLTypeName>> TYPE_MAP = ImmutableMap.builder().put((Object)Schema.Type.INT, (Object)ImmutableSet.of((Object)LegacySQLTypeName.INTEGER)).put((Object)Schema.Type.LONG, (Object)ImmutableSet.of((Object)LegacySQLTypeName.INTEGER)).put((Object)Schema.Type.STRING, (Object)ImmutableSet.of((Object)LegacySQLTypeName.STRING, (Object)LegacySQLTypeName.DATETIME)).put((Object)Schema.Type.FLOAT, (Object)ImmutableSet.of((Object)LegacySQLTypeName.FLOAT)).put((Object)Schema.Type.DOUBLE, (Object)ImmutableSet.of((Object)LegacySQLTypeName.FLOAT)).put((Object)Schema.Type.BOOLEAN, (Object)ImmutableSet.of((Object)LegacySQLTypeName.BOOLEAN)).put((Object)Schema.Type.BYTES, (Object)ImmutableSet.of((Object)LegacySQLTypeName.BYTES)).put((Object)Schema.Type.RECORD, (Object)ImmutableSet.of((Object)LegacySQLTypeName.RECORD)).build();
    private static final Map<Schema.LogicalType, Set<LegacySQLTypeName>> LOGICAL_TYPE_MAP = ImmutableMap.builder().put((Object)Schema.LogicalType.DATE, (Object)ImmutableSet.of((Object)LegacySQLTypeName.DATE)).put((Object)Schema.LogicalType.DATETIME, (Object)ImmutableSet.of((Object)LegacySQLTypeName.DATETIME, (Object)LegacySQLTypeName.STRING)).put((Object)Schema.LogicalType.TIME_MILLIS, (Object)ImmutableSet.of((Object)LegacySQLTypeName.TIME)).put((Object)Schema.LogicalType.TIME_MICROS, (Object)ImmutableSet.of((Object)LegacySQLTypeName.TIME)).put((Object)Schema.LogicalType.TIMESTAMP_MILLIS, (Object)ImmutableSet.of((Object)LegacySQLTypeName.TIMESTAMP)).put((Object)Schema.LogicalType.TIMESTAMP_MICROS, (Object)ImmutableSet.of((Object)LegacySQLTypeName.TIMESTAMP)).put((Object)Schema.LogicalType.DECIMAL, (Object)ImmutableSet.of((Object)LegacySQLTypeName.NUMERIC, (Object)LegacySQLTypeName.BIGNUMERIC)).build();

    public static io.cdap.cdap.api.data.schema.Schema getNonNullableSchema(io.cdap.cdap.api.data.schema.Schema schema) {
        return schema.isNullable() ? schema.getNonNullable() : schema;
    }

    public static Configuration getBigQueryConfig(@Nullable String serviceAccountInfo, String projectId, @Nullable CryptoKeyName cmekKeyName, String serviceAccountType) throws IOException {
        Job job = Job.getInstance();
        if (UserGroupInformation.isSecurityEnabled()) {
            Credentials credentials = UserGroupInformation.getCurrentUser().getCredentials();
            job.getCredentials().addAll(credentials);
        }
        Configuration configuration = job.getConfiguration();
        configuration.clear();
        Map<String, String> authProperties = GCPUtils.generateBigQueryAuthProperties(serviceAccountInfo, serviceAccountType);
        authProperties.forEach((arg_0, arg_1) -> ((Configuration)configuration).set(arg_0, arg_1));
        configuration.set("fs.gs.impl", "com.google.cloud.hadoop.fs.gcs.GoogleHadoopFileSystem");
        configuration.set("fs.AbstractFileSystem.gs.impl", "com.google.cloud.hadoop.fs.gcs.GoogleHadoopFS");
        configuration.set("fs.gs.project.id", projectId);
        configuration.set("fs.gs.working.dir", "/");
        configuration.set("mapred.bq.project.id", projectId);
        if (cmekKeyName != null) {
            configuration.set("mapred.bq.output.table.kmskeyname", cmekKeyName.toString());
        }
        return configuration;
    }

    public static io.cdap.cdap.api.data.schema.Schema getTableSchema(Schema bqSchema, @Nullable FailureCollector collector) {
        FieldList fields = bqSchema.getFields();
        ArrayList<Schema.Field> schemafields = new ArrayList<Schema.Field>();
        for (Field field : fields) {
            Schema.Field schemaField = BigQueryUtil.getSchemaField(field, collector);
            if (schemaField == null) continue;
            schemafields.add(schemaField);
        }
        if (schemafields.isEmpty() && collector != null && !collector.getValidationFailures().isEmpty()) {
            collector.getOrThrowException();
        }
        if (schemafields.isEmpty()) {
            return null;
        }
        return io.cdap.cdap.api.data.schema.Schema.recordOf((String)"output", schemafields);
    }

    @Nullable
    public static Schema.Field getSchemaField(Field field, @Nullable FailureCollector collector) {
        return BigQueryUtil.getSchemaField(field, collector, null);
    }

    @Nullable
    private static Schema.Field getSchemaField(Field field, @Nullable FailureCollector collector, @Nullable String recordPrefix) {
        io.cdap.cdap.api.data.schema.Schema schema = BigQueryUtil.convertFieldType(field, collector, recordPrefix);
        if (schema == null) {
            return null;
        }
        Field.Mode mode = field.getMode() == null ? Field.Mode.NULLABLE : field.getMode();
        switch (mode) {
            case NULLABLE: {
                return Schema.Field.of((String)field.getName(), (io.cdap.cdap.api.data.schema.Schema)io.cdap.cdap.api.data.schema.Schema.nullableOf((io.cdap.cdap.api.data.schema.Schema)schema));
            }
            case REQUIRED: {
                return Schema.Field.of((String)field.getName(), (io.cdap.cdap.api.data.schema.Schema)schema);
            }
            case REPEATED: {
                return Schema.Field.of((String)field.getName(), (io.cdap.cdap.api.data.schema.Schema)io.cdap.cdap.api.data.schema.Schema.arrayOf((io.cdap.cdap.api.data.schema.Schema)schema));
            }
        }
        String error = String.format("Field '%s' has unsupported mode '%s'.", field.getName(), mode);
        if (collector == null) {
            throw new RuntimeException(error);
        }
        collector.addFailure(error, null);
        return null;
    }

    @Nullable
    public static io.cdap.cdap.api.data.schema.Schema convertFieldType(Field field, @Nullable FailureCollector collector) {
        return BigQueryUtil.convertFieldType(field, collector, null);
    }

    @Nullable
    public static io.cdap.cdap.api.data.schema.Schema convertFieldType(Field field, @Nullable FailureCollector collector, @Nullable String recordPrefix) {
        LegacySQLTypeName type = field.getType();
        StandardSQLTypeName standardType = type.getStandardType();
        switch (standardType) {
            case FLOAT64: {
                return io.cdap.cdap.api.data.schema.Schema.of((Schema.Type)Schema.Type.DOUBLE);
            }
            case BOOL: {
                return io.cdap.cdap.api.data.schema.Schema.of((Schema.Type)Schema.Type.BOOLEAN);
            }
            case INT64: {
                return io.cdap.cdap.api.data.schema.Schema.of((Schema.Type)Schema.Type.LONG);
            }
            case STRING: {
                return io.cdap.cdap.api.data.schema.Schema.of((Schema.Type)Schema.Type.STRING);
            }
            case DATETIME: {
                return io.cdap.cdap.api.data.schema.Schema.of((Schema.LogicalType)Schema.LogicalType.DATETIME);
            }
            case BYTES: {
                return io.cdap.cdap.api.data.schema.Schema.of((Schema.Type)Schema.Type.BYTES);
            }
            case TIME: {
                return io.cdap.cdap.api.data.schema.Schema.of((Schema.LogicalType)Schema.LogicalType.TIME_MICROS);
            }
            case DATE: {
                return io.cdap.cdap.api.data.schema.Schema.of((Schema.LogicalType)Schema.LogicalType.DATE);
            }
            case TIMESTAMP: {
                return io.cdap.cdap.api.data.schema.Schema.of((Schema.LogicalType)Schema.LogicalType.TIMESTAMP_MICROS);
            }
            case NUMERIC: {
                return io.cdap.cdap.api.data.schema.Schema.decimalOf((int)38, (int)9);
            }
            case BIGNUMERIC: {
                return io.cdap.cdap.api.data.schema.Schema.decimalOf((int)77, (int)38);
            }
            case STRUCT: {
                FieldList fields = field.getSubFields();
                ArrayList<Schema.Field> schemaFields = new ArrayList<Schema.Field>();
                String recordTypeName = "";
                if (recordPrefix != null) {
                    recordTypeName = recordPrefix + '.';
                }
                recordTypeName = recordTypeName + field.getName();
                for (Field f : fields) {
                    Schema.Field schemaField = BigQueryUtil.getSchemaField(f, collector, recordTypeName);
                    if (schemaField == null) continue;
                    schemaFields.add(schemaField);
                }
                if (!schemaFields.isEmpty()) {
                    io.cdap.cdap.api.data.schema.Schema namingSchema = io.cdap.cdap.api.data.schema.Schema.recordOf(schemaFields);
                    recordTypeName = recordTypeName + namingSchema.getRecordName();
                    return io.cdap.cdap.api.data.schema.Schema.recordOf((String)recordTypeName, schemaFields);
                }
                return null;
            }
        }
        String error = String.format("BigQuery column '%s' is of unsupported type '%s'.", field.getName(), standardType.name());
        String action = String.format("Supported column types are: %s.", BQ_TYPE_MAP.keySet().stream().map(t -> t.getStandardType().name()).collect(Collectors.joining(", ")));
        if (collector == null) {
            throw new RuntimeException(error + action);
        }
        collector.addFailure(error, action);
        return null;
    }

    @Nullable
    public static ValidationFailure validateFieldSchemaMatches(Field bqField, Schema.Field field, String dataset, String table, Set<Schema.Type> supportedTypes, FailureCollector collector) {
        String name = field.getName();
        io.cdap.cdap.api.data.schema.Schema fieldSchema = BigQueryUtil.getNonNullableSchema(field.getSchema());
        Schema.Type type = fieldSchema.getType();
        Schema.LogicalType logicalType = fieldSchema.getLogicalType();
        if (logicalType != null) {
            if (LOGICAL_TYPE_MAP.get(logicalType) == null) {
                return collector.addFailure(String.format("Field '%s' is of unsupported type '%s'.", field.getName(), fieldSchema.getDisplayName()), String.format("Supported types are: %s, date, time, timestamp and decimal.", supportedTypes.stream().map(t -> t.name().toLowerCase()).collect(Collectors.joining(", "))));
            }
            if (!LOGICAL_TYPE_MAP.get(logicalType).contains(bqField.getType())) {
                return collector.addFailure(String.format("Field '%s' of type '%s' has incompatible type with column '%s' in BigQuery table '%s.%s'.", name, fieldSchema.getDisplayName(), bqField.getName(), dataset, table), String.format("Modify the input so that it is of type '%s'.", BQ_TYPE_MAP.get(bqField.getType())));
            }
            if (logicalType == Schema.LogicalType.DECIMAL && (fieldSchema.getPrecision() > 77 || fieldSchema.getScale() > 38)) {
                return collector.addFailure(String.format("Decimal Field '%s' has invalid precision '%s' and scale '%s'. ", name, fieldSchema.getPrecision(), fieldSchema.getScale()), String.format("Precision must be at most '%s' and scale must be at most '%s'.", 77, 38));
            }
            return null;
        }
        if (!supportedTypes.contains(type)) {
            return collector.addFailure(String.format("Field '%s' is of unsupported type '%s'.", name, type.name().toLowerCase()), String.format("Supported types are: %s, date, time, timestamp and decimal.", supportedTypes.stream().map(t -> t.name().toLowerCase()).collect(Collectors.joining(", "))));
        }
        if (type == Schema.Type.ARRAY) {
            ValidationFailure failure = BigQueryUtil.validateArraySchema(field.getSchema(), field.getName(), collector);
            if (failure != null) {
                return failure;
            }
            if (bqField.getMode() == Field.Mode.REPEATED) {
                fieldSchema = fieldSchema.getComponentSchema();
                type = fieldSchema.getType();
            }
        }
        if (TYPE_MAP.get(type) != null && !TYPE_MAP.get(type).contains(bqField.getType())) {
            return collector.addFailure(String.format("Field '%s' of type '%s' is incompatible with column '%s' of type '%s' in BigQuery table '%s.%s'.", field.getName(), fieldSchema.getDisplayName(), bqField.getName(), BQ_TYPE_MAP.get(bqField.getType()), dataset, table), String.format("It must be of type '%s'.", BQ_TYPE_MAP.get(bqField.getType())));
        }
        return null;
    }

    public static void validateFieldModeMatches(Field bigQueryField, Schema.Field field, boolean allowSchemaRelaxation, FailureCollector collector) {
        Field.Mode mode = bigQueryField.getMode();
        boolean isBqFieldNullable = mode == null || mode.equals((Object)Field.Mode.NULLABLE);
        io.cdap.cdap.api.data.schema.Schema fieldSchema = field.getSchema();
        if (!allowSchemaRelaxation && fieldSchema.isNullable() && !isBqFieldNullable && !BigQueryUtil.getNonNullableSchema(fieldSchema).getType().equals((Object)Schema.Type.ARRAY)) {
            collector.addFailure(String.format("Field '%s' cannot be nullable.", bigQueryField.getName()), "Change the field to be required.").withOutputSchemaField(field.getName());
        }
    }

    public static List<String> getSchemaMinusBqFields(List<Schema.Field> schemaFields, FieldList bqFields) {
        ArrayList<String> diff = new ArrayList<String>();
        for (Schema.Field field : schemaFields) {
            diff.add(field.getName());
        }
        for (Schema.Field field : bqFields) {
            diff.remove(field.getName());
        }
        return diff;
    }

    public static List<String> getBqFieldsMinusSchema(FieldList bqFields, List<Schema.Field> schemaFields) {
        ArrayList<String> diff = new ArrayList<String>();
        for (Field field : bqFields) {
            diff.add(field.getName());
        }
        for (Schema.Field field : schemaFields) {
            diff.remove(field.getName());
        }
        return diff;
    }

    public static Map<String, String> configToMap(Configuration config) {
        return StreamSupport.stream(config.spliterator(), false).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    @Nullable
    public static ValidationFailure validateArraySchema(io.cdap.cdap.api.data.schema.Schema arraySchema, String name, FailureCollector collector) {
        io.cdap.cdap.api.data.schema.Schema nonNullableSchema = BigQueryUtil.getNonNullableSchema(arraySchema);
        io.cdap.cdap.api.data.schema.Schema componentSchema = nonNullableSchema.getComponentSchema();
        if (componentSchema.isNullable()) {
            return collector.addFailure(String.format("Field '%s' contains null values in its array.", name), "Change the array component type to be non-nullable.");
        }
        if (UNSUPPORTED_ARRAY_TYPES.contains(componentSchema.getType())) {
            return collector.addFailure(String.format("Field '%s' is an array of unsupported type '%s'.", name, componentSchema.getDisplayName()), "Change the array component type to be a valid type.");
        }
        return null;
    }

    @Nullable
    public static Table getBigQueryTable(String datasetProject, String datasetId, String tableName, @Nullable String serviceAccount, boolean isServiceAccountFilePath) {
        Table table;
        TableId tableId = TableId.of((String)datasetProject, (String)datasetId, (String)tableName);
        GoogleCredentials credentials = null;
        if (serviceAccount != null) {
            try {
                credentials = GCPUtils.loadServiceAccountCredentials(serviceAccount, isServiceAccountFilePath);
            }
            catch (IOException e) {
                throw new InvalidConfigPropertyException(String.format("Unable to load credentials from %s", isServiceAccountFilePath ? serviceAccount : " JSON."), "serviceFilePath");
            }
        }
        BigQuery bigQuery = GCPUtils.getBigQuery(datasetProject, credentials);
        try {
            table = bigQuery.getTable(tableId, new BigQuery.TableOption[0]);
        }
        catch (BigQueryException e) {
            throw new InvalidStageException("Unable to get details about the BigQuery table: " + e.getMessage(), (Throwable)e);
        }
        return table;
    }

    @Nullable
    public static Table getBigQueryTable(String projectId, String datasetId, String tableName, @Nullable String serviceAccountPath, FailureCollector collector) {
        return BigQueryUtil.getBigQueryTable(projectId, datasetId, tableName, serviceAccountPath, true, collector);
    }

    public static Table getBigQueryTable(String projectId, String dataset, String tableName, @Nullable String serviceAccount, @Nullable Boolean isServiceAccountFilePath, FailureCollector collector) {
        TableId tableId = TableId.of((String)projectId, (String)dataset, (String)tableName);
        GoogleCredentials credentials = null;
        if (serviceAccount != null) {
            try {
                credentials = GCPUtils.loadServiceAccountCredentials(serviceAccount, isServiceAccountFilePath);
            }
            catch (IOException e) {
                collector.addFailure(String.format("Unable to load credentials from %s.", isServiceAccountFilePath != false ? serviceAccount : "provided JSON key"), "Ensure the service account file is available on the local filesystem.").withConfigProperty("serviceFilePath");
                throw collector.getOrThrowException();
            }
        }
        BigQuery bigQuery = GCPUtils.getBigQuery(projectId, credentials);
        Table table = null;
        try {
            table = bigQuery.getTable(tableId, new BigQuery.TableOption[0]);
        }
        catch (BigQueryException e) {
            collector.addFailure("Unable to get details about the BigQuery table: " + e.getMessage(), null).withConfigProperty("table");
            throw collector.getOrThrowException();
        }
        return table;
    }

    public static void validateBucket(String bucket, String bucketPropertyName, FailureCollector collector) {
        String errorMessage = "Bucket name can only contain lowercase letters, numbers, '.', '_', and '-'.";
        BigQueryUtil.match(bucket, bucketPropertyName, BUCKET_PATTERN, collector, errorMessage);
    }

    public static void validateDataset(String dataset, String datasetPropertyName, FailureCollector collector) {
        String errorMessage = "Dataset name can only contain letters (lower or uppercase), numbers and '_'.";
        BigQueryUtil.match(dataset, datasetPropertyName, DATASET_PATTERN, collector, errorMessage);
    }

    public static void validateTable(String table, String tablePropertyName, FailureCollector collector) {
        String errorMessage = "Table name can only contain letters (lower or uppercase), numbers, '_' and '-'.";
        BigQueryUtil.match(table, tablePropertyName, TABLE_PATTERN, collector, errorMessage);
    }

    public static void validateGCSChunkSize(String chunkSize, String chunkSizePropertyName, FailureCollector collector) {
        if (!Strings.isNullOrEmpty((String)chunkSize)) {
            try {
                if (Integer.parseInt(chunkSize) % 262144 != 0) {
                    collector.addFailure(String.format("Value must be a multiple of %s.", 262144), null).withConfigProperty(chunkSizePropertyName);
                }
            }
            catch (NumberFormatException e) {
                collector.addFailure(e.getMessage(), "Input value must be a valid number.").withConfigProperty(chunkSizePropertyName);
            }
        }
    }

    private static void match(String text, String propertyName, String pattern, FailureCollector collector, String errorMessage) {
        Pattern p;
        if (!Strings.isNullOrEmpty((String)text) && !(p = Pattern.compile(pattern)).matcher(text).matches()) {
            collector.addFailure(errorMessage, null).withConfigProperty(propertyName);
        }
    }

    public static void deleteTemporaryDirectory(Configuration configuration, String dir) throws IOException {
        Path path = new Path(dir);
        FileSystem fs = path.getFileSystem(configuration);
        if (fs.exists(path)) {
            fs.delete(path, true);
            LOG.debug("Deleted temporary directory '{}'", (Object)path);
        }
    }

    public static String generateTimePartitionCondition(StandardTableDefinition tableDefinition, String partitionFromDate, String partitionToDate) {
        TimePartitioning timePartitioning = tableDefinition.getTimePartitioning();
        if (timePartitioning == null) {
            return "";
        }
        StringBuilder timePartitionCondition = new StringBuilder();
        String columnName = timePartitioning.getField() != null ? timePartitioning.getField() : DEFAULT_PARTITION_COLUMN_NAME;
        LegacySQLTypeName columnType = null;
        if (!DEFAULT_PARTITION_COLUMN_NAME.equals(columnName)) {
            columnType = tableDefinition.getSchema().getFields().get(columnName).getType();
        }
        String columnNameTS = columnName;
        if (!LegacySQLTypeName.TIMESTAMP.equals(columnType)) {
            columnNameTS = "TIMESTAMP(`" + columnNameTS + "`)";
        }
        if (partitionFromDate != null) {
            timePartitionCondition.append(columnNameTS).append(" >= ").append("TIMESTAMP(\"").append(partitionFromDate).append("\")");
        }
        if (partitionFromDate != null && partitionToDate != null) {
            timePartitionCondition.append(" and ");
        }
        if (partitionToDate != null) {
            timePartitionCondition.append(columnNameTS).append(" < ").append("TIMESTAMP(\"").append(partitionToDate).append("\")");
        }
        return timePartitionCondition.toString();
    }

    public static String getFQN(String datasetProject, String datasetName, String tableName) {
        return String.format("%s:%s.%s.%s", "bigquery", datasetProject, datasetName, tableName);
    }

    @Nullable
    public static String getBucketPrefix(Map<String, String> arguments) {
        if (arguments.containsKey(BIGQUERY_BUCKET_PREFIX_PROPERTY_NAME)) {
            String bucketPrefix = arguments.get(BIGQUERY_BUCKET_PREFIX_PROPERTY_NAME);
            BigQueryUtil.validateBucketPrefix(bucketPrefix);
            LOG.debug("Using bucket prefix for temporary buckets: {}", (Object)bucketPrefix);
            return bucketPrefix;
        }
        return null;
    }

    private static void validateBucketPrefix(String bucketPrefix) {
        if (!bucketPrefix.matches("^[a-z0-9-_.]+$")) {
            throw new IllegalArgumentException("The configured bucket prefix '" + bucketPrefix + "' is not a valid bucket name. Bucket names can only contain lowercase letters, numeric characters, dashes (-), underscores (_), and dots (.).");
        }
        if (!bucketPrefix.contains(".") && bucketPrefix.length() > 50) {
            throw new IllegalArgumentException("The configured bucket prefix '" + bucketPrefix + "' should be 50 characters or shorter.");
        }
    }

    @VisibleForTesting
    public static String crc32location(String location) {
        byte[] bytes = location.toLowerCase().getBytes();
        CRC32 checksum = new CRC32();
        checksum.update(bytes, 0, bytes.length);
        return Long.toHexString(checksum.getValue());
    }

    public static String getBucketNameForLocation(String bucketPrefix, String location) {
        return String.format("%s-%s", bucketPrefix, BigQueryUtil.crc32location(location));
    }

    public static Map<String, String> getJobTags(String jobType) {
        HashMap<String, String> labels = new HashMap<String, String>();
        labels.put("job_source", "cdap");
        labels.put("type", jobType);
        return labels;
    }

    @Nullable
    public static String getStagingBucketName(Map<String, String> arguments, @Nullable String configLocation, @Nullable Dataset dataset, @Nullable String bucket) {
        String bucketPrefix = BigQueryUtil.getBucketPrefix(arguments);
        if (Strings.isNullOrEmpty((String)bucket) && bucketPrefix != null) {
            String datasetLocation = dataset != null ? dataset.getLocation() : configLocation;
            bucket = BigQueryUtil.getBucketNameForLocation(bucketPrefix, datasetLocation);
        }
        return bucket;
    }
}

