/*
 * Decompiled with CFR 0.152.
 */
package io.cdap.plugin.gcp.datastore.sink;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.google.datastore.v1.Key;
import io.cdap.cdap.api.annotation.Description;
import io.cdap.cdap.api.annotation.Macro;
import io.cdap.cdap.api.annotation.Name;
import io.cdap.cdap.api.data.schema.Schema;
import io.cdap.cdap.etl.api.FailureCollector;
import io.cdap.plugin.gcp.common.GCPReferenceSinkConfig;
import io.cdap.plugin.gcp.datastore.exception.DatastoreInitializationException;
import io.cdap.plugin.gcp.datastore.sink.util.IndexStrategy;
import io.cdap.plugin.gcp.datastore.sink.util.SinkKeyType;
import io.cdap.plugin.gcp.datastore.util.DatastorePropertyUtil;
import io.cdap.plugin.gcp.datastore.util.DatastoreUtil;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Nullable;

public class DatastoreSinkConfig
extends GCPReferenceSinkConfig {
    @Name(value="namespace")
    @Macro
    @Nullable
    @Description(value="Namespace of the entities to write. A namespace partitions entities into a subset of Cloud Datastore.If no value is provided, the `default` namespace will be used.")
    private String namespace;
    @Name(value="kind")
    @Macro
    @Description(value="Kind of entities to write. Kinds are used to categorize entities in Cloud Datastore. A kind is equivalent to the relational database table notion.")
    private String kind;
    @Name(value="keyType")
    @Macro
    @Description(value="Type of key assigned to entities written to the Datastore. The type can be one of four values: `Auto-generated key` - key will be generated Cloud Datastore as a Numeric ID, `Custom name` - key will be provided as a field in the input records. The key field must not be nullable and must be of type STRING, INT or LONG, `Key literal` - key will be provided as a field in the input records in key literal format. The key field type must be a non-nullable string and the value must be in key literal format, `URL-safe key` - key will be provided as a field in the input records in encoded URL form. The key field type must be a non-nullable string and the value must be a URL-safe string.")
    private String keyType;
    @Name(value="keyAlias")
    @Macro
    @Nullable
    @Description(value="The field that will be used as the entity key when writing to Cloud Datastore. This must be provided when the Key Type is not auto generated.")
    private String keyAlias;
    @Name(value="ancestor")
    @Macro
    @Nullable
    @Description(value="Ancestor identifies the common root entity in which the entities are grouped. An ancestor must be specified in key literal format: key(<kind>, <identifier>, <kind>, <identifier>, [...]). Example: `key(kind_1, 'stringId', kind_2, 100)`")
    private String ancestor;
    @Name(value="indexStrategy")
    @Macro
    @Description(value="Defines which fields will be indexed in Cloud Datastore. Can be one of three options: `All` - all fields will be indexed, `None` - none of fields will be indexed, `Custom` - indexed fields will be provided in `Indexed Properties`.")
    private String indexStrategy;
    @Name(value="indexedProperties")
    @Macro
    @Nullable
    @Description(value="Fields to index in Cloud Datastore. A value must be provided if the Index Strategy is Custom, otherwise it is ignored.")
    private String indexedProperties;
    @Name(value="batchSize")
    @Macro
    @Description(value="Maximum number of entities that can be passed in one batch to a Commit operation. The minimum value is 1 and maximum value is 500")
    private int batchSize;
    @Name(value="useTransactions")
    @Macro
    @Nullable
    @Description(value="Define if this sink should use transactions to write records into Datastore.")
    private Boolean useTransactions;

    public DatastoreSinkConfig() {
    }

    @VisibleForTesting
    public DatastoreSinkConfig(String referenceName, String project, String serviceFilePath, @Nullable String namespace, String kind, String keyType, @Nullable String keyAlias, @Nullable String ancestor, String indexStrategy, int batchSize, @Nullable String indexedProperties) {
        this.referenceName = referenceName;
        this.project = project;
        this.serviceFilePath = serviceFilePath;
        this.namespace = namespace;
        this.kind = kind;
        this.indexStrategy = indexStrategy;
        this.indexedProperties = indexedProperties;
        this.keyType = keyType;
        this.keyAlias = keyAlias;
        this.ancestor = ancestor;
        this.batchSize = batchSize;
    }

    public String getNamespace() {
        return DatastorePropertyUtil.getNamespace(this.namespace);
    }

    public String getKind() {
        return this.kind;
    }

    public SinkKeyType getKeyType(FailureCollector collector) {
        Optional<SinkKeyType> sinkKeyType = SinkKeyType.fromValue(this.keyType);
        if (sinkKeyType.isPresent()) {
            return sinkKeyType.get();
        }
        collector.addFailure("Unsupported key type value: " + this.keyType, String.format("Supported types are: %s", SinkKeyType.getSupportedTypes())).withConfigProperty("keyType");
        throw collector.getOrThrowException();
    }

    public String getKeyAlias() {
        return DatastorePropertyUtil.getKeyAlias(this.keyAlias);
    }

    public List<Key.PathElement> getAncestor(FailureCollector collector) {
        try {
            return DatastorePropertyUtil.parseKeyLiteral(this.ancestor);
        }
        catch (IllegalArgumentException e) {
            collector.addFailure(e.getMessage(), null).withConfigProperty("ancestor");
            throw collector.getOrThrowException();
        }
    }

    public IndexStrategy getIndexStrategy(FailureCollector collector) {
        Optional<IndexStrategy> indexStrategy = IndexStrategy.fromValue(this.indexStrategy);
        if (indexStrategy.isPresent()) {
            return indexStrategy.get();
        }
        collector.addFailure("Unsupported index strategy value: " + this.indexStrategy, String.format("Supported index strategies are: %s", IndexStrategy.getSupportedStrategies())).withConfigProperty("indexStrategy");
        throw collector.getOrThrowException();
    }

    public Set<String> getIndexedProperties() {
        if (Strings.isNullOrEmpty((String)this.indexedProperties)) {
            return Collections.emptySet();
        }
        return Stream.of(this.indexedProperties.split(",")).map(String::trim).filter(name -> !name.isEmpty()).collect(Collectors.toSet());
    }

    public int getBatchSize() {
        return this.batchSize;
    }

    public boolean shouldUseTransactions() {
        return this.useTransactions != null ? this.useTransactions : true;
    }

    public boolean shouldUseAutoGeneratedKey(FailureCollector collector) {
        return this.getKeyType(collector) == SinkKeyType.AUTO_GENERATED_KEY;
    }

    public void validate(@Nullable Schema schema, FailureCollector collector) {
        super.validate(collector);
        this.validateKind(collector);
        this.validateAncestors(collector);
        this.validateBatchSize(collector);
        this.validateDatastoreConnection(collector);
        if (schema != null) {
            this.validateSchema(schema, collector);
            this.validateKeyType(schema, collector);
            this.validateIndexStrategy(schema, collector);
        }
    }

    private void validateSchema(Schema schema, FailureCollector collector) {
        List fields = schema.getFields();
        if (fields == null || fields.isEmpty()) {
            collector.addFailure("Sink schema must contain at least one field", null);
        } else {
            fields.forEach(f -> this.validateSinkFieldSchema(f.getName(), f.getSchema(), collector));
        }
    }

    private void validateSinkFieldSchema(String fieldName, Schema fieldSchema, FailureCollector collector) {
        Schema.LogicalType logicalType = fieldSchema.getLogicalType();
        if (logicalType != null) {
            switch (logicalType) {
                case TIMESTAMP_MICROS: 
                case TIMESTAMP_MILLIS: 
                case DATETIME: {
                    break;
                }
                default: {
                    collector.addFailure(String.format("Field '%s' is of unsupported type '%s'", fieldName, fieldSchema.getDisplayName()), "Supported types are: string, double, boolean, bytes, int, float, long, record, array, union and timestamp.").withInputSchemaField(fieldName);
                }
            }
            return;
        }
        switch (fieldSchema.getType()) {
            case STRING: 
            case DOUBLE: 
            case BOOLEAN: 
            case BYTES: 
            case INT: 
            case FLOAT: 
            case LONG: 
            case NULL: {
                return;
            }
            case RECORD: {
                this.validateSchema(fieldSchema, collector);
                return;
            }
            case ARRAY: {
                if (fieldSchema.getComponentSchema() == null) {
                    collector.addFailure(String.format("Field '%s' has no schema for array type", fieldName), "Ensure array component has schema.").withInputSchemaField(fieldName);
                    return;
                }
                Schema componentSchema = fieldSchema.getComponentSchema();
                if (Schema.Type.ARRAY == componentSchema.getType()) {
                    collector.addFailure(String.format("Field '%s' is of unsupported type array of array.", fieldName), "Ensure the field has valid type.").withInputSchemaField(fieldName);
                    return;
                }
                this.validateSinkFieldSchema(fieldName, componentSchema, collector);
                return;
            }
            case UNION: {
                fieldSchema.getUnionSchemas().forEach(unionSchema -> this.validateSinkFieldSchema(fieldName, (Schema)unionSchema, collector));
                return;
            }
        }
        collector.addFailure(String.format("Field '%s' is of unsupported type '%s'", fieldName, fieldSchema.getDisplayName()), "Supported types are: string, double, boolean, bytes, long, record, array, union and timestamp.").withInputSchemaField(fieldName);
    }

    private void validateIndexStrategy(Schema schema, FailureCollector collector) {
        if (this.containsMacro("indexStrategy")) {
            return;
        }
        if (this.getIndexStrategy(collector) == IndexStrategy.CUSTOM) {
            this.validateIndexedProperties(schema, collector);
        }
    }

    @VisibleForTesting
    void validateDatastoreConnection(FailureCollector collector) {
        if (!this.shouldConnect()) {
            return;
        }
        try {
            DatastoreUtil.getDatastoreV1(this.getServiceAccount(), this.isServiceAccountFilePath(), this.getProject());
        }
        catch (DatastoreInitializationException e) {
            collector.addFailure(e.getMessage(), "Ensure properties like project, service account file path are correct.").withConfigProperty("serviceFilePath").withConfigProperty("project");
        }
    }

    private void validateKind(FailureCollector collector) {
        if (this.containsMacro("kind")) {
            return;
        }
        if (Strings.isNullOrEmpty((String)this.kind)) {
            collector.addFailure("Kind must be specified.", null).withConfigProperty("kind");
        }
    }

    private void validateKeyType(Schema schema, FailureCollector collector) {
        if (this.containsMacro("keyType") || this.shouldUseAutoGeneratedKey(collector)) {
            return;
        }
        SinkKeyType keyType = this.getKeyType(collector);
        Schema.Field field = schema.getField(this.keyAlias);
        if (field == null) {
            collector.addFailure(String.format("Key field '%s' does not exist in the schema", this.keyAlias), "Change the Key field to be one of the schema fields.").withConfigProperty("keyAlias");
            return;
        }
        Schema fieldSchema = field.getSchema();
        Schema.Type type = fieldSchema.getType();
        if (Schema.Type.STRING != type && Schema.Type.LONG != type && Schema.Type.INT != type) {
            fieldSchema = fieldSchema.isNullable() ? fieldSchema.getNonNullable() : fieldSchema;
            collector.addFailure(String.format("Key field '%s' is of unsupported type '%s'", this.keyAlias, fieldSchema.getDisplayName()), "Ensure the type is non-nullable string, int or long.").withConfigProperty("keyAlias").withInputSchemaField(this.keyAlias);
        } else if ((Schema.Type.LONG == type || Schema.Type.INT == type) && keyType != SinkKeyType.CUSTOM_NAME) {
            collector.addFailure(String.format("Incorrect Key field '%s' type defined '%s'.", this.keyAlias, fieldSchema.getDisplayName()), String.format("'%s' type supported only by Key type '%s'", type, SinkKeyType.CUSTOM_NAME.getValue())).withConfigProperty("keyAlias").withInputSchemaField(this.keyAlias);
        }
    }

    private void validateIndexedProperties(Schema schema, FailureCollector collector) {
        if (this.containsMacro("indexedProperties")) {
            return;
        }
        Set<String> indexedProperties = this.getIndexedProperties();
        if (indexedProperties.isEmpty()) {
            return;
        }
        List missedProperties = indexedProperties.stream().filter(name -> schema.getField(name) == null).collect(Collectors.toList());
        for (String missingProperty : missedProperties) {
            collector.addFailure(String.format("Index Property '%s' does not exist in the input schema.", missingProperty), "Change Index property to be one of the schema fields.").withConfigElement("indexedProperties", missingProperty);
        }
    }

    private void validateAncestors(FailureCollector collector) {
        if (this.containsMacro("ancestor")) {
            return;
        }
        this.getAncestor(collector);
    }

    private void validateBatchSize(FailureCollector collector) {
        if (this.containsMacro("batchSize")) {
            return;
        }
        if (this.batchSize < 1 || this.batchSize > 500) {
            collector.addFailure(String.format("Invalid Datastore batch size '%d'.", this.batchSize), String.format("Ensure the batch size is at least 1 or at most '%d'", 500)).withConfigProperty("batchSize");
        }
    }

    public boolean shouldConnect() {
        return !this.containsMacro("serviceFilePath") && !this.containsMacro("serviceAccountJSON") && !this.containsMacro("serviceAccountType") && !this.containsMacro("project") && !this.containsMacro("kind") && !this.containsMacro("namespace") && !this.containsMacro("ancestor") && this.tryGetProject() != null && !this.autoServiceAccountUnavailable();
    }
}

