/*
 * Decompiled with CFR 0.152.
 */
package io.cdap.plugin.gcp.datastore.sink;

import com.google.datastore.v1.Entity;
import com.google.datastore.v1.Key;
import com.google.datastore.v1.PartitionId;
import com.google.datastore.v1.Value;
import com.google.datastore.v1.client.DatastoreHelper;
import com.google.protobuf.ByteString;
import com.google.protobuf.Message;
import com.google.protobuf.NullValue;
import com.google.protobuf.TextFormat;
import io.cdap.cdap.api.data.format.StructuredRecord;
import io.cdap.cdap.api.data.format.UnexpectedFormatException;
import io.cdap.cdap.api.data.schema.Schema;
import io.cdap.plugin.gcp.datastore.sink.util.IndexStrategy;
import io.cdap.plugin.gcp.datastore.sink.util.SinkKeyType;
import io.cdap.plugin.gcp.datastore.util.DatastorePropertyUtil;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Array;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RecordToEntityTransformer {
    private static final Logger LOG = LoggerFactory.getLogger(RecordToEntityTransformer.class);
    private final String project;
    private final String namespace;
    private final String kind;
    private final SinkKeyType keyType;
    private final String keyAlias;
    private final List<Key.PathElement> ancestors;
    private final Set<String> indexedProperties;
    private final Predicate<String> excludedFromIndex;

    public RecordToEntityTransformer(String project, String namespace, String kind, SinkKeyType keyType, String keyAlias, List<Key.PathElement> ancestors, IndexStrategy indexStrategy, Set<String> indexedProperties) {
        this.project = project;
        this.namespace = namespace;
        this.kind = kind;
        this.keyType = keyType;
        this.keyAlias = keyAlias;
        this.ancestors = ancestors;
        this.indexedProperties = indexedProperties;
        this.excludedFromIndex = this.isExcludedFromIndex(indexStrategy);
    }

    public Entity transformStructuredRecord(StructuredRecord record) {
        Key.Builder keyBuilder = Key.newBuilder().setPartitionId(PartitionId.newBuilder().setNamespaceId(this.namespace).setProjectId(this.project).build());
        Entity.Builder entityBuilder = Entity.newBuilder();
        List fields = Objects.requireNonNull(record.getSchema().getFields(), "Schema fields cannot be empty");
        boolean useAutoGeneratedKey = SinkKeyType.AUTO_GENERATED_KEY == this.keyType;
        for (Schema.Field field : fields) {
            String fieldName = field.getName();
            if (!useAutoGeneratedKey && this.keyAlias.equals(fieldName)) {
                entityBuilder.setKey(this.convertToKey(record, keyBuilder, field));
                continue;
            }
            entityBuilder.putProperties(fieldName, this.convertToValue(fieldName, field.getSchema(), record, this.excludedFromIndex.test(fieldName)));
        }
        if (useAutoGeneratedKey) {
            entityBuilder.setKey(this.getAutoGeneratedKey(keyBuilder));
        }
        return entityBuilder.build();
    }

    private Predicate<String> isExcludedFromIndex(IndexStrategy indexStrategy) {
        switch (indexStrategy) {
            case ALL: {
                return fieldName -> false;
            }
            case NONE: {
                return fieldName -> true;
            }
            case CUSTOM: {
                return fieldName -> !this.indexedProperties.contains(fieldName);
            }
        }
        throw new IllegalStateException(String.format("Unsupported index strategy '%s'", new Object[]{indexStrategy}));
    }

    private Key getAutoGeneratedKey(Key.Builder keyBuilder) {
        if (!this.ancestors.isEmpty()) {
            keyBuilder.addAllPath(this.ancestors);
        }
        keyBuilder.addPath(Key.PathElement.newBuilder().setKind(this.kind));
        return keyBuilder.build();
    }

    private Key convertToKey(StructuredRecord record, Key.Builder keyBuilder, Schema.Field field) {
        Schema.Type schemaType = field.getSchema().getType();
        switch (schemaType) {
            case STRING: {
                String strValue = (String)record.get(field.getName());
                if (strValue.isEmpty()) {
                    throw new IllegalStateException(String.format("Key value cannot be empty. Key field: '%s', Key type: '%s'", field.getName(), this.keyType.getValue()));
                }
                switch (this.keyType) {
                    case CUSTOM_NAME: {
                        if (!this.ancestors.isEmpty()) {
                            keyBuilder.addAllPath(this.ancestors);
                        }
                        return keyBuilder.addPath(Key.PathElement.newBuilder().setName(strValue).setKind(this.kind)).build();
                    }
                    case KEY_LITERAL: {
                        return this.transformKeyLiteralToKey(field, keyBuilder, strValue);
                    }
                    case URL_SAFE_KEY: {
                        return this.transformToUrlSafeKey(field, this.keyType, strValue);
                    }
                }
                throw new IllegalStateException(String.format("Field '%s' of type '%s' cannot be used as a Cloud Datastore key type '%s'", field.getName(), schemaType, this.keyType.getValue()));
            }
            case INT: 
            case LONG: {
                if (this.keyType == SinkKeyType.CUSTOM_NAME) {
                    if (!this.ancestors.isEmpty()) {
                        keyBuilder.addAllPath(this.ancestors);
                    }
                    Number numValue = (Number)record.get(field.getName());
                    return keyBuilder.addPath(Key.PathElement.newBuilder().setKind(this.kind).setId(numValue.longValue()).build()).build();
                }
                throw new IllegalStateException(String.format("Key field '%s' of type '%s' is not supported for record type: '%s'", field.getName(), this.keyType.getValue(), schemaType));
            }
        }
        String foundType = field.getSchema().isNullable() ? "nullable " + field.getSchema().getNonNullable().getType().name() : schemaType.name();
        throw new IllegalStateException(String.format("Key field '%s' of type '%s' is not supported by key type: '%s'", field.getName(), foundType, this.keyType.getValue()));
    }

    private Key transformToUrlSafeKey(Schema.Field field, SinkKeyType keyType, String urlSafeKey) {
        Key.Builder builder = Key.newBuilder();
        try {
            String utf8Str = URLDecoder.decode(urlSafeKey, StandardCharsets.UTF_8.name());
            TextFormat.merge((CharSequence)utf8Str, (Message.Builder)builder);
        }
        catch (UnsupportedEncodingException e) {
            throw new IllegalStateException("Unexpected decoding exception", e);
        }
        catch (TextFormat.ParseException e) {
            throw new IllegalArgumentException("Could not parse key", e);
        }
        Key key = builder.build();
        List pathElements = key.getPathList();
        Key.PathElement keyElement = (Key.PathElement)pathElements.get(pathElements.size() - 1);
        if (!Objects.equals(this.project, key.getPartitionId().getProjectId())) {
            throw new IllegalArgumentException(String.format("%s projectId must be equal to defined projectId. Key field: '%s', Expected: '%s', Received: '%s'", keyType.getValue(), field.getName(), this.project, key.getPartitionId().getProjectId()));
        }
        if (!Objects.equals(this.namespace, key.getPartitionId().getNamespaceId())) {
            throw new IllegalArgumentException(String.format("%s namespace must be equal to defined namespace. Key field: '%s', Expected: '%s', Received: '%s'", keyType.getValue(), field.getName(), this.namespace, key.getPartitionId().getNamespaceId()));
        }
        if (!Objects.equals(this.kind, keyElement.getKind())) {
            throw new IllegalArgumentException(String.format("%s kind must be equal to defined kind. Key field: '%s', Expected: '%s', Received: '%s'", keyType.getValue(), field.getName(), this.kind, keyElement.getKind()));
        }
        return key;
    }

    private Key transformKeyLiteralToKey(Schema.Field field, Key.Builder keyBuilder, String keyLiteral) {
        List<Key.PathElement> pathElements = DatastorePropertyUtil.parseKeyLiteral(keyLiteral);
        LOG.trace("Key Literal '{}' path elements: {}", (Object)keyLiteral, pathElements);
        if (pathElements.isEmpty()) {
            throw new IllegalArgumentException(String.format("Field '%s' value has unexpected Key Literal format: '%s'", field.getName(), keyLiteral));
        }
        Key.PathElement keyElement = pathElements.get(pathElements.size() - 1);
        LOG.trace("Detected keyElement: '{}'", (Object)keyElement);
        if (!this.kind.equals(keyElement.getKind())) {
            throw new IllegalArgumentException(String.format("Received Key Literal kind '%s' must match defined kind '%s'. Field '%s', key literal: '%s'", keyElement.getKind(), this.kind, field.getName(), keyLiteral));
        }
        keyBuilder.addAllPath(pathElements);
        return keyBuilder.build();
    }

    private Value convertToValue(String fieldName, Schema fieldSchema, StructuredRecord record, boolean excludeFromIndex) {
        if (record.get(fieldName) == null) {
            return Value.newBuilder().setNullValue(NullValue.NULL_VALUE).setExcludeFromIndexes(excludeFromIndex).build();
        }
        Schema.LogicalType logicalType = fieldSchema.getLogicalType();
        if (logicalType != null) {
            switch (logicalType) {
                case TIMESTAMP_MILLIS: 
                case TIMESTAMP_MICROS: {
                    ZonedDateTime ts = this.getValue(arg_0 -> ((StructuredRecord)record).getTimestamp(arg_0), fieldName, logicalType.getToken(), ZonedDateTime.class);
                    return DatastoreHelper.makeValue((Date)Date.from(ts.toInstant())).setExcludeFromIndexes(excludeFromIndex).build();
                }
                case DATETIME: {
                    break;
                }
                default: {
                    throw new IllegalStateException(String.format("Record type '%s' is not supported for field '%s'", logicalType.getToken(), fieldName));
                }
            }
        }
        Schema.Type fieldType = fieldSchema.getType();
        switch (fieldType) {
            case STRING: {
                String stringValue = this.getValue(arg_0 -> ((StructuredRecord)record).get(arg_0), fieldName, fieldType.toString(), String.class);
                return DatastoreHelper.makeValue((String)stringValue).setExcludeFromIndexes(excludeFromIndex).build();
            }
            case INT: 
            case LONG: {
                Number longValue = this.getValue(arg_0 -> ((StructuredRecord)record).get(arg_0), fieldName, fieldType.toString(), Number.class);
                return DatastoreHelper.makeValue((long)longValue.longValue()).setExcludeFromIndexes(excludeFromIndex).build();
            }
            case FLOAT: 
            case DOUBLE: {
                Number doubleValue = this.getValue(arg_0 -> ((StructuredRecord)record).get(arg_0), fieldName, fieldType.toString(), Number.class);
                return DatastoreHelper.makeValue((double)doubleValue.doubleValue()).setExcludeFromIndexes(excludeFromIndex).build();
            }
            case BOOLEAN: {
                Boolean booleanValue = this.getValue(arg_0 -> ((StructuredRecord)record).get(arg_0), fieldName, fieldType.toString(), Boolean.class);
                return DatastoreHelper.makeValue((boolean)booleanValue).setExcludeFromIndexes(excludeFromIndex).build();
            }
            case BYTES: {
                byte[] byteArray = this.getValue(arg_0 -> ((StructuredRecord)record).get(arg_0), fieldName, fieldType.toString(), byte[].class);
                return DatastoreHelper.makeValue((ByteString)ByteString.copyFrom((byte[])byteArray)).setExcludeFromIndexes(excludeFromIndex).build();
            }
            case RECORD: {
                StructuredRecord nestedRecord = this.getValue(arg_0 -> ((StructuredRecord)record).get(arg_0), fieldName, fieldType.toString(), StructuredRecord.class);
                Entity.Builder nestedBuilder = Entity.newBuilder();
                Objects.requireNonNull(fieldSchema.getFields(), "Nested Schema fields cannot be empty").forEach(nestedField -> nestedBuilder.putProperties(nestedField.getName(), this.convertToValue(nestedField.getName(), nestedField.getSchema(), nestedRecord, excludeFromIndex)));
                return DatastoreHelper.makeValue((Entity)nestedBuilder.build()).setExcludeFromIndexes(excludeFromIndex).build();
            }
            case ARRAY: {
                Schema elementSchema = Schema.recordOf((String)"arrayElementSchema", (Schema.Field[])new Schema.Field[]{Schema.Field.of((String)fieldName, (Schema)fieldSchema.getComponentSchema())});
                Collection<Object> arrayValues = this.toCollection(fieldName, fieldType, record.get(fieldName));
                List values = arrayValues.stream().map(value -> {
                    StructuredRecord structuredRecord = StructuredRecord.builder((Schema)elementSchema).set(fieldName, value).build();
                    return this.convertToValue(fieldName, elementSchema.getField(fieldName).getSchema(), structuredRecord, false);
                }).collect(Collectors.toList());
                return DatastoreHelper.makeValue(values).build();
            }
            case UNION: {
                if (fieldSchema.isNullable()) {
                    return this.convertToValue(fieldName, fieldSchema.getNonNullable(), record, excludeFromIndex);
                }
                for (Schema unionSchema : fieldSchema.getUnionSchemas()) {
                    try {
                        return this.convertToValue(fieldName, unionSchema, record, excludeFromIndex);
                    }
                    catch (UnexpectedFormatException | IllegalStateException throwable) {
                    }
                }
                throw new IllegalStateException(String.format("Field '%s' is of unexpected type '%s'. Declared 'complex UNION' types: %s", fieldName, record.get(fieldName).getClass().getSimpleName(), fieldSchema.getUnionSchemas()));
            }
        }
        throw new IllegalStateException(String.format("Record type '%s' is not supported for field '%s'", fieldType.name(), fieldName));
    }

    private <T> T getValue(Function<String, T> valueExtractor, String fieldName, String fieldType, Class<T> clazz) {
        T value = valueExtractor.apply(fieldName);
        if (clazz.isAssignableFrom(value.getClass())) {
            return clazz.cast(value);
        }
        throw new UnexpectedFormatException(String.format("Field '%s' is not of expected type '%s'", fieldName, fieldType));
    }

    private Collection<Object> toCollection(String fieldName, Schema.Type fieldType, Object value) {
        Function<String, Collection> valueExtractor = name -> {
            throw new UnexpectedFormatException(String.format("Field '%s' of type '%s' has unexpected value '%s'", name, fieldType, value));
        };
        if (value instanceof Collection) {
            valueExtractor = name -> (Collection)value;
        } else if (value.getClass().isArray()) {
            valueExtractor = name -> this.convertToObjectCollection(value);
        }
        return this.getValue(valueExtractor, fieldName, fieldType.toString(), Collection.class);
    }

    private Collection<Object> convertToObjectCollection(Object array) {
        Class<?> ofArray = array.getClass().getComponentType();
        if (ofArray.isPrimitive()) {
            ArrayList<Object> list = new ArrayList<Object>();
            int length = Array.getLength(array);
            for (int i = 0; i < length; ++i) {
                list.add(Array.get(array, i));
            }
            return list;
        }
        return Arrays.asList((Object[])array);
    }
}

