/*
 * Decompiled with CFR 0.152.
 */
package io.cdap.plugin.gcp.datastore.source;

import com.google.cloud.Timestamp;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import com.google.datastore.v1.Key;
import com.google.datastore.v1.KindExpression;
import com.google.datastore.v1.PartitionId;
import com.google.datastore.v1.PropertyFilter;
import com.google.datastore.v1.Query;
import com.google.datastore.v1.Value;
import com.google.datastore.v1.client.DatastoreHelper;
import com.google.protobuf.NullValue;
import io.cdap.cdap.api.annotation.Description;
import io.cdap.cdap.api.annotation.Macro;
import io.cdap.cdap.api.annotation.Name;
import io.cdap.cdap.api.data.schema.Schema;
import io.cdap.cdap.api.dataset.lib.KeyValue;
import io.cdap.cdap.etl.api.FailureCollector;
import io.cdap.plugin.common.KeyValueListParser;
import io.cdap.plugin.gcp.common.GCPReferenceSourceConfig;
import io.cdap.plugin.gcp.datastore.exception.DatastoreInitializationException;
import io.cdap.plugin.gcp.datastore.source.util.SourceKeyType;
import io.cdap.plugin.gcp.datastore.util.DatastorePropertyUtil;
import io.cdap.plugin.gcp.datastore.util.DatastoreUtil;
import java.io.IOException;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import javax.annotation.Nullable;

public class DatastoreSourceConfig
extends GCPReferenceSourceConfig {
    private static final KeyValueListParser KV_PARSER = new KeyValueListParser(";", "\\|");
    private static final Set<Schema.LogicalType> supportedLogicalTypes = new ImmutableSet.Builder().add((Object[])new Schema.LogicalType[]{Schema.LogicalType.DATETIME, Schema.LogicalType.TIMESTAMP_MICROS}).build();
    @Name(value="namespace")
    @Macro
    @Nullable
    @Description(value="Namespace of the entities to read. A namespace partitions entities into a subset of Cloud Datastore. If no value is provided, the `default` namespace will be used.")
    private String namespace;
    @Name(value="kind")
    @Macro
    @Description(value="Kind of entities to read. Kinds are used to categorize entities in Cloud Datastore. A kind is equivalent to the relational database table notion.")
    private String kind;
    @Name(value="ancestor")
    @Macro
    @Nullable
    @Description(value="Ancestor of entities to read. An ancestor identifies the common parent entity that all the child entities share. The value must be provided in key literal format: key(<kind>, <identifier>, <kind>, <identifier>, [...]). For example: `key(kind_1, 'stringId', kind_2, 100)`")
    private String ancestor;
    @Name(value="filters")
    @Macro
    @Nullable
    @Description(value="List of filters to apply when reading entities from Cloud Datastore. Only entities that satisfy all the filters will be read. The filter key corresponds to a field in the schema. The field type must be STRING, LONG, DOUBLE, BOOLEAN, or TIMESTAMP. The filter value indicates what value that field must have in order to be read. If no value is provided, it means the value must be null in order to be read. TIMESTAMP string should be in the RFC 3339 format without the timezone offset (always ends in Z). Expected pattern: `yyyy-MM-dd'T'HH:mm:ssX`, for example: `2011-10-02T13:12:55Z`.")
    private String filters;
    @Name(value="numSplits")
    @Macro
    @Description(value="Desired number of splits to divide the query into when reading from Cloud Datastore. Fewer splits may be created if the query cannot be divided into the desired number of splits.")
    private int numSplits;
    @Name(value="keyType")
    @Macro
    @Description(value="Type of entity key read from the Cloud Datastore. The type can be one of three values: `None` - key will not be included, `Key literal` - key will be included in Cloud Datastore key literal format including complete path with ancestors, `URL-safe key` - key will be included in the encoded form that can be used as part of a URL. Note, if `Key literal` or `URL-safe key` is selected, default key name (`__key__`) or its alias must be present in the schema with non-nullable STRING type.")
    private String keyType;
    @Name(value="keyAlias")
    @Macro
    @Nullable
    @Description(value="Name of the field to set as the key field. This value is ignored if the `Key Type` is set to `None`. If no value is provided, `__key__` is used.")
    private String keyAlias;
    @Name(value="schema")
    @Macro
    @Nullable
    @Description(value="Schema of the data to read. Can be imported or fetched by clicking the `Get Schema` button.")
    private String schema;

    public DatastoreSourceConfig() {
    }

    @VisibleForTesting
    DatastoreSourceConfig(String referenceName, String project, String serviceFilePath, @Nullable String namespace, String kind, @Nullable String ancestor, @Nullable String filters, int numSplits, String keyType, @Nullable String keyAlias, String schema) {
        this.referenceName = referenceName;
        this.project = project;
        this.serviceFilePath = serviceFilePath;
        this.namespace = namespace;
        this.kind = kind;
        this.ancestor = ancestor;
        this.filters = filters;
        this.numSplits = numSplits;
        this.keyType = keyType;
        this.keyAlias = keyAlias;
        this.schema = schema;
    }

    public String getReferenceName() {
        return this.referenceName;
    }

    public Schema getSchema(FailureCollector collector) {
        if (Strings.isNullOrEmpty((String)this.schema)) {
            return null;
        }
        try {
            return Schema.parseJson((String)this.schema);
        }
        catch (IOException e) {
            collector.addFailure("Invalid schema: " + e.getMessage(), null).withConfigProperty("schema");
            throw collector.getOrThrowException();
        }
    }

    public String getNamespace() {
        return DatastorePropertyUtil.getNamespace(this.namespace);
    }

    public String getKind() {
        return this.kind;
    }

    public List<Key.PathElement> getAncestor(FailureCollector collector) {
        try {
            return DatastorePropertyUtil.parseKeyLiteral(this.ancestor);
        }
        catch (IllegalArgumentException e) {
            collector.addFailure(e.getMessage(), null).withConfigProperty("ancestor");
            throw collector.getOrThrowException();
        }
    }

    public Map<String, String> getFilters() {
        if (Strings.isNullOrEmpty((String)this.filters)) {
            return Collections.emptyMap();
        }
        return StreamSupport.stream(KV_PARSER.parse(this.filters).spliterator(), false).collect(Collectors.toMap(KeyValue::getKey, KeyValue::getValue, (o, n) -> n, LinkedHashMap::new));
    }

    public int getNumSplits() {
        return this.numSplits;
    }

    public SourceKeyType getKeyType(FailureCollector collector) {
        Optional<SourceKeyType> sourceKeyType = SourceKeyType.fromValue(this.keyType);
        if (sourceKeyType.isPresent()) {
            return sourceKeyType.get();
        }
        collector.addFailure("Unsupported key type value: " + this.keyType, String.format("Supported key types are: %s", SourceKeyType.getSupportedTypes())).withConfigProperty("keyType");
        throw collector.getOrThrowException();
    }

    public boolean isIncludeKey(FailureCollector collector) {
        return SourceKeyType.NONE != this.getKeyType(collector);
    }

    public String getKeyAlias() {
        return DatastorePropertyUtil.getKeyAlias(this.keyAlias);
    }

    @Override
    public void validate(FailureCollector collector) {
        super.validate(collector);
        this.validateDatastoreConnection(collector);
        this.validateKind(collector);
        this.validateAncestor(collector);
        this.validateNumSplits(collector);
        if (this.containsMacro("schema")) {
            return;
        }
        Schema schema = this.getSchema(collector);
        if (schema != null) {
            this.validateSchema(schema, collector);
            this.validateFilters(schema, collector);
            this.validateKeyType(schema, collector);
        }
    }

    @VisibleForTesting
    void validateDatastoreConnection(FailureCollector collector) {
        if (!this.shouldConnect()) {
            return;
        }
        try {
            DatastoreUtil.getDatastoreV1(this.getServiceAccount(), this.isServiceAccountFilePath(), this.getProject());
        }
        catch (DatastoreInitializationException e) {
            collector.addFailure(e.getMessage(), "Ensure properties like project, service account file path are correct.").withConfigProperty("serviceFilePath").withConfigProperty("project");
        }
    }

    private void validateKind(FailureCollector collector) {
        if (this.containsMacro("kind")) {
            return;
        }
        if (Strings.isNullOrEmpty((String)this.kind)) {
            collector.addFailure("Kind must be specified.", null).withConfigProperty("kind");
        }
    }

    private void validateAncestor(FailureCollector collector) {
        if (!this.containsMacro("ancestor")) {
            this.getAncestor(collector);
        }
    }

    private void validateNumSplits(FailureCollector collector) {
        if (this.containsMacro("numSplits")) {
            return;
        }
        if (this.numSplits < 1) {
            collector.addFailure("Number of splits must be greater than 0", null).withConfigProperty("numSplits");
        }
    }

    private void validateSchema(Schema schema, FailureCollector collector) {
        List fields = schema.getFields();
        if (fields == null || fields.isEmpty()) {
            collector.addFailure("Source schema must contain at least one field", null).withConfigProperty("schema");
        } else {
            fields.forEach(f -> this.validateFieldSchema(f.getName(), f.getSchema(), collector));
        }
    }

    private void validateFieldSchema(String fieldName, Schema fieldSchema, FailureCollector collector) {
        Schema.LogicalType logicalType = fieldSchema.getLogicalType();
        if (logicalType != null && !supportedLogicalTypes.contains(logicalType)) {
            collector.addFailure(String.format("Field '%s' is of unsupported type '%s'", fieldName, fieldSchema.getDisplayName()), "Supported types are: string, double, boolean, bytes, long, record, array, union and timestamp.").withOutputSchemaField(fieldName);
            return;
        }
        switch (fieldSchema.getType()) {
            case STRING: 
            case DOUBLE: 
            case BOOLEAN: 
            case BYTES: 
            case LONG: 
            case NULL: {
                return;
            }
            case RECORD: {
                this.validateSchema(fieldSchema, collector);
                return;
            }
            case ARRAY: {
                if (fieldSchema.getComponentSchema() == null) {
                    collector.addFailure(String.format("Field '%s' has no schema for array type", fieldName), "Ensure array component has schema.").withOutputSchemaField(fieldName);
                    return;
                }
                Schema componentSchema = fieldSchema.getComponentSchema();
                if (Schema.Type.ARRAY == componentSchema.getType()) {
                    collector.addFailure(String.format("Field '%s' is of unsupported type array of array.", fieldName), "Ensure the field has valid type.").withOutputSchemaField(fieldName);
                    return;
                }
                this.validateFieldSchema(fieldName, componentSchema, collector);
                return;
            }
            case UNION: {
                fieldSchema.getUnionSchemas().forEach(unionSchema -> this.validateFieldSchema(fieldName, (Schema)unionSchema, collector));
                return;
            }
        }
        collector.addFailure(String.format("Field '%s' is of unsupported type '%s'", fieldName, fieldSchema.getDisplayName()), "Supported types are: string, double, boolean, bytes, long, record, array, union and timestamp.").withOutputSchemaField(fieldName);
    }

    private void validateFilters(Schema schema, FailureCollector collector) {
        if (this.containsMacro("filters")) {
            return;
        }
        try {
            Map<String, String> filters = this.getFilters();
            List missingProperties = filters.keySet().stream().filter(k -> schema.getField(k) == null).collect(Collectors.toList());
            for (String missingProperty : missingProperties) {
                collector.addFailure(String.format("Property '%s' does not exist in the schema.", missingProperty), "Change Property to be one of the schema fields.").withConfigElement("filters", missingProperty + "|" + filters.get(missingProperty));
            }
        }
        catch (IllegalArgumentException e) {
            collector.addFailure(e.getMessage(), null).withConfigProperty("filters");
        }
    }

    private void validateKeyType(Schema schema, FailureCollector collector) {
        if (this.containsMacro("keyType") || this.containsMacro("keyAlias")) {
            return;
        }
        if (this.isIncludeKey(collector)) {
            String key = this.getKeyAlias();
            Schema.Field field = schema.getField(key);
            if (field == null) {
                collector.addFailure(String.format("Key field '%s' does not exist in the schema.", key), "Change the Key field to be one of the schema fields.").withConfigProperty("keyAlias");
                return;
            }
            Schema fieldSchema = field.getSchema();
            Schema.Type type = fieldSchema.getType();
            if (Schema.Type.STRING != type) {
                fieldSchema = fieldSchema.isNullable() ? fieldSchema.getNonNullable() : fieldSchema;
                collector.addFailure(String.format("Key field '%s' is of unsupported type '%s'", key, fieldSchema.getDisplayName()), "Ensure the type is non-nullable String.").withConfigProperty("keyAlias").withOutputSchemaField(field.getName());
            }
        }
    }

    public Query constructPbQuery(FailureCollector collector) {
        Query.Builder builder = Query.newBuilder().addKind(KindExpression.newBuilder().setName(this.getKind()));
        List filters = this.getFilters().entrySet().stream().map(e -> DatastoreHelper.makeFilter((String)((String)e.getKey()), (PropertyFilter.Operator)PropertyFilter.Operator.EQUAL, (Value)this.constructFilterValue((String)e.getKey(), (String)e.getValue(), this.getSchema(collector))).build()).collect(Collectors.toList());
        List<Key.PathElement> ancestors = this.getAncestor(collector);
        if (!ancestors.isEmpty()) {
            filters.add(DatastoreHelper.makeAncestorFilter((Key)this.constructKey(ancestors, this.getProject(), this.getNamespace())).build());
        }
        if (!filters.isEmpty()) {
            builder.setFilter(DatastoreHelper.makeAndFilter(filters));
        }
        return builder.build();
    }

    private Key constructKey(List<Key.PathElement> pathElements, String project, String namespace) {
        Object[] elements = pathElements.stream().flatMap(pathElement -> Stream.of(pathElement.getKind(), pathElement.getIdTypeCase() == Key.PathElement.IdTypeCase.ID ? Long.valueOf(pathElement.getId()) : pathElement.getName())).toArray();
        return DatastoreHelper.makeKey((Object[])elements).setPartitionId(PartitionId.newBuilder().setProjectId(project).setNamespaceId(namespace).build()).build();
    }

    private Value constructFilterValue(String name, @Nullable String value, Schema schema) {
        Schema.Field field = Objects.requireNonNull(schema.getField(name));
        Schema fieldSchema = field.getSchema();
        if (Strings.isNullOrEmpty((String)value)) {
            return Value.newBuilder().setNullValue(NullValue.NULL_VALUE).build();
        }
        return this.constructFilterValue(name, fieldSchema, value);
    }

    private Value constructFilterValue(String name, Schema schema, String value) {
        Schema.LogicalType logicalType = schema.getLogicalType();
        if (logicalType != null) {
            if (logicalType == Schema.LogicalType.TIMESTAMP_MICROS) {
                Timestamp timestamp = Timestamp.parseTimestamp((String)value);
                return Value.newBuilder().setTimestampValue(timestamp.toProto()).build();
            }
            throw new IllegalStateException(String.format("Filter field '%s' is of unsupported type '%s'", name, logicalType.getToken()));
        }
        switch (schema.getType()) {
            case STRING: {
                return DatastoreHelper.makeValue((String)value).build();
            }
            case DOUBLE: {
                return DatastoreHelper.makeValue((double)Double.valueOf(value)).build();
            }
            case LONG: {
                return DatastoreHelper.makeValue((long)Long.valueOf(value)).build();
            }
            case BOOLEAN: {
                return DatastoreHelper.makeValue((boolean)Boolean.valueOf(value)).build();
            }
            case UNION: {
                if (schema.isNullable()) {
                    return this.constructFilterValue(name, schema.getNonNullable(), value);
                }
                throw new IllegalStateException(String.format("Filter field '%s' is of unsupported type 'complex UNION'", name));
            }
        }
        throw new IllegalStateException(String.format("Filter field '%s' is of unsupported type '%s'", name, schema.getType()));
    }

    public String toString() {
        return "DatastoreSourceConfig{referenceName='" + this.referenceName + '\'' + ", project='" + this.project + '\'' + ", serviceFilePath='" + this.serviceFilePath + '\'' + ", namespace='" + this.namespace + '\'' + ", kind='" + this.kind + '\'' + ", ancestor='" + this.ancestor + '\'' + ", filters='" + this.filters + '\'' + ", numSplits=" + this.numSplits + ", keyType='" + this.keyType + '\'' + ", keyAlias='" + this.keyAlias + '\'' + ", schema='" + this.schema + '\'' + "} ";
    }

    boolean shouldConnect() {
        return !this.containsMacro("schema") && !this.containsMacro("serviceAccountType") && !this.containsMacro("serviceFilePath") && !this.containsMacro("serviceAccountJSON") && !this.containsMacro("project") && !this.containsMacro("kind") && !this.containsMacro("namespace") && !this.containsMacro("ancestor") && !this.containsMacro("keyType") && !this.containsMacro("keyAlias") && this.tryGetProject() != null && !this.autoServiceAccountUnavailable();
    }
}

