/*
 * Decompiled with CFR 0.152.
 */
package io.cdap.plugin.gcp.datastore.source;

import com.google.common.annotations.VisibleForTesting;
import com.google.datastore.v1.ArrayValue;
import com.google.datastore.v1.Entity;
import com.google.datastore.v1.Key;
import com.google.datastore.v1.Value;
import com.google.protobuf.NullValue;
import com.google.protobuf.Timestamp;
import io.cdap.cdap.api.data.format.StructuredRecord;
import io.cdap.cdap.api.data.format.UnexpectedFormatException;
import io.cdap.cdap.api.data.schema.Schema;
import io.cdap.plugin.gcp.datastore.source.util.SourceKeyType;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.format.DateTimeParseException;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

public class EntityToRecordTransformer {
    private final Schema schema;
    private final SourceKeyType keyType;
    private final String keyAlias;

    public EntityToRecordTransformer(Schema schema, SourceKeyType keyType, String keyAlias) {
        this.schema = schema;
        this.keyType = keyType;
        this.keyAlias = keyAlias;
    }

    public StructuredRecord transformEntity(Entity entity) {
        StructuredRecord.Builder builder = StructuredRecord.builder((Schema)this.schema);
        List fields = Objects.requireNonNull(this.schema.getFields());
        for (Schema.Field field : fields) {
            String fieldName = field.getName();
            if (SourceKeyType.NONE != this.keyType && fieldName.equals(this.keyAlias)) {
                builder.set(fieldName, (Object)this.transformKeyToKeyString(entity.getKey()));
                continue;
            }
            this.populateRecordBuilder(builder, entity, field.getName(), field.getSchema());
        }
        return builder.build();
    }

    @VisibleForTesting
    String transformKeyToKeyString(Key key) {
        switch (this.keyType) {
            case KEY_LITERAL: {
                StringBuilder builder = new StringBuilder("key(");
                boolean firstIteration = true;
                block10: for (Key.PathElement pathElement : key.getPathList()) {
                    if (firstIteration) {
                        firstIteration = false;
                    } else {
                        builder.append(", ");
                    }
                    builder.append(pathElement.getKind()).append(", ");
                    switch (pathElement.getIdTypeCase()) {
                        case ID: {
                            builder.append(pathElement.getId());
                            continue block10;
                        }
                        case NAME: {
                            builder.append("'").append(pathElement.getName()).append("'");
                            continue block10;
                        }
                    }
                    throw new IllegalStateException(String.format("Unexpected path element type %s", pathElement.getIdTypeCase().toString()));
                }
                return builder.append(")").toString();
            }
            case URL_SAFE_KEY: {
                try {
                    return URLEncoder.encode(key.toString(), StandardCharsets.UTF_8.name());
                }
                catch (UnsupportedEncodingException e) {
                    throw new IllegalStateException("Unexpected encoding exception", e);
                }
            }
        }
        throw new IllegalStateException(String.format("Unable to transform key '%s' to type '%s' string representation", key, this.keyType.getValue()));
    }

    private void populateRecordBuilder(StructuredRecord.Builder builder, Entity entity, String fieldName, Schema fieldSchema) {
        Value defaultValue = Value.newBuilder().setNullValue(NullValue.NULL_VALUE).build();
        Value value = entity.getPropertiesOrDefault(fieldName, defaultValue);
        builder.set(fieldName, this.getValue(value, fieldName, fieldSchema));
    }

    private Object getValue(Value value, String fieldName, Schema fieldSchema) {
        if (value.getValueTypeCase() == Value.ValueTypeCase.NULL_VALUE) {
            return null;
        }
        Schema.LogicalType logicalType = fieldSchema.getLogicalType();
        if (logicalType != null) {
            switch (logicalType) {
                case TIMESTAMP_MICROS: {
                    Timestamp timestamp = (Timestamp)this.castValue(value, Value.ValueTypeCase.TIMESTAMP_VALUE, fieldName);
                    Instant zonedInstant = Instant.ofEpochSecond(timestamp.getSeconds()).plusNanos(timestamp.getNanos());
                    long micros = TimeUnit.SECONDS.toMicros(zonedInstant.getEpochSecond());
                    return Math.addExact(micros, TimeUnit.NANOSECONDS.toMicros(zonedInstant.getNano()));
                }
                case DATETIME: {
                    try {
                        LocalDateTime.parse(value.getStringValue());
                    }
                    catch (DateTimeParseException exception) {
                        throw new UnexpectedFormatException(String.format("Datetime field '%s' with value '%s' is not in ISO-8601 format.", fieldName, value.getStringValue()), (Throwable)exception);
                    }
                    return value.getStringValue();
                }
            }
            throw new IllegalStateException(String.format("Field '%s' is of unsupported type '%s'", fieldName, logicalType.getToken()));
        }
        Schema.Type fieldType = fieldSchema.getType();
        switch (fieldType) {
            case STRING: {
                return this.castValue(value, Value.ValueTypeCase.STRING_VALUE, fieldName);
            }
            case DOUBLE: {
                return this.castValue(value, Value.ValueTypeCase.DOUBLE_VALUE, fieldName);
            }
            case BOOLEAN: {
                return this.castValue(value, Value.ValueTypeCase.BOOLEAN_VALUE, fieldName);
            }
            case LONG: {
                return this.castValue(value, Value.ValueTypeCase.INTEGER_VALUE, fieldName);
            }
            case BYTES: {
                return this.castValue(value, Value.ValueTypeCase.BLOB_VALUE, fieldName);
            }
            case RECORD: {
                Entity nestedEntity = (Entity)this.castValue(value, Value.ValueTypeCase.ENTITY_VALUE, fieldName);
                StructuredRecord.Builder nestedBuilder = StructuredRecord.builder((Schema)fieldSchema);
                Objects.requireNonNull(fieldSchema.getFields()).forEach(nestedField -> this.populateRecordBuilder(nestedBuilder, nestedEntity, nestedField.getName(), nestedField.getSchema()));
                return nestedBuilder.build();
            }
            case ARRAY: {
                Schema componentSchema = fieldSchema.getComponentSchema();
                ArrayValue arrayValue = (ArrayValue)this.castValue(value, Value.ValueTypeCase.ARRAY_VALUE, fieldName);
                return arrayValue.getValuesList().stream().map(v -> this.getValue((Value)v, fieldName, componentSchema)).collect(Collectors.toList());
            }
            case UNION: {
                if (fieldSchema.isNullable()) {
                    return this.getValue(value, fieldName, fieldSchema.getNonNullable());
                }
                List unionSchemas = fieldSchema.getUnionSchemas();
                for (Schema unionSchema : unionSchemas) {
                    try {
                        return this.getValue(value, fieldName, unionSchema);
                    }
                    catch (UnexpectedFormatException | IllegalStateException throwable) {
                    }
                }
                throw new IllegalStateException(String.format("Field '%s' is of unexpected type '%s'. Declared 'complex UNION' types: %s", fieldName, value.getValueTypeCase(), unionSchemas));
            }
        }
        throw new IllegalStateException(String.format("Field '%s' is of unsupported type '%s'", fieldName, fieldType));
    }

    private Object castValue(Value value, Value.ValueTypeCase valueTypeCase, String fieldName) {
        if (value.getValueTypeCase() != valueTypeCase) {
            throw new UnexpectedFormatException(String.format("Field '%s' is not of expected type '%s'", fieldName, valueTypeCase.toString()));
        }
        switch (valueTypeCase) {
            case STRING_VALUE: {
                return value.getStringValue();
            }
            case DOUBLE_VALUE: {
                return value.getDoubleValue();
            }
            case BOOLEAN_VALUE: {
                return value.getBooleanValue();
            }
            case INTEGER_VALUE: {
                return value.getIntegerValue();
            }
            case BLOB_VALUE: {
                return value.getBlobValue().toByteArray();
            }
            case ARRAY_VALUE: {
                return value.getArrayValue();
            }
            case ENTITY_VALUE: {
                return value.getEntityValue();
            }
            case TIMESTAMP_VALUE: {
                return value.getTimestampValue();
            }
        }
        throw new UnexpectedFormatException(String.format("Unexpected value type '%s'", valueTypeCase.toString()));
    }
}

