/*
 * Decompiled with CFR 0.152.
 */
package io.cdap.plugin.gcp.gcs;

import com.google.api.gax.paging.Page;
import com.google.auth.Credentials;
import com.google.cloud.kms.v1.CryptoKeyName;
import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.Bucket;
import com.google.cloud.storage.CopyWriter;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageException;
import com.google.cloud.storage.StorageOptions;
import com.google.common.annotations.VisibleForTesting;
import io.cdap.plugin.gcp.common.GCPConnectorConfig;
import io.cdap.plugin.gcp.common.GCPUtils;
import io.cdap.plugin.gcp.gcs.GCSPath;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Map;
import java.util.function.Consumer;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StorageClient {
    private static final Logger LOG = LoggerFactory.getLogger(StorageClient.class);
    private final Storage storage;

    public StorageClient(Storage storage) {
        this.storage = storage;
    }

    public Blob pickABlob(String path) {
        if (path == null || path.isEmpty()) {
            return null;
        }
        GCSPath gcsPath = GCSPath.from(path);
        Page blobPage = this.storage.list(gcsPath.getBucket(), new Storage.BlobListOption[]{Storage.BlobListOption.prefix((String)gcsPath.getName())});
        for (Blob blob : blobPage.getValues()) {
            if (blob.getName().endsWith("/")) continue;
            return blob;
        }
        return null;
    }

    public void setMetaData(Blob blob, Map<String, String> metaData) {
        if (blob == null || metaData == null || metaData.isEmpty()) {
            return;
        }
        this.storage.update(BlobInfo.newBuilder((BlobId)blob.getBlobId()).setMetadata(metaData).build());
    }

    public void mapMetaDataForAllBlobs(String path, Consumer<Map<String, String>> function) {
        if (path == null || path.isEmpty() || function == null) {
            return;
        }
        GCSPath gcsPath = GCSPath.from(path);
        Page blobPage = this.storage.list(gcsPath.getBucket(), new Storage.BlobListOption[]{Storage.BlobListOption.prefix((String)gcsPath.getName())});
        for (Blob blob : blobPage.iterateAll()) {
            Map metadata = blob.getMetadata();
            if (metadata == null) continue;
            function.accept(metadata);
        }
    }

    public void createBucketIfNotExists(GCSPath path, @Nullable String location, @Nullable CryptoKeyName cmekKeyName) {
        Bucket bucket = null;
        try {
            bucket = this.storage.get(path.getBucket(), new Storage.BucketGetOption[0]);
        }
        catch (StorageException e) {
            throw new RuntimeException(String.format("Unable to access bucket %s. ", path.getBucket()) + "Ensure you entered the correct bucket path and have permissions for it.", e);
        }
        if (bucket == null) {
            GCPUtils.createBucket(this.storage, path.getBucket(), location, cmekKeyName);
        }
    }

    public void copy(GCSPath sourcePath, GCSPath destPath, boolean recursive, boolean overwrite) {
        this.pairTraverse(sourcePath, destPath, recursive, overwrite, rec$ -> ((BlobPair)rec$).copy());
    }

    public void move(GCSPath sourcePath, GCSPath destPath, boolean recursive, boolean overwrite) {
        this.pairTraverse(sourcePath, destPath, recursive, overwrite, rec$ -> ((BlobPair)rec$).move());
    }

    private void pairTraverse(GCSPath sourcePath, GCSPath destPath, boolean recursive, boolean overwrite, Consumer<BlobPair> consumer) {
        boolean destinationBaseExists;
        Bucket sourceBucket = null;
        try {
            sourceBucket = this.storage.get(sourcePath.getBucket(), new Storage.BucketGetOption[0]);
        }
        catch (StorageException e) {
            throw new RuntimeException(String.format("Unable to access source bucket %s. ", sourcePath.getBucket()) + "Ensure you entered the correct bucket path.", e);
        }
        if (sourceBucket == null) {
            throw new IllegalArgumentException(String.format("Source bucket '%s' does not exist.", sourcePath.getBucket()));
        }
        Bucket destBucket = null;
        try {
            destBucket = this.storage.get(destPath.getBucket(), new Storage.BucketGetOption[0]);
        }
        catch (StorageException e) {
            throw new RuntimeException(String.format("Unable to access destination bucket %s. ", destPath.getBucket()) + "Ensure you entered the correct bucket path.", e);
        }
        if (destBucket == null) {
            throw new IllegalArgumentException(String.format("Destination bucket '%s' does not exist. Please create it first.", destPath.getBucket()));
        }
        String baseDestName = destPath.getName();
        if (destPath.isBucket() || this.storage.get(BlobId.of((String)destPath.getBucket(), (String)baseDestName)) != null) {
            destinationBaseExists = true;
        } else {
            String modifiedName = baseDestName.endsWith("/") ? baseDestName.substring(0, baseDestName.length() - 1) : baseDestName + "/";
            destinationBaseExists = this.storage.get(BlobId.of((String)destPath.getBucket(), (String)modifiedName)) != null;
        }
        ArrayList copyList = new ArrayList();
        this.traverse(BlobId.of((String)sourcePath.getBucket(), (String)sourcePath.getName()), recursive, sourceBlob -> {
            Blob destBlob;
            BlobId destBlobID = StorageClient.resolve(sourcePath.getName(), sourceBlob.getBlobId().getName(), destPath, destinationBaseExists);
            if (!overwrite && (destBlob = this.storage.get(destBlobID)) != null && !destBlob.getName().endsWith("/") && destBlob.getSize() != 0L) {
                throw new IllegalArgumentException(String.format("%s already exists.", StorageClient.toPath(destBlobID)));
            }
            copyList.add(new BlobPair((Blob)sourceBlob, destBlobID));
        });
        LOG.debug("Found {} objects.", (Object)copyList.size());
        for (BlobPair blobPair : copyList) {
            consumer.accept(blobPair);
        }
    }

    @VisibleForTesting
    static BlobId resolve(String baseName, String sourceName, GCSPath dest, boolean destExists) {
        String relativePart = sourceName.substring(baseName.length());
        if (dest.isBucket()) {
            return BlobId.of((String)dest.getBucket(), (String)sourceName);
        }
        if (destExists || dest.getName().endsWith("/")) {
            int lastDirIndex = baseName.lastIndexOf("/");
            String lastPart = lastDirIndex > 0 ? baseName.substring(lastDirIndex) : baseName;
            return BlobId.of((String)dest.getBucket(), (String)StorageClient.append(StorageClient.append(dest.getName(), lastPart), relativePart));
        }
        return BlobId.of((String)dest.getBucket(), (String)StorageClient.append(dest.getName(), relativePart));
    }

    @VisibleForTesting
    static String append(String base, String part) {
        boolean baseEndsWithDivider = base.endsWith("/");
        boolean partStartWithDivider = part.startsWith("/");
        if (baseEndsWithDivider && partStartWithDivider) {
            return base.substring(0, base.length() - 1) + part;
        }
        if (!(baseEndsWithDivider || base.isEmpty() || partStartWithDivider || part.isEmpty())) {
            return base + "/" + part;
        }
        return base + part;
    }

    private void traverse(BlobId blobId, boolean recursive, Consumer<Blob> consumer) {
        Page blobList = this.storage.list(blobId.getBucket(), new Storage.BlobListOption[]{Storage.BlobListOption.currentDirectory(), Storage.BlobListOption.prefix((String)blobId.getName())});
        for (Blob blob : blobList.iterateAll()) {
            if (!blob.isDirectory()) {
                consumer.accept(blob);
                continue;
            }
            if (!recursive) continue;
            this.traverse(blob.getBlobId(), true, consumer);
        }
    }

    private static String toPath(BlobId blobId) {
        return String.format("gs://%s/%s", blobId.getBucket(), blobId.getName());
    }

    public static StorageClient create(String project, @Nullable String serviceAccount, Boolean isServiceAccountFilePath) throws IOException {
        StorageOptions.Builder builder = (StorageOptions.Builder)StorageOptions.newBuilder().setProjectId(project);
        if (serviceAccount != null) {
            builder.setCredentials((Credentials)GCPUtils.loadServiceAccountCredentials(serviceAccount, isServiceAccountFilePath));
        }
        Storage storage = (Storage)builder.build().getService();
        return new StorageClient(storage);
    }

    public static StorageClient create(GCPConnectorConfig config) throws IOException {
        return StorageClient.create(config.getProject(), config.getServiceAccount(), config.isServiceAccountFilePath());
    }

    private static class BlobPair {
        private final Blob sourceBlob;
        private final BlobId destination;

        private BlobPair(Blob sourceBlob, BlobId destination) {
            this.sourceBlob = sourceBlob;
            this.destination = destination;
        }

        private Blob copy() {
            LOG.debug("Copying {} to {}.", (Object)StorageClient.toPath(this.sourceBlob.getBlobId()), (Object)StorageClient.toPath(this.destination));
            CopyWriter copyWriter = this.sourceBlob.copyTo(this.destination, new Blob.BlobSourceOption[0]);
            Blob copied = copyWriter.getResult();
            LOG.debug("Successfully copied {} to {}.", (Object)StorageClient.toPath(this.sourceBlob.getBlobId()), (Object)StorageClient.toPath(this.destination));
            return copied;
        }

        private Blob move() {
            Blob moved = this.copy();
            LOG.debug("Deleting {}.", (Object)StorageClient.toPath(this.sourceBlob.getBlobId()));
            this.sourceBlob.delete(new Blob.BlobSourceOption[0]);
            LOG.debug("Successfully deleted {}.", (Object)StorageClient.toPath(this.sourceBlob.getBlobId()));
            return moved;
        }
    }
}

