/*
 * Decompiled with CFR 0.152.
 */
package io.cdap.plugin.gcp.gcs.actions;

import com.google.auth.Credentials;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.cloud.kms.v1.CryptoKeyName;
import com.google.cloud.storage.Bucket;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageException;
import io.cdap.cdap.api.annotation.Description;
import io.cdap.cdap.api.annotation.Macro;
import io.cdap.cdap.api.annotation.Name;
import io.cdap.cdap.api.annotation.Plugin;
import io.cdap.cdap.etl.api.FailureCollector;
import io.cdap.cdap.etl.api.PipelineConfigurer;
import io.cdap.cdap.etl.api.action.Action;
import io.cdap.cdap.etl.api.action.ActionContext;
import io.cdap.plugin.gcp.common.CmekUtils;
import io.cdap.plugin.gcp.common.GCPConfig;
import io.cdap.plugin.gcp.common.GCPUtils;
import io.cdap.plugin.gcp.gcs.GCSPath;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Plugin(type="action")
@Name(value="GCSBucketCreate")
@Description(value="Creates objects in a Google Cloud Storage bucket.")
public final class GCSBucketCreate
extends Action {
    private static final Logger LOG = LoggerFactory.getLogger(GCSBucketCreate.class);
    public static final String NAME = "GCSBucketCreate";
    private Config config;

    public void configurePipeline(PipelineConfigurer pipelineConfigurer) {
        this.config.validate(pipelineConfigurer.getStageConfigurer().getFailureCollector());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void run(ActionContext context) throws Exception {
        FileSystem fs;
        FailureCollector collector = context.getFailureCollector();
        this.config.validate(collector, context.getArguments().asMap());
        Configuration configuration = new Configuration();
        Boolean isServiceAccountFilePath = this.config.isServiceAccountFilePath();
        if (isServiceAccountFilePath == null) {
            collector.addFailure("Service account type is undefined.", "Must be `filePath` or `JSON`");
            collector.getOrThrowException();
            return;
        }
        String serviceAccount = this.config.getServiceAccount();
        GoogleCredentials credentials = serviceAccount == null ? null : GCPUtils.loadServiceAccountCredentials(serviceAccount, isServiceAccountFilePath);
        Map<String, String> map = GCPUtils.generateGCSAuthProperties(serviceAccount, this.config.getServiceAccountType());
        map.forEach((arg_0, arg_1) -> ((Configuration)configuration).set(arg_0, arg_1));
        configuration.set("fs.gs.impl", "com.google.cloud.hadoop.fs.gcs.GoogleHadoopFileSystem");
        configuration.set("fs.AbstractFileSystem.gs.impl", "com.google.cloud.hadoop.fs.gcs.GoogleHadoopFS");
        String projectId = this.config.getProject();
        configuration.set("fs.gs.project.id", projectId);
        configuration.set("fs.gs.path.encoding", "uri-path");
        configuration.setBoolean("fs.gs.impl.disable.cache", true);
        configuration.setBoolean("fs.gs.metadata.cache.enable", false);
        ArrayList<Path> undo = new ArrayList<Path>();
        ArrayList<GCSPath> undoBucket = new ArrayList<GCSPath>();
        ArrayList<Path> gcsPaths = new ArrayList<Path>();
        Storage storage = GCPUtils.getStorage(this.config.getProject(), (Credentials)credentials);
        boolean rollback = false;
        try {
            CryptoKeyName cmekKeyName = CmekUtils.getCmekKey(this.config.cmekKey, context.getArguments().asMap(), collector);
            collector.getOrThrowException();
            for (String path : this.config.getPaths()) {
                GCSPath bucketPath;
                GCSPath gcsPath = GCSPath.from(path);
                if (!gcsPath.equals(bucketPath = GCSPath.from("gs://" + gcsPath.getBucket()))) {
                    gcsPaths.add(new Path(gcsPath.getUri()));
                }
                Bucket bucket = null;
                try {
                    bucket = storage.get(gcsPath.getBucket(), new Storage.BucketGetOption[0]);
                }
                catch (StorageException e) {
                    throw new RuntimeException(String.format("Unable to access or create bucket %s. ", gcsPath.getBucket()) + "Ensure you entered the correct bucket path and have permissions for it.", e);
                }
                if (bucket == null) {
                    GCPUtils.createBucket(storage, gcsPath.getBucket(), this.config.location, cmekKeyName);
                    undoBucket.add(bucketPath);
                    continue;
                }
                if (!gcsPath.equals(bucketPath) || !this.config.failIfExists()) continue;
                rollback = true;
                throw new Exception(String.format("Path %s already exists", gcsPath));
            }
            for (Path gcsPath : gcsPaths) {
                try {
                    fs = gcsPath.getFileSystem(configuration);
                }
                catch (IOException e) {
                    rollback = true;
                    throw new Exception("Unable to get GCS filesystem handler. " + e.getMessage(), e);
                }
                if (!fs.exists(gcsPath)) {
                    try {
                        fs.mkdirs(gcsPath);
                        undo.add(gcsPath);
                        LOG.info(String.format("Created GCS directory '%s''", gcsPath.toUri().getPath()));
                        continue;
                    }
                    catch (IOException e) {
                        LOG.warn(String.format("Failed to create path '%s'", gcsPath));
                        rollback = true;
                        throw e;
                    }
                }
                if (!this.config.failIfExists()) continue;
                rollback = true;
                throw new Exception(String.format("Path %s already exists", gcsPath));
            }
        }
        finally {
            if (rollback) {
                context.getMetrics().gauge("gc.file.create.error", 1L);
                for (Path path : undo) {
                    try {
                        fs = path.getFileSystem(configuration);
                        fs.delete(path, true);
                    }
                    catch (IOException iOException) {}
                }
                for (GCSPath bucket : undoBucket) {
                    storage.delete(bucket.getBucket(), new Storage.BucketSourceOption[0]);
                }
            } else {
                context.getMetrics().gauge("gc.file.create.count", (long)gcsPaths.size());
            }
        }
    }

    public static final class Config
    extends GCPConfig {
        public static final String NAME_PATHS = "paths";
        public static final String NAME_LOCATION = "location";
        @Name(value="paths")
        @Description(value="Comma separated list of objects to be created.")
        @Macro
        private String paths;
        @Name(value="failIfExists")
        @Description(value="Fail if path exists.")
        @Macro
        private boolean failIfExists;
        @Name(value="location")
        @Macro
        @Nullable
        @Description(value="The location where the gcs buckets will get created. This value is ignored if the bucket already exists.")
        protected String location;
        @Name(value="cmekKey")
        @Macro
        @Nullable
        @Description(value="The GCP customer managed encryption key (CMEK) name used to encrypt data written to any bucket created by the plugin. If the bucket already exists, this is ignored. More information can be found at https://cloud.google.com/data-fusion/docs/how-to/customer-managed-encryption-keys")
        private String cmekKey;

        public Config(@Nullable String project, @Nullable String serviceAccountType, @Nullable String serviceFilePath, @Nullable String serviceAccountJson, @Nullable String paths, @Nullable String location, @Nullable String cmekKey) {
            this.serviceAccountType = serviceAccountType;
            this.serviceAccountJson = serviceAccountJson;
            this.serviceFilePath = serviceFilePath;
            this.project = project;
            this.paths = paths;
            this.location = location;
            this.cmekKey = cmekKey;
        }

        public List<String> getPaths() {
            return Arrays.stream(this.paths.split(",")).map(String::trim).collect(Collectors.toList());
        }

        public boolean failIfExists() {
            return this.failIfExists;
        }

        void validate(FailureCollector collector) {
            this.validate(collector, Collections.emptyMap());
        }

        void validate(FailureCollector collector, Map<String, String> arguments) {
            if (!this.containsMacro(NAME_PATHS)) {
                for (String path : this.getPaths()) {
                    try {
                        GCSPath.from(path);
                    }
                    catch (IllegalArgumentException e) {
                        collector.addFailure(e.getMessage(), null).withConfigElement(NAME_PATHS, path);
                    }
                }
            }
            if (!this.containsMacro("cmekKey")) {
                this.validateCmekKey(collector, arguments);
            }
            collector.getOrThrowException();
        }

        void validateCmekKey(FailureCollector failureCollector, Map<String, String> arguments) {
            CryptoKeyName cmekKeyName = CmekUtils.getCmekKey(this.cmekKey, arguments, failureCollector);
            if (cmekKeyName == null || this.containsMacro(NAME_PATHS) || this.containsMacro(NAME_LOCATION) || this.projectOrServiceAccountContainsMacro()) {
                return;
            }
            Storage storage = GCPUtils.getStorage(this.getProject(), this.getCredentials(failureCollector));
            if (storage == null) {
                return;
            }
            for (String path : this.getPaths()) {
                CmekUtils.validateCmekKeyAndBucketLocation(storage, GCSPath.from(path), cmekKeyName, this.location, failureCollector);
            }
        }

        public static Builder builder() {
            return new Builder();
        }

        public static class Builder {
            private String serviceAccountType;
            private String serviceFilePath;
            private String serviceAccountJson;
            private String project;
            private String gcsPaths;
            private String cmekKey;
            private String location;

            public Builder setProject(@Nullable String project) {
                this.project = project;
                return this;
            }

            public Builder setServiceAccountType(@Nullable String serviceAccountType) {
                this.serviceAccountType = serviceAccountType;
                return this;
            }

            public Builder setServiceFilePath(@Nullable String serviceFilePath) {
                this.serviceFilePath = serviceFilePath;
                return this;
            }

            public Builder setServiceAccountJson(@Nullable String serviceAccountJson) {
                this.serviceAccountJson = serviceAccountJson;
                return this;
            }

            public Builder setGcsPath(@Nullable String gcsPaths) {
                this.gcsPaths = gcsPaths;
                return this;
            }

            public Builder setCmekKey(@Nullable String cmekKey) {
                this.cmekKey = cmekKey;
                return this;
            }

            public Builder setLocation(@Nullable String location) {
                this.location = location;
                return this;
            }

            public Config build() {
                return new Config(this.project, this.serviceAccountType, this.serviceFilePath, this.serviceAccountJson, this.gcsPaths, this.location, this.cmekKey);
            }
        }
    }
}

