/*
 * Decompiled with CFR 0.152.
 */
package io.cdap.plugin.gcp.spanner.source;

import com.google.cloud.ByteArray;
import com.google.cloud.Date;
import com.google.cloud.Timestamp;
import com.google.cloud.spanner.ResultSet;
import com.google.cloud.spanner.Type;
import io.cdap.cdap.api.data.format.StructuredRecord;
import io.cdap.cdap.api.data.format.UnexpectedFormatException;
import io.cdap.cdap.api.data.schema.Schema;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

public class ResultSetToRecordTransformer {
    private final Schema schema;

    public ResultSetToRecordTransformer(Schema schema) {
        this.schema = schema;
    }

    public StructuredRecord transform(ResultSet resultSet) {
        StructuredRecord.Builder builder = StructuredRecord.builder((Schema)this.schema);
        List fields = this.schema.getFields();
        for (Schema.Field field : fields) {
            String fieldName = field.getName();
            Type columnType = resultSet.getColumnType(fieldName);
            Type.Code code = columnType.getCode();
            if (columnType == null || resultSet.isNull(fieldName) && code != Type.Code.ARRAY) continue;
            switch (columnType.getCode()) {
                case BOOL: {
                    builder.set(fieldName, (Object)resultSet.getBoolean(fieldName));
                    break;
                }
                case INT64: {
                    builder.set(fieldName, (Object)resultSet.getLong(fieldName));
                    break;
                }
                case FLOAT64: {
                    builder.set(fieldName, (Object)resultSet.getDouble(fieldName));
                    break;
                }
                case STRING: {
                    String value = resultSet.getString(fieldName);
                    this.validateDateTime(field.getSchema().isNullable() ? field.getSchema().getNonNullable() : field.getSchema(), fieldName, value);
                    builder.set(fieldName, (Object)value);
                    break;
                }
                case BYTES: {
                    ByteArray byteArray = resultSet.getBytes(fieldName);
                    builder.set(fieldName, (Object)byteArray.toByteArray());
                    break;
                }
                case DATE: {
                    Date spannerDate = resultSet.getDate(fieldName);
                    builder.setDate(fieldName, LocalDate.of(spannerDate.getYear(), spannerDate.getMonth(), spannerDate.getDayOfMonth()));
                    break;
                }
                case TIMESTAMP: {
                    Timestamp spannerTs = resultSet.getTimestamp(fieldName);
                    Instant instant = Instant.ofEpochSecond(spannerTs.getSeconds()).plusNanos(spannerTs.getNanos());
                    builder.setTimestamp(fieldName, ZonedDateTime.ofInstant(instant, ZoneId.ofOffset("UTC", ZoneOffset.UTC)));
                    break;
                }
                case ARRAY: {
                    builder.set(fieldName, (Object)this.transformArrayToList(resultSet, fieldName, field.getSchema(), columnType.getArrayElementType()).toArray());
                }
            }
        }
        return builder.build();
    }

    private void validateDateTime(Schema schema, String fieldName, String value) {
        if (schema.getLogicalType() == Schema.LogicalType.DATETIME) {
            try {
                LocalDateTime.parse(value);
            }
            catch (DateTimeParseException exception) {
                throw new UnexpectedFormatException(String.format("Datetime field '%s' with value '%s' is not in ISO-8601 format.", fieldName, value.toString()), (Throwable)exception);
            }
        }
    }

    private List<?> transformArrayToList(ResultSet resultSet, String fieldName, Schema fieldSchema, Type arrayElementType) {
        if (resultSet.isNull(fieldName)) {
            return Collections.emptyList();
        }
        switch (arrayElementType.getCode()) {
            case BOOL: {
                return resultSet.getBooleanList(fieldName);
            }
            case INT64: {
                return resultSet.getLongList(fieldName);
            }
            case FLOAT64: {
                return resultSet.getDoubleList(fieldName);
            }
            case STRING: {
                return resultSet.getStringList(fieldName);
            }
            case BYTES: {
                return resultSet.getBytesList(fieldName).stream().map(byteArray -> byteArray == null ? null : byteArray.toByteArray()).collect(Collectors.toList());
            }
            case DATE: {
                return resultSet.getDateList(fieldName).stream().map(this::convertDateToLong).collect(Collectors.toList());
            }
            case TIMESTAMP: {
                return resultSet.getTimestampList(fieldName).stream().map(timestamp -> this.convertTimestampToLong(fieldName, (Timestamp)timestamp, fieldSchema.getLogicalType())).collect(Collectors.toList());
            }
        }
        return Collections.emptyList();
    }

    private Integer convertDateToLong(Date date) {
        if (date == null) {
            return null;
        }
        return Math.toIntExact(LocalDate.of(date.getYear(), date.getMonth(), date.getDayOfMonth()).toEpochDay());
    }

    private Long convertTimestampToLong(String fieldName, Timestamp timestamp, Schema.LogicalType logicalType) {
        if (timestamp == null) {
            return null;
        }
        Instant instant = Instant.ofEpochSecond(timestamp.getSeconds()).plusNanos(timestamp.getNanos()).atZone(ZoneId.ofOffset("UTC", ZoneOffset.UTC)).toInstant();
        try {
            if (logicalType == Schema.LogicalType.TIMESTAMP_MILLIS) {
                long millis = TimeUnit.SECONDS.toMillis(instant.getEpochSecond());
                return Math.addExact(millis, TimeUnit.NANOSECONDS.toMillis(instant.getNano()));
            }
            long micros = TimeUnit.SECONDS.toMicros(instant.getEpochSecond());
            return Math.addExact(micros, TimeUnit.NANOSECONDS.toMicros(instant.getNano()));
        }
        catch (ArithmeticException e) {
            throw new UnexpectedFormatException(String.format("Field %s was set to a %s that is too large.", fieldName, logicalType.getToken()));
        }
    }
}

