/*
 * Decompiled with CFR 0.152.
 */
package org.apache.twill.yarn;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.AbstractIdleService;
import com.google.common.util.concurrent.Futures;
import java.io.IOException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import org.apache.hadoop.conf.Configuration;
import org.apache.twill.api.Configs;
import org.apache.twill.common.Threads;
import org.apache.twill.filesystem.Location;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class LocationCacheCleaner
extends AbstractIdleService {
    private static final Logger LOG = LoggerFactory.getLogger(LocationCacheCleaner.class);
    private final Location cacheBaseLocation;
    private final String sessionId;
    private final long expiry;
    private final long antiqueExpiry;
    private final Predicate<Location> cleanupPredicate;
    private final Set<PendingCleanup> pendingCleanups;
    private ScheduledExecutorService scheduler;

    LocationCacheCleaner(Configuration config, Location cacheBaseLocation, String sessionId, Predicate<Location> cleanupPredicate) {
        this.cacheBaseLocation = cacheBaseLocation;
        this.sessionId = sessionId;
        this.expiry = config.getLong("twill.location.cache.expiry.ms", Configs.Defaults.LOCATION_CACHE_EXPIRY_MS);
        this.antiqueExpiry = config.getLong("twill.location.cache.antique.expiry.ms", Configs.Defaults.LOCATION_CACHE_ANTIQUE_EXPIRY_MS);
        this.cleanupPredicate = cleanupPredicate;
        this.pendingCleanups = new HashSet<PendingCleanup>();
    }

    protected void startUp() {
        this.scheduler = Executors.newSingleThreadScheduledExecutor(Threads.createDaemonThreadFactory((String)"location-cache-cleanup"));
        this.scheduler.execute(new Runnable(){

            @Override
            public void run() {
                long currentTime = System.currentTimeMillis();
                LocationCacheCleaner.this.cleanup(currentTime);
                long scheduleDelay = LocationCacheCleaner.this.expiry / 2L;
                for (PendingCleanup pendingCleanup : LocationCacheCleaner.this.pendingCleanups) {
                    if (pendingCleanup.getExpireTime() - currentTime >= scheduleDelay) continue;
                    scheduleDelay = pendingCleanup.getExpireTime() - currentTime;
                }
                LocationCacheCleaner.this.scheduler.schedule(this, scheduleDelay, TimeUnit.MILLISECONDS);
            }
        });
    }

    protected void shutDown() {
        this.scheduler.shutdownNow();
    }

    @VisibleForTesting
    void forceCleanup(long currentTime) {
        Futures.getUnchecked(this.scheduler.submit(() -> this.cleanup(currentTime)));
    }

    @VisibleForTesting
    void cleanup(long currentTime) {
        Iterator<PendingCleanup> iterator = this.pendingCleanups.iterator();
        while (iterator.hasNext()) {
            PendingCleanup pendingCleanup = iterator.next();
            if (!this.cleanupPredicate.test(pendingCleanup.getLocation())) {
                iterator.remove();
                continue;
            }
            try {
                if (!pendingCleanup.deleteIfExpired(currentTime)) continue;
                iterator.remove();
            }
            catch (IOException e) {
                LOG.warn("Failed to delete {}", (Object)pendingCleanup.getLocation(), (Object)e);
            }
        }
        try {
            for (Location cacheDir : this.cacheBaseLocation.list()) {
                try {
                    boolean currentSession = cacheDir.getName().equals(this.sessionId);
                    List entries = cacheDir.list();
                    if (!currentSession && entries.isEmpty()) {
                        cacheDir.delete();
                        continue;
                    }
                    long expireTime = this.computeExpiry(currentTime, currentSession ? this.expiry : this.antiqueExpiry);
                    for (Location location : entries) {
                        if (!this.cleanupPredicate.test(location) || !this.pendingCleanups.add(new PendingCleanup(location, expireTime))) continue;
                        LOG.debug("Pending deletion of location {} with expiration time at {}", (Object)location, (Object)expireTime);
                    }
                }
                catch (IOException e) {
                    LOG.warn("Failed to list cache content from {}", (Object)cacheDir, (Object)e);
                }
            }
        }
        catch (IOException e) {
            LOG.warn("Failed to list cache directories from {}", (Object)this.cacheBaseLocation, (Object)e);
        }
    }

    private long computeExpiry(long currentTime, long increment) {
        if (Long.MAX_VALUE - increment < currentTime) {
            return Long.MAX_VALUE;
        }
        return currentTime + increment;
    }

    private static final class PendingCleanup {
        private final Location location;
        private final long expireTime;

        PendingCleanup(Location location, long expireTime) {
            this.location = location;
            this.expireTime = expireTime;
        }

        Location getLocation() {
            return this.location;
        }

        long getExpireTime() {
            return this.expireTime;
        }

        boolean deleteIfExpired(long currentTime) throws IOException {
            if (currentTime < this.expireTime) {
                return false;
            }
            if (this.location.delete()) {
                LOG.debug("Cached location removed {}", (Object)this.location);
            } else {
                LOG.debug("Failed to delete cached location {}", (Object)this.location);
            }
            return true;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            PendingCleanup that = (PendingCleanup)o;
            return this.location.equals(that.location);
        }

        public int hashCode() {
            return Objects.hash(this.location);
        }
    }
}

