/*
 * Decompiled with CFR 0.152.
 */
package io.cobrowse;

import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import com.upokecenter.cbor.CBORException;
import com.upokecenter.cbor.CBORObject;
import com.upokecenter.cbor.CBORType;
import io.cobrowse.Tls12OkHttpClient;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.WebSocket;
import okhttp3.WebSocketListener;
import okio.ByteString;

final class CBORSocket {
    @NonNull
    private final OkHttpClient client;
    @NonNull
    private final URLRequest request;
    @Nullable
    private WebSocket socket;
    private boolean connected = false;
    @Nullable
    private Timer reconnectTimer;
    private int attempts = 0;
    private double maxReconnectDelay = 30000.0;
    @NonNull
    private final Object socketLock = new Object();
    private long messageId = 0L;
    private long lastMessageAck = 0L;
    @NonNull
    private final HashMap<String, HashSet<MessageListener>> messageListeners = new HashMap();
    @NonNull
    private final HashSet<Runnable> connectListeners = new HashSet();
    @NonNull
    private final HashSet<ErrorListener> errorListeners = new HashSet();
    @NonNull
    private final Handler handler = new Handler(Looper.getMainLooper());
    @NonNull
    private final WebSocketListener socketListener = new WebSocketListener(){

        public void onMessage(@NonNull WebSocket webSocket, @NonNull ByteString bytes) {
            try {
                CBORObject cbor = CBORObject.DecodeFromBytes((byte[])bytes.toByteArray());
                CBORObject ack = cbor.get("ack");
                if (ack != null && ack.getType() == CBORType.Integer) {
                    CBORSocket.this.lastMessageAck = ack.AsInt64();
                    return;
                }
                CBORObject event = cbor.get("event");
                if (event == null || event.getType() != CBORType.TextString) {
                    Log.w((String)"CobrowseIO", (String)("Invalid event: " + event));
                    return;
                }
                Object data = CBORSocket.this.toNative(cbor.get("data"));
                HashMap message = new HashMap();
                if (data != null) {
                    if (!(data instanceof Map)) {
                        Log.w((String)"CobrowseIO", (String)("Invalid data type: " + data));
                        return;
                    }
                    for (Object key : ((Map)data).keySet()) {
                        Object value = ((Map)data).get(key);
                        if (!(key instanceof String)) continue;
                        message.put((String)key, value);
                    }
                }
                CBORSocket.this.handler.post(() -> {
                    HashSet listeners = (HashSet)CBORSocket.this.messageListeners.get(event.AsString());
                    if (listeners != null) {
                        for (MessageListener listener : listeners) {
                            listener.onMessage(message);
                        }
                    }
                });
            }
            catch (CBORException e) {
                Log.w((String)"CobrowseIO", (String)"Invalid CBOR");
            }
        }

        public void onOpen(@NonNull WebSocket webSocket, @NonNull Response response) {
            CBORSocket.this.connected = true;
            CBORSocket.this.attempts = 0;
            for (Runnable cb : new HashSet(CBORSocket.this.connectListeners)) {
                CBORSocket.this.handler.post(cb);
            }
        }

        public void onClosed(@NonNull WebSocket webSocket, int code, @NonNull String reason) {
            CBORSocket.this.connected = false;
            CBORSocket.this.reconnect();
        }

        public void onFailure(@NonNull WebSocket webSocket, @NonNull Throwable t, @Nullable Response response) {
            CBORSocket.this.connected = false;
            CBORSocket.this.reconnect();
            CBORSocket.this.handler.post(() -> {
                for (ErrorListener cb : new HashSet(CBORSocket.this.errorListeners)) {
                    cb.onError(t, response);
                }
            });
        }
    };

    CBORSocket(@NonNull URLRequest request) {
        this.request = request;
        this.client = Tls12OkHttpClient.reuse();
        this.socket = this.createSocket(request);
    }

    @NonNull
    private WebSocket createSocket(@NonNull URLRequest request) {
        this.lastMessageAck = 0L;
        this.messageId = 0L;
        return this.client.newWebSocket(request.create(), this.socketListener);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void reconnect() {
        Object object = this.socketLock;
        synchronized (object) {
            if (this.socket == null) {
                return;
            }
            if (this.reconnectTimer != null) {
                return;
            }
            this.reconnectTimer = new Timer();
            this.reconnectTimer.schedule(new TimerTask(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void run() {
                    Object object = CBORSocket.this.socketLock;
                    synchronized (object) {
                        CBORSocket.this.reconnectTimer = null;
                        if (CBORSocket.this.socket == null) {
                            Log.w((String)"CobrowseIO", (String)"Socket tried to reconnect after close");
                            return;
                        }
                        CBORSocket.this.socket = CBORSocket.this.createSocket(CBORSocket.this.request);
                    }
                }
            }, this.reconnectDelay(this.attempts));
            ++this.attempts;
        }
    }

    private long reconnectDelay(int attempts) {
        double random = Math.random();
        double randomFactor = 1.0 + 0.5 * random;
        double delay = Math.pow(300 * attempts, 1.2) * randomFactor;
        return (long)Math.min(delay, this.maxReconnectDelay);
    }

    boolean isConnected() {
        return this.connected;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void disconnect() {
        Object object = this.socketLock;
        synchronized (object) {
            this.messageListeners.clear();
            this.errorListeners.clear();
            this.connectListeners.clear();
            if (this.socket != null) {
                this.socket.close(1001, "client disconnect");
            }
            if (this.reconnectTimer != null) {
                this.reconnectTimer.cancel();
                this.reconnectTimer.purge();
                this.reconnectTimer = null;
            }
            this.socket = null;
        }
    }

    void send(@NonNull String event, @NonNull Map<String, Object> data) throws IOException {
        if (!this.isConnected()) {
            Log.w((String)"CobrowseIO", (String)String.format("Cannot send a socket message when not connected. Dropping \"%s\" event", event));
            return;
        }
        WebSocket socket = this.socket;
        if (socket == null) {
            Log.w((String)"CobrowseIO", (String)String.format("Cannot send a socket message when socket is not ready. Dropping \"%s\" event", event));
            return;
        }
        long newMessageId = this.messageId++;
        HashMap<String, Object> message = new HashMap<String, Object>();
        message.put("event", event);
        message.put("data", data);
        message.put("id", newMessageId);
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        CBORObject.Write((CBORObject)CBORObject.FromObject(message), (OutputStream)output);
        socket.send(ByteString.of((byte[])output.toByteArray()));
    }

    long queueSize() {
        if (!this.isConnected()) {
            Log.w((String)"CobrowseIO", (String)"Requested the queue size but socket is not connected");
            return Long.MAX_VALUE;
        }
        WebSocket socket = this.socket;
        if (socket == null) {
            Log.w((String)"CobrowseIO", (String)"Requested the queue size but socket is null");
            return Long.MAX_VALUE;
        }
        return socket.queueSize();
    }

    long messageLag() {
        long lastMessageAck = this.lastMessageAck;
        long messageId = this.messageId;
        if (lastMessageAck == 0L) {
            return 0L;
        }
        return messageId - lastMessageAck;
    }

    public void setMaxReconnectDelay(long millis) {
        this.maxReconnectDelay = millis;
    }

    void onMessage(@NonNull String event, @NonNull MessageListener listener) {
        HashSet<MessageListener> existing = this.messageListeners.get(event);
        if (existing == null) {
            existing = new HashSet();
            this.messageListeners.put(event, existing);
        }
        existing.add(listener);
    }

    void onConnect(@NonNull Runnable cb) {
        this.connectListeners.add(cb);
    }

    void onError(@NonNull ErrorListener cb) {
        this.errorListeners.add(cb);
    }

    @Nullable
    private Object toNative(@Nullable CBORObject cbor) {
        if (cbor == null) {
            return null;
        }
        switch (cbor.getType()) {
            case Boolean: {
                return cbor.AsBoolean();
            }
            case SimpleValue: {
                return cbor.getSimpleValue();
            }
            case ByteString: {
                return cbor.GetByteString();
            }
            case TextString: {
                return cbor.AsString();
            }
            case Array: {
                ArrayList<Object> out = new ArrayList<Object>();
                for (CBORObject obj : cbor.getValues()) {
                    out.add(this.toNative(obj));
                }
                return out;
            }
            case Map: {
                HashMap<String, Object> out = new HashMap<String, Object>();
                for (CBORObject key : cbor.getKeys()) {
                    out.put(key.AsString(), this.toNative(cbor.get(key)));
                }
                return out;
            }
            case Number: 
            case Integer: {
                if (cbor.CanTruncatedIntFitInInt32()) {
                    return cbor.AsInt32();
                }
                return cbor.AsInt64();
            }
            case FloatingPoint: {
                if (cbor.CanFitInSingle()) {
                    return Float.valueOf(cbor.AsSingle());
                }
                return cbor.AsDouble();
            }
        }
        return null;
    }

    public static interface URLRequest {
        @NonNull
        public Request create();
    }

    public static interface ErrorListener {
        public void onError(@NonNull Throwable var1, @Nullable Response var2);
    }

    public static interface MessageListener {
        public void onMessage(@NonNull Map<String, Object> var1);
    }
}

