/*
 * Decompiled with CFR 0.152.
 */
package io.cobrowse;

import android.graphics.Bitmap;
import android.graphics.Point;
import android.media.Image;
import android.media.MediaCodec;
import android.media.MediaCodecInfo;
import android.media.MediaCodecList;
import android.media.MediaFormat;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.HandlerThread;
import android.util.Log;
import android.util.Range;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import io.cobrowse.BitmapScaleConverter;
import io.cobrowse.Encoder;
import io.cobrowse.Frame;
import io.cobrowse.PixelFormatConverter;
import io.cobrowse.SystemProperties;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.Queue;

@RequiresApi(api=21)
final class H264Encoder
extends Encoder {
    @RequiresApi(api=19)
    static final String MIME_TYPE = "video/avc";
    private static final int BITRATE = 524288;
    private static final int FRAME_RATE = 10;
    private static final int IFRAME_INTERVAL = 10;
    private static final int INTRA_REFRESH_PERIOD = 120;
    private static final Point FRAME_SIZE_CONSTRAINTS = new Point(16, 16);
    @NonNull
    private final Iterable<String> boardNames;
    @Nullable
    private EncoderState encoder;
    private final Object mediaCodecLock;
    private final Handler mediaCodecHandler;
    private final HandlerThread mediaCodecHandlerThread;
    private long generateIndex;
    private byte[] configByte;
    private int equalFrameCount;
    @NonNull
    private final BitmapScaleConverter scaleConverter;
    private Frame lastFrame;
    private int[] lastFramePixelsData;
    private byte[] lastFrameConvertedPixelsData;
    private Bitmap unprocessedFrame;
    private final Queue<Integer> unprocessedInputBuffers = new LinkedList<Integer>();
    private boolean isResetRequested;

    H264Encoder(Encoder.Delegate delegate) {
        this(delegate, null);
    }

    H264Encoder(Encoder.Delegate delegate, Handler handler) {
        super(delegate, handler);
        this.mediaCodecHandlerThread = new HandlerThread("H264Encoder");
        this.mediaCodecHandlerThread.start();
        this.mediaCodecHandler = new Handler(this.mediaCodecHandlerThread.getLooper());
        this.mediaCodecLock = new Object();
        this.scaleConverter = new BitmapScaleConverter();
        this.boardNames = new ArrayList<String>(){
            {
                this.add(SystemProperties.read("ro.board.platform"));
                this.add(Build.BOARD);
            }
        };
    }

    @RequiresApi(api=19)
    public static boolean isSupported() {
        if (Build.VERSION.SDK_INT < 21) {
            return false;
        }
        ArrayList<MediaCodecInfo> codecInfo = H264Encoder.findMediaCodecs(MIME_TYPE);
        if (codecInfo.isEmpty()) {
            Log.e((String)"CobrowseIO", (String)"Cannot find a codec for video/avc");
            return false;
        }
        return true;
    }

    @Override
    @NonNull
    public String mimeType() {
        return MIME_TYPE;
    }

    @Override
    public void encode(@NonNull Frame frame) {
        if (this.isResetRequested || this.lastFrame != null && !frame.sameSizeAs(this.lastFrame)) {
            if (this.lastFrame != null && !frame.sameSizeAs(this.lastFrame)) {
                Log.d((String)"CobrowseIO", (String)"Received a bitmap of a different size, need to destroy the codec");
                this.destroyCodec();
            }
            this.equalFrameCount = 0;
            this.isResetRequested = false;
        } else if (this.lastFrame != null && frame.sameSizeAs(this.lastFrame) && !frame.isDirty()) {
            ++this.equalFrameCount;
            if (this.equalFrameCount > 5) {
                return;
            }
        } else {
            this.equalFrameCount = 0;
        }
        Bitmap scaled = this.scaleConverter.downscale(frame.getBitmap(), frame.getScale(), FRAME_SIZE_CONSTRAINTS);
        try {
            this.lastFrame = frame;
            this.unprocessedFrame = this.prepareNextFrame(scaled);
            this.encodeNextFrame();
        }
        catch (Exception e) {
            this.lastFrame = null;
            this.unprocessedFrame = null;
            this.raiseEncoderError(new Error(e));
        }
    }

    @Override
    public int maximumFramesInFlight() {
        return 60;
    }

    @Override
    public void reset() {
        Log.d((String)"CobrowseIO", (String)"Received a 'reset' request");
        if (this.isMediaCodecRunning()) {
            Bundle syncFrame = new Bundle();
            syncFrame.putInt("request-sync", 0);
            this.encoder.mediaCodec.setParameters(syncFrame);
        }
        this.isResetRequested = true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void destroyCodec() {
        Object object = this.mediaCodecLock;
        synchronized (object) {
            if (this.isMediaCodecRunning()) {
                this.encoder.destroy();
                this.encoder = null;
            }
            this.scaleConverter.clear();
            this.lastFrameConvertedPixelsData = null;
            this.lastFramePixelsData = null;
            this.lastFrame = null;
            this.unprocessedFrame = null;
            this.unprocessedInputBuffers.clear();
            this.isResetRequested = false;
        }
    }

    @Override
    public void destroy() {
        this.destroyCodec();
        this.mediaCodecHandlerThread.quitSafely();
        super.destroy();
    }

    @NonNull
    private Bitmap prepareNextFrame(@NonNull Bitmap input) {
        if (this.isMediaCodecRunning()) {
            return this.downscaleForCodec(this.encoder.mediaCodec.getInputFormat(), input);
        }
        for (MediaCodecInfo codecInfo : H264Encoder.findMediaCodecs(MIME_TYPE)) {
            try {
                MediaCodecInfo.CodecCapabilities capabilities = codecInfo.getCapabilitiesForType(MIME_TYPE);
                input = this.downscaleForCodec(capabilities, input);
                int colorFormat = this.findColorFormat(capabilities);
                Log.d((String)"CobrowseIO", (String)String.format("Color format 0x%x will be used", colorFormat));
                MediaFormat format = this.createMediaFormat(capabilities, input.getWidth(), input.getHeight(), colorFormat);
                this.encoder = new EncoderState(this.createMediaCodec(codecInfo, format), colorFormat);
                return input;
            }
            catch (Exception e) {
                Log.e((String)"CobrowseIO", (String)String.format("Failed to initialize %s codec", codecInfo.getName()), (Throwable)e);
            }
        }
        throw new UnsupportedOperationException("Failed to initialise an encoder: there is none that supports H264");
    }

    private Bitmap downscaleForCodec(@NonNull MediaFormat mediaFormat, @NonNull Bitmap input) {
        int expectedWidth = mediaFormat.getInteger("width");
        int expectedHeight = mediaFormat.getInteger("height");
        if (input.getWidth() != expectedWidth || input.getHeight() != expectedHeight) {
            input = this.scaleConverter.createScaledBitmap(input, expectedWidth, expectedHeight);
        }
        return input;
    }

    private Bitmap downscaleForCodec(@NonNull MediaCodecInfo.CodecCapabilities capabilities, @NonNull Bitmap input) {
        if (H264Encoder.isSizeSupported(capabilities, input.getWidth(), input.getHeight())) {
            return input;
        }
        int maxWidth = (Integer)capabilities.getVideoCapabilities().getSupportedWidths().getUpper();
        int maxHeight = (Integer)capabilities.getVideoCapabilities().getSupportedHeights().getUpper();
        float scale = Math.min((float)maxWidth / (float)input.getWidth(), (float)maxHeight / (float)input.getHeight());
        return this.scaleConverter.downscale(input, scale, FRAME_SIZE_CONSTRAINTS);
    }

    @NonNull
    private MediaFormat createMediaFormat(@NonNull MediaCodecInfo.CodecCapabilities capabilities, int width, int height, int colorFormat) {
        boolean supported;
        Integer complexity;
        boolean supported2;
        if (!H264Encoder.isSizeSupported(capabilities, width, height)) {
            throw new UnsupportedOperationException(String.format("Size %d:%d is not supported by codec", width, height));
        }
        MediaFormat mediaFormat = MediaFormat.createVideoFormat((String)MIME_TYPE, (int)width, (int)height);
        mediaFormat.setInteger("bitrate", 524288);
        mediaFormat.setInteger("frame-rate", 10);
        mediaFormat.setInteger("color-format", colorFormat);
        mediaFormat.setInteger("i-frame-interval", 10);
        if (Build.VERSION.SDK_INT >= 24 && (supported2 = capabilities.isFeatureSupported("intra-refresh")) && !this.isUnreliableIntraRefresh()) {
            mediaFormat.setInteger("intra-refresh-period", 120);
            mediaFormat.setInteger("i-frame-interval", -1);
        }
        if (!this.isUnreliableProfileAndLevel()) {
            MediaCodecInfo.CodecProfileLevel profileLevel = H264Encoder.findProfileAndLevel(capabilities);
            if (profileLevel != null) {
                mediaFormat.setInteger("profile", profileLevel.profile);
                mediaFormat.setInteger("level", profileLevel.level);
            } else {
                Log.w((String)"CobrowseIO", (String)"Cannot find any profile and level, keeping profile and level defaults");
            }
        }
        if (Build.VERSION.SDK_INT >= 28) {
            int quality = H264Encoder.findDesiredQuality(capabilities);
            if (quality > 0) {
                mediaFormat.setInteger("quality", quality);
            } else {
                Log.w((String)"CobrowseIO", (String)"Cannot get supported quality range, keeping the quality value default");
            }
        }
        if (Build.VERSION.SDK_INT >= 23) {
            mediaFormat.setInteger("priority", 0);
        }
        if ((complexity = H264Encoder.findDesiredComplexity(capabilities)) != null) {
            mediaFormat.setInteger("complexity", complexity.intValue());
        }
        if (Build.VERSION.SDK_INT >= 26) {
            mediaFormat.setInteger("latency", 0);
        }
        if (Build.VERSION.SDK_INT >= 30 && (supported = capabilities.isFeatureSupported("low-latency"))) {
            mediaFormat.setInteger("low-latency", 1);
        }
        return mediaFormat;
    }

    @NonNull
    private MediaCodec createMediaCodec(@NonNull MediaCodecInfo codecInfo, @NonNull MediaFormat mediaFormat) {
        MediaCodec encoder;
        try {
            encoder = MediaCodec.createByCodecName((String)codecInfo.getName());
        }
        catch (IOException e) {
            throw new UnsupportedOperationException("Cannot create a MediaCodec instance", e);
        }
        encoder.setCallback(this.getMediaCodecCallback());
        try {
            encoder.configure(mediaFormat, null, null, 1);
            encoder.start();
        }
        catch (IllegalStateException e) {
            encoder.release();
            throw new IllegalStateException("Cannot configure MediaCodec instance", e);
        }
        return encoder;
    }

    @RequiresApi(api=28)
    private static int findDesiredQuality(@NonNull MediaCodecInfo.CodecCapabilities capabilities) {
        Range qr = capabilities.getEncoderCapabilities().getQualityRange();
        if (qr != null && !((Integer)qr.getUpper()).equals(qr.getLower())) {
            return ((Integer)qr.getUpper() + (Integer)qr.getLower()) / 2;
        }
        return 0;
    }

    private static Integer findDesiredComplexity(@NonNull MediaCodecInfo.CodecCapabilities capabilities) {
        Range qr = capabilities.getEncoderCapabilities().getComplexityRange();
        return qr != null ? (Integer)qr.getLower() : null;
    }

    @NonNull
    private static ArrayList<MediaCodecInfo> findMediaCodecs(@NonNull String mimeType) {
        ArrayList<MediaCodecInfo> rvalue = new ArrayList<MediaCodecInfo>();
        int numCodecs = MediaCodecList.getCodecCount();
        for (int i = 0; i < numCodecs; ++i) {
            String[] types;
            MediaCodecInfo codecInfo = MediaCodecList.getCodecInfoAt((int)i);
            if (!codecInfo.isEncoder()) continue;
            for (String type : types = codecInfo.getSupportedTypes()) {
                if (!type.equalsIgnoreCase(mimeType) || H264Encoder.isBrokenCodec(codecInfo)) continue;
                rvalue.add(codecInfo);
            }
        }
        return rvalue;
    }

    private static boolean isBrokenCodec(@NonNull MediaCodecInfo codecInfo) {
        if (Build.VERSION.SDK_INT <= 22 && (codecInfo.getName().startsWith("OMX.google.") || codecInfo.getName().startsWith("c2.android."))) {
            return true;
        }
        if (Build.VERSION.SDK_INT <= 22 && codecInfo.getName().equals("OMX.Freescale.std.video_encoder.avc.hw-based")) {
            return true;
        }
        return Build.VERSION.SDK_INT == 33 && codecInfo.getName().equals("c2.v4l2.avc.encoder");
    }

    @Nullable
    private static MediaCodecInfo.CodecProfileLevel findProfileAndLevel(@NonNull MediaCodecInfo.CodecCapabilities capabilities) {
        if (capabilities.profileLevels == null || capabilities.profileLevels.length == 0) {
            return null;
        }
        for (Integer recognizedProfile : H264Encoder.getRecognizedProfiles()) {
            for (MediaCodecInfo.CodecProfileLevel profileLevel : capabilities.profileLevels) {
                if (recognizedProfile != profileLevel.profile) continue;
                return profileLevel;
            }
        }
        return null;
    }

    private boolean isUnreliableProfileAndLevel() {
        for (String soc : this.boardNames) {
            if (soc == null || soc.isEmpty()) continue;
            if (soc.equalsIgnoreCase("msm8953")) {
                return true;
            }
            if (!soc.equalsIgnoreCase("sdm660")) continue;
            return Build.VERSION.SDK_INT < 29;
        }
        return false;
    }

    private static boolean isSizeSupported(@NonNull MediaCodecInfo.CodecCapabilities capabilities, int width, int height) {
        return capabilities.getVideoCapabilities().isSizeSupported(width, height);
    }

    private int findColorFormat(@NonNull MediaCodecInfo.CodecCapabilities capabilities) {
        for (Integer recognizedFormat : RecognizedColorFormats.get()) {
            for (int colorFormat : capabilities.colorFormats) {
                if (recognizedFormat != colorFormat) continue;
                return colorFormat;
            }
        }
        Log.w((String)"CobrowseIO", (String)"Cannot find any color format for codec, falling back to COLOR_FormatYUV420SemiPlanar");
        return 21;
    }

    private static boolean isYUV420FlexibleFormat(int colorFormat) {
        return colorFormat == 2135033992;
    }

    private static boolean isI420PlanarFormat(int colorFormat) {
        switch (colorFormat) {
            case 19: 
            case 20: {
                return true;
            }
        }
        return false;
    }

    private static boolean isARGB8888PlanarFormat(int colorFormat) {
        if (Build.VERSION.SDK_INT >= 23) {
            return colorFormat == 2130747392;
        }
        return false;
    }

    private static ArrayList<Integer> getRecognizedProfiles() {
        ArrayList<Integer> rvalue = new ArrayList<Integer>();
        rvalue.add(8);
        rvalue.add(2);
        rvalue.add(1);
        return rvalue;
    }

    @RequiresApi(api=24)
    private boolean isUnreliableIntraRefresh() {
        for (String soc : this.boardNames) {
            if (soc == null || soc.isEmpty()) continue;
            if (soc.equalsIgnoreCase("msm8996")) {
                return Build.VERSION.SDK_INT == 26;
            }
            if (!soc.equalsIgnoreCase("msm8953")) continue;
            return Build.VERSION.SDK_INT == 28;
        }
        return false;
    }

    private void encodeNextFrame() {
        this.mediaCodecHandler.post(this::encodeNextFrameAsynchronously);
    }

    private void encodeNextFrameAsynchronously() {
        Bitmap frame = this.unprocessedFrame;
        if (this.unprocessedInputBuffers.isEmpty() || frame == null) {
            return;
        }
        this.unprocessedFrame = null;
        Integer bufferIndex = this.unprocessedInputBuffers.remove();
        this.writeFrameToBuffer(frame, bufferIndex);
    }

    @NonNull
    private MediaCodec.Callback getMediaCodecCallback() {
        return new MediaCodec.Callback(){

            public void onInputBufferAvailable(@NonNull MediaCodec codec, int index) {
                H264Encoder.this.mediaCodecHandler.post(() -> {
                    H264Encoder.this.unprocessedInputBuffers.add(index);
                    H264Encoder.this.encodeNextFrameAsynchronously();
                });
            }

            public void onOutputBufferAvailable(@NonNull MediaCodec codec, int index, @NonNull MediaCodec.BufferInfo info) {
                H264Encoder.this.mediaCodecHandler.post(() -> H264Encoder.this.readFrameFromBuffer(index, info));
            }

            public void onError(@NonNull MediaCodec codec, @NonNull MediaCodec.CodecException e) {
                Log.e((String)"CobrowseIO", (String)"onError", (Throwable)e);
                H264Encoder.this.mediaCodecHandler.post(() -> H264Encoder.this.raiseEncoderError(new Error((Throwable)e)));
            }

            public void onOutputFormatChanged(@NonNull MediaCodec codec, @NonNull MediaFormat format) {
                Log.d((String)"CobrowseIO", (String)("onOutputFormatChanged" + format));
                H264Encoder.this.mediaCodecHandler.post(() -> H264Encoder.this.readNewOutputFormat(format));
            }
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeFrameToBuffer(@NonNull Bitmap frame, int index) {
        Object object = this.mediaCodecLock;
        synchronized (object) {
            int inputBufferSize;
            if (!this.isMediaCodecRunning()) {
                return;
            }
            int width = frame.getWidth();
            int height = frame.getHeight();
            if (this.lastFramePixelsData == null) {
                this.lastFramePixelsData = new int[width * height];
            }
            frame.getPixels(this.lastFramePixelsData, 0, width, 0, 0, width, height);
            ByteBuffer inputBuffer = this.encoder.mediaCodec.getInputBuffer(index);
            if (inputBuffer == null) {
                Log.e((String)"CobrowseIO", (String)("MediaCodec failed to return an input buffer #" + index));
                return;
            }
            if (H264Encoder.isYUV420FlexibleFormat(this.encoder.colorFormat)) {
                if (this.lastFrameConvertedPixelsData == null) {
                    this.lastFrameConvertedPixelsData = new byte[width * height * 3 / 2];
                }
                PixelFormatConverter.argb2i420(this.lastFramePixelsData, this.lastFrameConvertedPixelsData, width, height);
                inputBufferSize = inputBuffer.capacity();
                Image inputImage = this.encoder.mediaCodec.getInputImage(index);
                if (inputImage == null) {
                    Log.e((String)"CobrowseIO", (String)("MediaCodec failed to return an input image #" + index));
                    return;
                }
                if (inputImage.getFormat() != 35) {
                    Log.e((String)"CobrowseIO", (String)"MediaCodec InputImage is not YUV_420_888!");
                    return;
                }
                H264Encoder.writeYuvToImage(this.lastFrameConvertedPixelsData, inputImage);
            } else {
                if (H264Encoder.isARGB8888PlanarFormat(this.encoder.colorFormat)) {
                    if (this.lastFrameConvertedPixelsData == null) {
                        this.lastFrameConvertedPixelsData = new byte[width * height * 4];
                    }
                    PixelFormatConverter.argb2rgba(this.lastFramePixelsData, this.lastFrameConvertedPixelsData, width, height);
                } else if (H264Encoder.isI420PlanarFormat(this.encoder.colorFormat)) {
                    if (this.lastFrameConvertedPixelsData == null) {
                        this.lastFrameConvertedPixelsData = new byte[width * height * 3 / 2];
                    }
                    PixelFormatConverter.argb2i420(this.lastFramePixelsData, this.lastFrameConvertedPixelsData, width, height);
                } else {
                    if (this.lastFrameConvertedPixelsData == null) {
                        this.lastFrameConvertedPixelsData = new byte[width * height * 3 / 2];
                    }
                    PixelFormatConverter.argb2nv12(this.lastFramePixelsData, this.lastFrameConvertedPixelsData, width, height);
                }
                inputBufferSize = this.lastFrameConvertedPixelsData.length;
                inputBuffer.clear();
                inputBuffer.put(this.lastFrameConvertedPixelsData);
            }
            this.encoder.mediaCodec.queueInputBuffer(index, 0, inputBufferSize, H264Encoder.computePresentationTime(this.generateIndex), 1);
            ++this.generateIndex;
        }
    }

    private static void writeYuvToImage(@NonNull byte[] data, @NonNull Image dstImage) {
        ByteBuffer srcBuffer = ByteBuffer.wrap(data);
        int width = dstImage.getWidth();
        int height = dstImage.getHeight();
        Image.Plane[] planes = dstImage.getPlanes();
        for (int n = 0; n < planes.length; ++n) {
            ByteBuffer dstBuffer = planes[n].getBuffer();
            int colStride = planes[n].getPixelStride();
            int srcPlanePos = 0;
            int div = 1;
            if (n > 0) {
                div = 2;
                srcPlanePos = width * height * (n + 3) / 4;
            }
            for (int i = 0; i < height / div; ++i) {
                srcBuffer.position(srcPlanePos + i * width / div);
                dstBuffer.position(i * planes[n].getRowStride());
                for (int j = 0; j < width / div; ++j) {
                    dstBuffer.put(srcBuffer.get());
                    if (colStride <= 1 || j == width / div - 1) continue;
                    dstBuffer.position(dstBuffer.position() + colStride - 1);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void readFrameFromBuffer(int index, @NonNull MediaCodec.BufferInfo info) {
        Object object = this.mediaCodecLock;
        synchronized (object) {
            if (!this.isMediaCodecRunning()) {
                return;
            }
            ByteBuffer outputBuffer = this.encoder.mediaCodec.getOutputBuffer(index);
            if (outputBuffer == null) {
                return;
            }
            byte[] buffer = new byte[info.size];
            outputBuffer.get(buffer, 0, buffer.length);
            byte[] h264Data = null;
            Error error = null;
            if ((info.flags & 2) != 0) {
                this.configByte = buffer;
                int nalType = this.configByte[4] & 0x1F;
                if (nalType == 7) {
                    this.configByte[6] = (byte)(this.configByte[6] | 0x10);
                }
            } else if ((info.flags & 1) != 0) {
                if (this.configByte == null) {
                    Log.e((String)"CobrowseIO", (String)"Encoder produced a key frame without SPS and PPS data");
                    error = new Error("Encoder produced a key frame without SPS and PPS data");
                } else {
                    h264Data = new byte[buffer.length + this.configByte.length];
                    System.arraycopy(this.configByte, 0, h264Data, 0, this.configByte.length);
                    System.arraycopy(buffer, 0, h264Data, this.configByte.length, buffer.length);
                }
            } else {
                h264Data = buffer;
            }
            this.encoder.mediaCodec.releaseOutputBuffer(index, false);
            if (h264Data != null) {
                this.raiseEncodedFrameData(h264Data);
            } else if (error != null) {
                this.raiseEncoderError(error);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void readNewOutputFormat(@NonNull MediaFormat format) {
        Object object = this.mediaCodecLock;
        synchronized (object) {
            if (!this.isMediaCodecRunning()) {
                return;
            }
            ByteBuffer spsBuffer = format.getByteBuffer("csd-0");
            ByteBuffer ppsBuffer = format.getByteBuffer("csd-1");
            if (spsBuffer == null || ppsBuffer == null) {
                return;
            }
            byte[] sps = spsBuffer.array();
            byte[] pps = ppsBuffer.array();
            this.configByte = new byte[sps.length + pps.length];
            System.arraycopy(sps, 0, this.configByte, 0, sps.length);
            System.arraycopy(pps, 0, this.configByte, sps.length, pps.length);
        }
    }

    private boolean isMediaCodecRunning() {
        return this.encoder != null;
    }

    private static long computePresentationTime(long frameIndex) {
        return 132L + frameIndex * 1000000L / 10L;
    }

    private static class EncoderState {
        @NonNull
        private final MediaCodec mediaCodec;
        private final int colorFormat;

        EncoderState(@NonNull MediaCodec codec, int colorFormat) {
            this.mediaCodec = codec;
            this.colorFormat = colorFormat;
        }

        private void destroy() {
            try {
                this.mediaCodec.stop();
                this.mediaCodec.release();
            }
            catch (IllegalStateException e) {
                Log.e((String)"CobrowseIO", (String)"Trying to stop the codec that is in Released state already", (Throwable)e);
            }
        }
    }

    private static class RecognizedColorFormats {
        private RecognizedColorFormats() {
        }

        private static int COLOR_FormatYUV420SemiPlanar() {
            return 21;
        }

        private static int COLOR_TI_FormatYUV420PackedSemiPlanar() {
            return 2130706688;
        }

        private static int COLOR_FormatYUV420Flexible() {
            return 2135033992;
        }

        private static int COLOR_FormatYUV420Planar() {
            return 19;
        }

        @RequiresApi(api=23)
        private static int COLOR_Format32bitABGR8888() {
            return 2130747392;
        }

        private static int COLOR_FormatYUV420PackedPlanar() {
            return 20;
        }

        private static int COLOR_FormatYUV420PackedSemiPlanar() {
            return 39;
        }

        @NonNull
        private static ArrayList<Integer> get() {
            ArrayList<Integer> rvalue = new ArrayList<Integer>();
            if (Build.VERSION.SDK_INT < 31) {
                rvalue.add(RecognizedColorFormats.COLOR_FormatYUV420SemiPlanar());
            }
            rvalue.add(RecognizedColorFormats.COLOR_TI_FormatYUV420PackedSemiPlanar());
            if (Build.VERSION.SDK_INT != 33) {
                rvalue.add(RecognizedColorFormats.COLOR_FormatYUV420Flexible());
                rvalue.add(RecognizedColorFormats.COLOR_FormatYUV420Planar());
            } else {
                rvalue.add(RecognizedColorFormats.COLOR_FormatYUV420Planar());
                rvalue.add(RecognizedColorFormats.COLOR_FormatYUV420Flexible());
            }
            if (Build.VERSION.SDK_INT >= 23) {
                rvalue.add(RecognizedColorFormats.COLOR_Format32bitABGR8888());
            }
            rvalue.add(RecognizedColorFormats.COLOR_FormatYUV420PackedPlanar());
            return rvalue;
        }
    }
}

