/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tuweni.ssz;

import java.math.BigInteger;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;
import java.util.function.LongFunction;
import java.util.function.Supplier;
import org.apache.tuweni.bytes.Bytes;
import org.apache.tuweni.ssz.EndOfSSZException;
import org.apache.tuweni.ssz.InvalidSSZTypeException;
import org.apache.tuweni.ssz.SSZReadable;
import org.apache.tuweni.ssz.SSZReader;
import org.apache.tuweni.units.bigints.UInt256;
import org.apache.tuweni.units.bigints.UInt384;

final class BytesSSZReader
implements SSZReader {
    private final Bytes content;
    private int index = 0;

    BytesSSZReader(Bytes content) {
        this.content = content;
    }

    @Override
    public Bytes readBytes(int limit) {
        int size;
        int byteLength = 4;
        this.ensureBytes(byteLength, () -> "SSZ encoded data is not a byte array");
        try {
            size = this.content.getInt(this.index, ByteOrder.LITTLE_ENDIAN);
        }
        catch (IndexOutOfBoundsException e) {
            throw new EndOfSSZException();
        }
        if (size < 0 || size > limit) {
            throw new InvalidSSZTypeException("length of bytes would exceed limit");
        }
        this.index += 4;
        if (this.content.size() - this.index - size < 0) {
            throw new InvalidSSZTypeException("SSZ encoded data has insufficient bytes for decoded byte array length");
        }
        return this.consumeBytes(size);
    }

    @Override
    public Bytes readFixedBytes(int byteLength, int limit) {
        this.ensureBytes(byteLength, () -> "SSZ encoded data is not a fixed-length byte array");
        return this.consumeBytes(byteLength);
    }

    @Override
    public int readInt(int bitLength) {
        if (bitLength % 8 != 0) {
            throw new IllegalArgumentException("bitLength must be a multiple of 8");
        }
        int byteLength = bitLength / 8;
        this.ensureBytes(byteLength, () -> "SSZ encoded data has insufficient length to read a " + bitLength + "-bit integer");
        Bytes bytes = this.content.slice(this.index, byteLength);
        int zeroBytes = bytes.numberOfTrailingZeroBytes();
        if (byteLength - zeroBytes > 4) {
            throw new InvalidSSZTypeException("decoded integer is too large for an int");
        }
        this.index += byteLength;
        return bytes.slice(0, bytes.size() - zeroBytes).toInt(ByteOrder.LITTLE_ENDIAN);
    }

    @Override
    public long readLong(int bitLength) {
        if (bitLength % 8 != 0) {
            throw new IllegalArgumentException("bitLength must be a multiple of 8");
        }
        int byteLength = bitLength / 8;
        this.ensureBytes(byteLength, () -> "SSZ encoded data has insufficient length to read a " + bitLength + "-bit integer");
        Bytes bytes = this.content.slice(this.index, byteLength);
        int zeroBytes = bytes.numberOfTrailingZeroBytes();
        if (byteLength - zeroBytes > 8) {
            throw new InvalidSSZTypeException("decoded integer is too large for a long");
        }
        this.index += byteLength;
        return bytes.slice(0, bytes.size() - zeroBytes).toLong(ByteOrder.LITTLE_ENDIAN);
    }

    @Override
    public BigInteger readBigInteger(int bitLength) {
        if (bitLength % 8 != 0) {
            throw new IllegalArgumentException("bitLength must be a multiple of 8");
        }
        int byteLength = bitLength / 8;
        this.ensureBytes(byteLength, () -> "SSZ encoded data has insufficient length to read a " + bitLength + "-bit integer");
        return this.consumeBytes(byteLength).toBigInteger(ByteOrder.LITTLE_ENDIAN);
    }

    @Override
    public BigInteger readUnsignedBigInteger(int bitLength) {
        if (bitLength % 8 != 0) {
            throw new IllegalArgumentException("bitLength must be a multiple of 8");
        }
        int byteLength = bitLength / 8;
        this.ensureBytes(byteLength, () -> "SSZ encoded data has insufficient length to read a " + bitLength + "-bit integer");
        return this.consumeBytes(byteLength).toUnsignedBigInteger(ByteOrder.LITTLE_ENDIAN);
    }

    @Override
    public UInt256 readUInt256() {
        this.ensureBytes(32, () -> "SSZ encoded data has insufficient length to read a 256-bit integer");
        return UInt256.fromBytes((Bytes)this.consumeBytes(32).reverse());
    }

    @Override
    public UInt384 readUInt384() {
        this.ensureBytes(48, () -> "SSZ encoded data has insufficient length to read a 384-bit integer");
        return UInt384.fromBytes((Bytes)this.consumeBytes(48).reverse());
    }

    @Override
    public Bytes readAddress() {
        this.ensureBytes(20, () -> "SSZ encoded data has insufficient length to read a 20-byte address");
        return this.consumeBytes(20);
    }

    @Override
    public Bytes readHash(int hashLength) {
        this.ensureBytes(hashLength, () -> "SSZ encoded data has insufficient length to read a " + hashLength + "-byte hash");
        return this.consumeBytes(hashLength);
    }

    @Override
    public List<Bytes> readBytesList(int limit) {
        return this.readList(remaining -> this.readBytes(limit));
    }

    @Override
    public List<Bytes> readVector(long listSize, int limit) {
        return this.readList(listSize, remaining -> this.readByteArray(limit), Bytes::wrap);
    }

    @Override
    public List<Bytes> readFixedBytesVector(int listSize, int byteLength, int limit) {
        return this.readFixedList(listSize, remaining -> this.readFixedByteArray(byteLength, limit), Bytes::wrap);
    }

    @Override
    public List<Bytes> readFixedBytesList(int byteLength, int limit) {
        return this.readList(byteLength, () -> this.readFixedBytes(byteLength, limit));
    }

    @Override
    public List<String> readStringList(int limit) {
        return this.readList(remaining -> this.readBytes(limit), (Bytes bytes) -> new String(bytes.toArrayUnsafe(), StandardCharsets.UTF_8));
    }

    @Override
    public List<Integer> readIntList(int bitLength) {
        if (bitLength % 8 != 0) {
            throw new IllegalArgumentException("bitLength must be a multiple of 8");
        }
        return this.readList(bitLength / 8, () -> this.readInt(bitLength));
    }

    @Override
    public List<Long> readLongIntList(int bitLength) {
        if (bitLength % 8 != 0) {
            throw new IllegalArgumentException("bitLength must be a multiple of 8");
        }
        return this.readList(bitLength / 8, () -> this.readLong(bitLength));
    }

    @Override
    public List<BigInteger> readBigIntegerList(int bitLength) {
        if (bitLength % 8 != 0) {
            throw new IllegalArgumentException("bitLength must be a multiple of 8");
        }
        return this.readList(bitLength / 8, () -> this.readBigInteger(bitLength));
    }

    @Override
    public List<BigInteger> readUnsignedBigIntegerList(int bitLength) {
        if (bitLength % 8 != 0) {
            throw new IllegalArgumentException("bitLength must be a multiple of 8");
        }
        return this.readList(bitLength / 8, () -> this.readUnsignedBigInteger(bitLength));
    }

    @Override
    public List<UInt256> readUInt256List() {
        return this.readList(32, this::readUInt256);
    }

    @Override
    public List<UInt384> readUInt384List() {
        return this.readList(48, this::readUInt384);
    }

    @Override
    public List<Bytes> readAddressList() {
        return this.readList(20, this::readAddress);
    }

    @Override
    public List<Bytes> readHashList(int hashLength) {
        return this.readList(hashLength, () -> this.readHash(hashLength));
    }

    @Override
    public List<Boolean> readBooleanList() {
        return this.readList(1, this::readBoolean);
    }

    @Override
    public boolean isComplete() {
        return this.index >= this.content.size();
    }

    private void ensureBytes(int byteLength, Supplier<String> message) {
        if (this.index == this.content.size()) {
            throw new EndOfSSZException();
        }
        if (this.content.size() - this.index - byteLength < 0) {
            throw new InvalidSSZTypeException(message.get());
        }
    }

    private Bytes consumeBytes(int size) {
        Bytes bytes = this.content.slice(this.index, size);
        this.index += size;
        return bytes;
    }

    @Override
    public Bytes consumeRemainingBytes(int limit) {
        if (this.content.size() - this.index > limit) {
            throw new InvalidSSZTypeException("Too many bytes to consume");
        }
        return this.consumeBytes(this.content.size() - this.index);
    }

    private List<Bytes> readList(LongFunction<Bytes> bytesSupplier) {
        ArrayList<Bytes> elements;
        this.ensureBytes(4, () -> "SSZ encoded data is not a list");
        int originalIndex = this.index;
        try {
            Bytes bytes;
            elements = new ArrayList<Bytes>();
            for (long listSize = this.consumeBytes(4).toLong(ByteOrder.LITTLE_ENDIAN); listSize > 0L; listSize -= (long)bytes.size()) {
                bytes = bytesSupplier.apply(listSize);
                elements.add(bytes);
                if ((listSize -= 4L) >= 0L) continue;
                throw new InvalidSSZTypeException("SSZ encoded list length does not align with lengths of its elements");
            }
        }
        catch (Exception e) {
            this.index = originalIndex;
            throw e;
        }
        return elements;
    }

    private <T> List<T> readList(LongFunction<Bytes> bytesSupplier, Function<Bytes, T> converter) {
        ArrayList<T> elements;
        this.ensureBytes(4, () -> "SSZ encoded data is not a list");
        int originalIndex = this.index;
        try {
            Bytes bytes;
            elements = new ArrayList<T>();
            for (long listSize = this.consumeBytes(4).toLong(ByteOrder.LITTLE_ENDIAN); listSize > 0L; listSize -= (long)bytes.size()) {
                bytes = bytesSupplier.apply(listSize);
                elements.add(converter.apply(bytes));
                if ((listSize -= 4L) >= 0L) continue;
                throw new InvalidSSZTypeException("SSZ encoded list length does not align with lengths of its elements");
            }
        }
        catch (Exception e) {
            this.index = originalIndex;
            throw e;
        }
        return elements;
    }

    private <T> List<T> readList(long listSize, LongFunction<byte[]> bytesSupplier, Function<byte[], T> converter) {
        ArrayList<T> elements;
        int originalIndex = this.index;
        try {
            elements = new ArrayList<T>();
            while (listSize > 0L) {
                byte[] bytes = bytesSupplier.apply(listSize);
                elements.add(converter.apply(bytes));
                if (--listSize >= 0L) continue;
                throw new InvalidSSZTypeException("SSZ encoded list length does not align with lengths of its elements");
            }
        }
        catch (Exception e) {
            this.index = originalIndex;
            throw e;
        }
        return elements;
    }

    private <T> List<T> readFixedList(int listSize, LongFunction<byte[]> bytesSupplier, Function<byte[], T> converter) {
        ArrayList<T> elements;
        int originalIndex = this.index;
        try {
            elements = new ArrayList<T>();
            while (listSize > 0) {
                byte[] bytes = bytesSupplier.apply(listSize);
                elements.add(converter.apply(bytes));
                if (--listSize >= 0) continue;
                throw new InvalidSSZTypeException("SSZ encoded list length does not align with lengths of its elements");
            }
        }
        catch (Exception e) {
            this.index = originalIndex;
            throw e;
        }
        return elements;
    }

    private <T> List<T> readList(int elementSize, Supplier<T> elementSupplier) {
        ArrayList<T> bytesList;
        this.ensureBytes(4, () -> "SSZ encoded data is not a list");
        int originalIndex = this.index;
        try {
            int listSize = this.consumeBytes(4).toInt(ByteOrder.LITTLE_ENDIAN);
            if (listSize % elementSize != 0) {
                throw new InvalidSSZTypeException("SSZ encoded list length does not align with lengths of its elements");
            }
            int nElements = listSize / elementSize;
            bytesList = new ArrayList<T>(nElements);
            for (int i = 0; i < nElements; ++i) {
                bytesList.add(elementSupplier.get());
            }
        }
        catch (Exception e) {
            this.index = originalIndex;
            throw e;
        }
        return bytesList;
    }

    @Override
    public void readAsContainer(SSZReadable ... elements) {
        ArrayList<ElementOffset> variableElements = new ArrayList<ElementOffset>();
        for (SSZReadable element : elements) {
            if (element.isFixed()) {
                element.populateFromReader(this);
                continue;
            }
            variableElements.add(new ElementOffset(this.readUInt32(), element));
        }
        if (variableElements.isEmpty()) {
            return;
        }
        for (int i = 0; i < variableElements.size() - 1; ++i) {
            if (((ElementOffset)variableElements.get(i)).getOffset() != (long)this.index) {
                throw new InvalidSSZTypeException("Variable elements are not in order");
            }
            int length = (int)(((ElementOffset)variableElements.get((int)(i + 1))).offset - ((ElementOffset)variableElements.get((int)i)).offset);
            ((ElementOffset)variableElements.get(i)).getElement().populateFromReader(this.slice(length));
            this.index += length;
        }
        ((ElementOffset)variableElements.get(variableElements.size() - 1)).getElement().populateFromReader(this.slice(this.content.size() - this.index));
    }

    private SSZReader slice(int length) {
        return new BytesSSZReader(this.content.slice(this.index, length));
    }

    @Override
    public <T extends SSZReadable> List<T> readFixedTypedList(int elementSize, Supplier<T> supplier) {
        int listSize = (this.content.size() - this.index) / elementSize;
        return this.readFixedList(listSize, remaining -> this.readFixedByteArray(elementSize, listSize * elementSize), bytes -> {
            SSZReadable t = (SSZReadable)supplier.get();
            t.populateFromReader(new BytesSSZReader(Bytes.wrap((byte[])bytes)));
            return t;
        });
    }

    @Override
    public <T extends SSZReadable> List<T> readTypedVector(int listSize, int elementSize, Supplier<T> supplier) {
        return this.readFixedList(listSize, remaining -> this.readFixedByteArray(elementSize, listSize * elementSize), bytes -> {
            SSZReadable t = (SSZReadable)supplier.get();
            t.populateFromReader(new BytesSSZReader(Bytes.wrap((byte[])bytes)));
            return t;
        });
    }

    @Override
    public <T extends SSZReadable> List<T> readVariableSizeTypeList(Supplier<T> supplier) {
        if (this.content.size() == this.index) {
            return Collections.emptyList();
        }
        long firstOffset = this.readUInt32();
        int size = (int)(firstOffset / 4L) + 1;
        ArrayList<Integer> lengths = new ArrayList<Integer>(size);
        for (int i = 0; i < size; ++i) {
            lengths.add((int)(this.readUInt32() - firstOffset));
        }
        ArrayList<SSZReadable> elements = new ArrayList<SSZReadable>(size);
        for (Integer length : lengths) {
            SSZReadable t = (SSZReadable)supplier.get();
            t.populateFromReader(this.slice(length));
            this.index += length.intValue();
            elements.add(t);
        }
        SSZReadable t = (SSZReadable)supplier.get();
        t.populateFromReader(this.slice(this.content.size() - this.index));
        elements.add(t);
        return Collections.unmodifiableList(elements);
    }

    private static class ElementOffset {
        final long offset;
        final SSZReadable element;

        public ElementOffset(long offset, SSZReadable element) {
            this.offset = offset;
            this.element = element;
        }

        public long getOffset() {
            return this.offset;
        }

        public SSZReadable getElement() {
            return this.element;
        }
    }
}

