/*
 * Decompiled with CFR 0.152.
 */
package io.dapr.client;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.dapr.exceptions.DaprError;
import io.dapr.exceptions.DaprException;
import io.dapr.internal.exceptions.DaprHttpException;
import io.dapr.utils.Version;
import java.io.IOException;
import java.net.URI;
import java.net.URLEncoder;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import reactor.core.publisher.Mono;
import reactor.util.context.ContextView;

public class DaprHttp
implements AutoCloseable {
    public static final String API_VERSION = "v1.0";
    public static final String ALPHA_1_API_VERSION = "v1.0-alpha1";
    private static final String HEADER_DAPR_REQUEST_ID = "X-DaprRequestId";
    private static final String DEFAULT_HTTP_SCHEME = "http";
    private static final Set<String> ALLOWED_CONTEXT_IN_HEADERS = Set.of("grpc-trace-bin", "traceparent", "tracestate");
    private static final ObjectMapper DAPR_ERROR_DETAILS_OBJECT_MAPPER = new ObjectMapper();
    private static final String MEDIA_TYPE_APPLICATION_JSON = "application/json; charset=utf-8";
    private static final byte[] EMPTY_BYTES = new byte[0];
    private static final HttpRequest.BodyPublisher EMPTY_BODY_PUBLISHER = HttpRequest.BodyPublishers.noBody();
    private final URI uri;
    private final String daprApiToken;
    private final Duration readTimeout;
    private final HttpClient httpClient;

    DaprHttp(String hostname, int port, String daprApiToken, Duration readTimeout, HttpClient httpClient) {
        this.uri = URI.create("http://" + hostname + ":" + port);
        this.daprApiToken = daprApiToken;
        this.readTimeout = readTimeout;
        this.httpClient = httpClient;
    }

    DaprHttp(String uri, String daprApiToken, Duration readTimeout, HttpClient httpClient) {
        this.uri = URI.create(uri);
        this.daprApiToken = daprApiToken;
        this.readTimeout = readTimeout;
        this.httpClient = httpClient;
    }

    public Mono<Response> invokeApi(String method, String[] pathSegments, Map<String, List<String>> urlParameters, Map<String, String> headers, ContextView context) {
        return this.invokeApi(method, pathSegments, urlParameters, (byte[])null, headers, context);
    }

    public Mono<Response> invokeApi(String method, String[] pathSegments, Map<String, List<String>> urlParameters, String content, Map<String, String> headers, ContextView context) {
        return this.invokeApi(method, pathSegments, urlParameters, content == null ? EMPTY_BYTES : content.getBytes(StandardCharsets.UTF_8), headers, context);
    }

    public Mono<Response> invokeApi(String method, String[] pathSegments, Map<String, List<String>> urlParameters, byte[] content, Map<String, String> headers, ContextView context) {
        return Mono.fromCallable(() -> this.doInvokeApi(method, headers, pathSegments, urlParameters, content, context)).flatMap(Mono::fromFuture);
    }

    @Override
    public void close() {
    }

    private CompletableFuture<Response> doInvokeApi(String method, Map<String, String> headers, String[] pathSegments, Map<String, List<String>> urlParameters, byte[] content, ContextView context) {
        HttpRequest.Builder requestBuilder = HttpRequest.newBuilder();
        requestBuilder.uri(DaprHttp.createUri(this.uri, pathSegments, urlParameters));
        DaprHttp.addHeader(requestBuilder, "User-Agent", Version.getSdkVersion());
        DaprHttp.addHeader(requestBuilder, HEADER_DAPR_REQUEST_ID, UUID.randomUUID().toString());
        DaprHttp.addHeader(requestBuilder, "Content-Type", DaprHttp.getContentType(headers));
        DaprHttp.addHeaders(requestBuilder, headers);
        if (this.daprApiToken != null) {
            DaprHttp.addHeader(requestBuilder, "dapr-api-token", this.daprApiToken);
        }
        if (context != null) {
            context.stream().filter(entry -> ALLOWED_CONTEXT_IN_HEADERS.contains(entry.getKey().toString().toLowerCase())).forEach(entry -> DaprHttp.addHeader(requestBuilder, entry.getKey().toString(), entry.getValue().toString()));
        }
        HttpRequest.BodyPublisher body = DaprHttp.getBodyPublisher(content);
        if (HttpMethods.GET.name().equals(method)) {
            requestBuilder.GET();
        } else if (HttpMethods.DELETE.name().equals(method)) {
            requestBuilder.DELETE();
        } else if (HttpMethods.HEAD.name().equals(method)) {
            requestBuilder.method(HttpMethods.HEAD.name(), EMPTY_BODY_PUBLISHER);
        } else {
            requestBuilder.method(method, body);
        }
        HttpRequest request = requestBuilder.timeout(this.readTimeout).build();
        return this.httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofByteArray()).thenApply(this::createResponse);
    }

    private static String getContentType(Map<String, String> headers) {
        String result = headers != null ? headers.get("content-type") : null;
        return result == null ? MEDIA_TYPE_APPLICATION_JSON : result;
    }

    private static URI createUri(URI uri, String[] pathSegments, Map<String, List<String>> urlParameters) {
        String path = DaprHttp.createPath(uri, pathSegments);
        String query = DaprHttp.createQuery(urlParameters);
        StringBuilder result = new StringBuilder();
        result.append(uri.getScheme()).append("://").append(uri.getAuthority()).append(path);
        if (query != null) {
            result.append("?").append(query);
        }
        try {
            return URI.create(result.toString());
        }
        catch (IllegalArgumentException exception) {
            throw new DaprException(exception);
        }
    }

    private static String createPath(URI uri, String[] pathSegments) {
        String basePath = uri.getPath();
        if (pathSegments == null || pathSegments.length == 0) {
            return basePath;
        }
        StringBuilder pathBuilder = new StringBuilder(basePath);
        if (!basePath.endsWith("/")) {
            pathBuilder.append("/");
        }
        for (String segment : pathSegments) {
            if (segment == null || segment.isEmpty()) continue;
            pathBuilder.append(DaprHttp.encodePathSegment(segment)).append("/");
        }
        pathBuilder.deleteCharAt(pathBuilder.length() - 1);
        return pathBuilder.toString();
    }

    private static String createQuery(Map<String, List<String>> urlParameters) {
        if (urlParameters == null || urlParameters.isEmpty()) {
            return null;
        }
        StringBuilder queryBuilder = new StringBuilder();
        for (Map.Entry<String, List<String>> entry : urlParameters.entrySet()) {
            String key = entry.getKey();
            if (key == null || key.isEmpty()) continue;
            List<String> values = entry.getValue();
            for (String value : values) {
                if (queryBuilder.length() > 0) {
                    queryBuilder.append("&");
                }
                queryBuilder.append(DaprHttp.encodeQueryParam(key, value));
            }
        }
        return queryBuilder.toString();
    }

    private static String encodePathSegment(String segment) {
        return URLEncoder.encode(segment, StandardCharsets.UTF_8).replace("+", "%20");
    }

    private static String encodeQueryParam(String key, String value) {
        return URLEncoder.encode(key, StandardCharsets.UTF_8) + "=" + URLEncoder.encode(value, StandardCharsets.UTF_8);
    }

    private static void addHeader(HttpRequest.Builder requestBuilder, String name, String value) {
        requestBuilder.header(name, value);
    }

    private static void addHeaders(HttpRequest.Builder requestBuilder, Map<String, String> headers) {
        if (headers == null || headers.isEmpty()) {
            return;
        }
        headers.forEach((k, v) -> DaprHttp.addHeader(requestBuilder, k, v));
    }

    private static HttpRequest.BodyPublisher getBodyPublisher(byte[] content) {
        return HttpRequest.BodyPublishers.ofByteArray(Objects.requireNonNullElse(content, EMPTY_BYTES));
    }

    private Response createResponse(HttpResponse<byte[]> httpResponse) {
        Optional<String> headerValue = httpResponse.headers().firstValue("Metadata.statuscode");
        int httpStatusCode = DaprHttp.parseHttpStatusCode(headerValue, httpResponse.statusCode());
        byte[] body = DaprHttp.getBodyBytesOrEmptyArray(httpResponse.body());
        if (!DaprHttpException.isSuccessfulHttpStatusCode(httpStatusCode)) {
            DaprError error = DaprHttp.parseDaprError(body);
            if (error != null) {
                throw new DaprException(error, body, httpStatusCode);
            }
            throw new DaprException("UNKNOWN", "", body, httpStatusCode);
        }
        HashMap<String, String> responseHeaders = new HashMap<String, String>();
        httpResponse.headers().map().forEach((k, v) -> responseHeaders.put((String)k, v.isEmpty() ? null : (String)v.get(0)));
        return new Response(body, responseHeaders, httpStatusCode);
    }

    private static DaprError parseDaprError(byte[] json) {
        if (json == null || json.length == 0) {
            return null;
        }
        try {
            return (DaprError)DAPR_ERROR_DETAILS_OBJECT_MAPPER.readValue(json, DaprError.class);
        }
        catch (IOException e) {
            return null;
        }
    }

    private static byte[] getBodyBytesOrEmptyArray(byte[] body) {
        return body == null ? EMPTY_BYTES : body;
    }

    private static int parseHttpStatusCode(Optional<String> headerValue, int defaultStatusCode) {
        if (headerValue.isEmpty()) {
            return defaultStatusCode;
        }
        try {
            int httpStatusCode = Integer.parseInt(headerValue.get());
            if (DaprHttpException.isValidHttpStatusCode(httpStatusCode)) {
                return httpStatusCode;
            }
            return defaultStatusCode;
        }
        catch (NumberFormatException nfe) {
            return defaultStatusCode;
        }
    }

    public static enum HttpMethods {
        NONE,
        GET,
        PUT,
        POST,
        DELETE,
        HEAD,
        CONNECT,
        OPTIONS,
        TRACE;

    }

    public static class Response {
        private final byte[] body;
        private final Map<String, String> headers;
        private final int statusCode;

        public Response(byte[] body, Map<String, String> headers, int statusCode) {
            this.body = body == null ? EMPTY_BYTES : Arrays.copyOf(body, body.length);
            this.headers = headers == null ? null : Collections.unmodifiableMap(headers);
            this.statusCode = statusCode;
        }

        public byte[] getBody() {
            return Arrays.copyOf(this.body, this.body.length);
        }

        public Map<String, String> getHeaders() {
            return this.headers;
        }

        public int getStatusCode() {
            return this.statusCode;
        }
    }
}

