/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.oracle;

import io.debezium.DebeziumException;
import io.debezium.config.CommonConnectorConfig;
import io.debezium.config.Field;
import io.debezium.connector.oracle.OracleConnectorConfig;
import io.debezium.connector.oracle.OracleDatabaseVersion;
import io.debezium.connector.oracle.OracleOffsetContext;
import io.debezium.connector.oracle.RedoThreadState;
import io.debezium.connector.oracle.Scn;
import io.debezium.connector.oracle.logminer.SqlUtils;
import io.debezium.connector.oracle.util.OracleUtils;
import io.debezium.jdbc.JdbcConfiguration;
import io.debezium.jdbc.JdbcConnection;
import io.debezium.pipeline.spi.OffsetContext;
import io.debezium.pipeline.spi.Partition;
import io.debezium.relational.Attribute;
import io.debezium.relational.Column;
import io.debezium.relational.ColumnEditor;
import io.debezium.relational.Table;
import io.debezium.relational.TableId;
import io.debezium.relational.Tables;
import io.debezium.util.Strings;
import java.sql.CallableStatement;
import java.sql.Clob;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLRecoverableException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.time.Duration;
import java.time.Instant;
import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import oracle.sql.CharacterSet;
import org.apache.kafka.connect.data.Struct;
import org.apache.kafka.connect.errors.RetriableException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OracleConnection
extends JdbcConnection {
    private static final Logger LOGGER = LoggerFactory.getLogger(OracleConnection.class);
    private static final int ORACLE_UNSET_SCALE = -127;
    private static final Pattern SYS_NC_PATTERN = Pattern.compile("^SYS_NC(?:_OID|_ROWINFO|[0-9][0-9][0-9][0-9][0-9])\\$$");
    private static final Pattern ADT_INDEX_NAMES_PATTERN = Pattern.compile("^\".*\"\\.\".*\".*");
    private static final Pattern MROW_PATTERN = Pattern.compile("^M_ROW\\$\\$");
    private static final Field URL = Field.create((String)"url", (String)"Raw JDBC url");
    private final OracleDatabaseVersion databaseVersion;
    private static final String QUOTED_CHARACTER = "\"";
    private static final Set<Integer> ORACLE_RESELECT_ERROR_CODE_FALLBACK = Set.of(Integer.valueOf(1555), Integer.valueOf(1466));
    private static final Set<String> ORACLE_RESELECT_ERROR_PREFIX_FALLBACK = Set.of("ORA-01555", "ORA-01466");

    public OracleConnection(JdbcConfiguration config) {
        this(config, true);
    }

    public OracleConnection(JdbcConfiguration config, JdbcConnection.ConnectionFactory connectionFactory) {
        this(config, connectionFactory, true);
    }

    public OracleConnection(JdbcConfiguration config, JdbcConnection.ConnectionFactory connectionFactory, boolean showVersion) {
        super(config, connectionFactory, QUOTED_CHARACTER, QUOTED_CHARACTER);
        LOGGER.trace("JDBC connection string: " + OracleConnection.connectionString(config));
        this.databaseVersion = this.resolveOracleDatabaseVersion();
        if (showVersion) {
            LOGGER.info("Database Version: {}", (Object)this.databaseVersion.getBanner());
        }
    }

    public OracleConnection(JdbcConfiguration config, boolean showVersion) {
        super(config, OracleConnection.resolveConnectionFactory(config), QUOTED_CHARACTER, QUOTED_CHARACTER);
        LOGGER.trace("JDBC connection string: " + OracleConnection.connectionString(config));
        this.databaseVersion = this.resolveOracleDatabaseVersion();
        if (showVersion) {
            LOGGER.info("Database Version: {}", (Object)this.databaseVersion.getBanner());
        }
    }

    public void setSessionToPdb(String pdbName) {
        this.setContainerAs(pdbName);
    }

    public void resetSessionToCdb() {
        this.setContainerAs("cdb$root");
    }

    private void setContainerAs(String containerName) {
        Statement statement = null;
        try {
            statement = this.connection().createStatement();
            statement.execute("alter session set container=" + containerName);
        }
        catch (SQLException e) {
            throw new RuntimeException(e);
        }
        finally {
            if (statement != null) {
                try {
                    statement.close();
                }
                catch (SQLException e) {
                    LOGGER.error("Couldn't close statement", (Throwable)e);
                }
            }
        }
    }

    public OracleDatabaseVersion getOracleVersion() {
        return this.databaseVersion;
    }

    private OracleDatabaseVersion resolveOracleDatabaseVersion() {
        String versionStr;
        try {
            try {
                versionStr = (String)this.queryAndMap("SELECT BANNER_FULL FROM V$VERSION WHERE BANNER_FULL LIKE 'Oracle Database%'", rs -> {
                    if (rs.next()) {
                        return rs.getString(1);
                    }
                    return null;
                });
            }
            catch (SQLException e) {
                if (e.getMessage().contains("ORA-00904: \"BANNER_FULL\"")) {
                    LOGGER.debug("BANNER_FULL column not in V$VERSION, using BANNER column as fallback");
                    versionStr = null;
                }
                throw e;
            }
            if (versionStr == null) {
                versionStr = (String)this.queryAndMap("SELECT BANNER FROM V$VERSION WHERE BANNER LIKE 'Oracle Database%'", rs -> {
                    if (rs.next()) {
                        return rs.getString(1);
                    }
                    return null;
                });
            }
        }
        catch (SQLException e) {
            if (e instanceof SQLRecoverableException) {
                throw new RetriableException("Failed to resolve Oracle database version", (Throwable)e);
            }
            throw new RuntimeException("Failed to resolve Oracle database version", e);
        }
        if (versionStr == null) {
            throw new RuntimeException("Failed to resolve Oracle database version");
        }
        return OracleDatabaseVersion.parse(versionStr);
    }

    public Set<TableId> readTableNames(String catalogName, String schemaNamePattern, String tableNamePattern, String[] tableTypes) throws SQLException {
        Set tableIds = super.readTableNames(null, schemaNamePattern, tableNamePattern, tableTypes);
        return tableIds.stream().map(t -> new TableId(catalogName, t.schema(), t.table())).collect(Collectors.toSet());
    }

    protected Set<TableId> getAllTableIds(String catalogName) throws SQLException {
        String query = "select owner, table_name from all_tables where table_name NOT LIKE 'MDRT_%' and table_name NOT LIKE 'MDRS_%' and table_name NOT LIKE 'MDXT_%' and (table_name NOT LIKE 'SYS_IOT_OVER_%' and IOT_NAME IS NULL) and nested = 'NO'and table_name not in (select PARENT_TABLE_NAME from ALL_NESTED_TABLES)";
        HashSet<TableId> tableIds = new HashSet<TableId>();
        this.query("select owner, table_name from all_tables where table_name NOT LIKE 'MDRT_%' and table_name NOT LIKE 'MDRS_%' and table_name NOT LIKE 'MDXT_%' and (table_name NOT LIKE 'SYS_IOT_OVER_%' and IOT_NAME IS NULL) and nested = 'NO'and table_name not in (select PARENT_TABLE_NAME from ALL_NESTED_TABLES)", rs -> {
            while (rs.next()) {
                tableIds.add(new TableId(catalogName, rs.getString(1), rs.getString(2)));
            }
            LOGGER.trace("TableIds are: {}", (Object)tableIds);
        });
        return tableIds;
    }

    protected String resolveCatalogName(String catalogName) {
        String pdbName = OracleUtils.getObjectName(this.config().getString("pdb.name"));
        String databaseName = OracleUtils.getObjectName(this.config().getString("dbname"));
        return !OracleUtils.isObjectNameNullOrEmpty(pdbName) ? pdbName : databaseName;
    }

    public List<String> readTableUniqueIndices(DatabaseMetaData metadata, TableId id) throws SQLException {
        return super.readTableUniqueIndices(metadata, id.toDoubleQuoted());
    }

    public Optional<Instant> getCurrentTimestamp() throws SQLException {
        return (Optional)this.queryAndMap("SELECT CURRENT_TIMESTAMP FROM DUAL", rs -> rs.next() ? Optional.of(rs.getTimestamp(1).toInstant()) : Optional.empty());
    }

    protected boolean isTableUniqueIndexIncluded(String indexName, String columnName) {
        if (columnName != null) {
            return !SYS_NC_PATTERN.matcher(columnName).matches() && !ADT_INDEX_NAMES_PATTERN.matcher(columnName).matches() && !MROW_PATTERN.matcher(columnName).matches();
        }
        return false;
    }

    public Scn getCurrentScn() throws SQLException {
        return (Scn)this.queryAndMap("SELECT CURRENT_SCN FROM V$DATABASE", rs -> {
            if (rs.next()) {
                return Scn.valueOf(rs.getString(1));
            }
            throw new IllegalStateException("Could not get SCN");
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String getTableMetadataDdl(TableId tableId) throws SQLException, NonRelationalTableException {
        String string;
        try {
            String tableType = "SELECT COUNT(1) FROM ALL_ALL_TABLES WHERE OWNER=? AND TABLE_NAME=? AND TABLE_TYPE IS NULL";
            if ((Integer)this.prepareQueryAndMap("SELECT COUNT(1) FROM ALL_ALL_TABLES WHERE OWNER=? AND TABLE_NAME=? AND TABLE_TYPE IS NULL", ps -> {
                ps.setString(1, tableId.schema());
                ps.setString(2, tableId.table());
            }, rs -> rs.next() ? rs.getInt(1) : 0) == 0) {
                throw new NonRelationalTableException("Table " + String.valueOf(tableId) + " was not found in ALL_ALL_TABLES, which could mean its a grant/permission issue or it's not a relational table.");
            }
            this.executeWithoutCommitting(new String[]{"begin dbms_metadata.set_transform_param(DBMS_METADATA.SESSION_TRANSFORM, 'STORAGE', false); end;"});
            this.executeWithoutCommitting(new String[]{"begin dbms_metadata.set_transform_param(DBMS_METADATA.SESSION_TRANSFORM, 'SEGMENT_ATTRIBUTES', false); end;"});
            this.executeWithoutCommitting(new String[]{"begin dbms_metadata.set_transform_param(DBMS_METADATA.SESSION_TRANSFORM, 'SQLTERMINATOR', true); end;"});
            string = (String)this.prepareQueryAndMap("SELECT dbms_metadata.get_ddl('TABLE',?,?) FROM DUAL", ps -> {
                ps.setString(1, tableId.table());
                ps.setString(2, tableId.schema());
            }, rs -> {
                if (!rs.next()) {
                    throw new DebeziumException("Could not get DDL metadata for table: " + String.valueOf(tableId));
                }
                Object res = rs.getObject(1);
                return ((Clob)res).getSubString(1L, (int)((Clob)res).length());
            });
        }
        catch (Throwable throwable) {
            this.executeWithoutCommitting(new String[]{"begin dbms_metadata.set_transform_param(DBMS_METADATA.SESSION_TRANSFORM, 'DEFAULT'); end;"});
            throw throwable;
        }
        this.executeWithoutCommitting(new String[]{"begin dbms_metadata.set_transform_param(DBMS_METADATA.SESSION_TRANSFORM, 'DEFAULT'); end;"});
        return string;
    }

    public Long getSessionStatisticByName(String name) throws SQLException {
        return (Long)this.queryAndMap("SELECT VALUE FROM v$statname n, v$mystat m WHERE n.name='" + name + "' AND n.statistic#=m.statistic#", rs -> rs.next() ? rs.getLong(1) : 0L);
    }

    public boolean isTableExists(TableId tableId) throws SQLException {
        if (Strings.isNullOrBlank((String)tableId.schema())) {
            return (Boolean)this.prepareQueryAndMap("SELECT COUNT(1) FROM USER_TABLES WHERE TABLE_NAME=?", ps -> ps.setString(1, tableId.table()), rs -> rs.next() && rs.getLong(1) > 0L);
        }
        return (Boolean)this.prepareQueryAndMap("SELECT COUNT(1) FROM ALL_TABLES WHERE OWNER=? AND TABLE_NAME=?", ps -> {
            ps.setString(1, tableId.schema());
            ps.setString(2, tableId.table());
        }, rs -> rs.next() && rs.getLong(1) > 0L);
    }

    public boolean isTableEmpty(TableId tableId) throws SQLException {
        return this.getRowCount(tableId) == 0L;
    }

    public long getRowCount(TableId tableId) throws SQLException {
        return (Long)this.queryAndMap("SELECT COUNT(1) FROM " + tableId.toDoubleQuotedString(), rs -> {
            if (rs.next()) {
                return rs.getLong(1);
            }
            return 0L;
        });
    }

    public <T> T singleOptionalValue(String query, JdbcConnection.ResultSetExtractor<T> extractor) throws SQLException {
        return (T)this.queryAndMap(query, rs -> rs.next() ? extractor.apply(rs) : null);
    }

    public Optional<Scn> getFirstScnInLogs(Duration archiveLogRetention, String archiveDestinationName) throws SQLException {
        String oldestFirstChangeQuery = SqlUtils.oldestFirstChangeQuery(archiveLogRetention, archiveDestinationName);
        String oldestScn = (String)this.singleOptionalValue(oldestFirstChangeQuery, rs -> rs.getString(1));
        if (oldestScn == null) {
            return Optional.empty();
        }
        LOGGER.trace("Oldest SCN in logs is '{}'", (Object)oldestScn);
        return Optional.of(Scn.valueOf(oldestScn));
    }

    public boolean validateLogPosition(Partition partition, OffsetContext offset, CommonConnectorConfig config) {
        Duration archiveLogRetention = ((OracleConnectorConfig)config).getArchiveLogRetention();
        String archiveDestinationName = ((OracleConnectorConfig)config).getArchiveLogDestinationName();
        Scn storedOffset = ((OracleConnectorConfig)config).getAdapter().getOffsetScn((OracleOffsetContext)offset);
        try {
            Optional<Scn> firstAvailableScn = this.getFirstScnInLogs(archiveLogRetention, archiveDestinationName);
            return firstAvailableScn.filter(OracleConnection.isLessThan(storedOffset)).isPresent();
        }
        catch (SQLException e) {
            throw new DebeziumException("Unable to get last available log position", (Throwable)e);
        }
    }

    private static Predicate<Scn> isLessThan(Scn storedOffset) {
        return scn -> scn.compareTo(storedOffset) < 0;
    }

    public String buildSelectWithRowLimits(TableId tableId, int limit, String projection, Optional<String> condition, Optional<String> additionalCondition, String orderBy) {
        TableId table = new TableId(null, tableId.schema(), tableId.table());
        StringBuilder sql = new StringBuilder("SELECT ");
        sql.append(projection).append(" FROM ");
        sql.append(this.quotedTableIdString(table));
        if (condition.isPresent()) {
            sql.append(" WHERE ").append(condition.get());
            if (additionalCondition.isPresent()) {
                sql.append(" AND ");
                sql.append(additionalCondition.get());
            }
        } else if (additionalCondition.isPresent()) {
            sql.append(" WHERE ");
            sql.append(additionalCondition.get());
        }
        if (this.getOracleVersion().getMajor() < 12) {
            sql.insert(0, " SELECT * FROM (").append(" ORDER BY ").append(orderBy).append(")").append(" WHERE ROWNUM <=").append(limit);
        } else {
            sql.append(" ORDER BY ").append(orderBy).append(" FETCH NEXT ").append(limit).append(" ROWS ONLY");
        }
        return sql.toString();
    }

    public static String connectionString(JdbcConfiguration config) {
        return config.getString(URL) != null ? config.getString(URL) : OracleConnectorConfig.ConnectorAdapter.parse(config.getString("connection.adapter")).getConnectionUrl();
    }

    private static JdbcConnection.ConnectionFactory resolveConnectionFactory(JdbcConfiguration config) {
        return JdbcConnection.patternBasedFactory((String)OracleConnection.connectionString(config), (Field[])new Field[0]);
    }

    protected boolean isArchiveLogMode() {
        try {
            String mode = (String)this.queryAndMap("SELECT LOG_MODE FROM V$DATABASE", rs -> rs.next() ? rs.getString(1) : "");
            LOGGER.debug("LOG_MODE={}", (Object)mode);
            return "ARCHIVELOG".equalsIgnoreCase(mode);
        }
        catch (SQLException e) {
            throw new DebeziumException("Unexpected error while connecting to Oracle and looking at LOG_MODE mode: ", (Throwable)e);
        }
    }

    public Optional<Instant> getScnToTimestamp(Scn scn) throws SQLException {
        try {
            return (Optional)this.queryAndMap("SELECT scn_to_timestamp('" + String.valueOf(scn) + "') FROM DUAL", rs -> rs.next() ? Optional.of(rs.getTimestamp(1).toInstant()) : Optional.empty());
        }
        catch (SQLException e) {
            if (e.getMessage().startsWith("ORA-08181")) {
                return Optional.empty();
            }
            throw e;
        }
    }

    public Scn getScnAdjustedByTime(Scn scn, Duration adjustment) throws SQLException {
        try {
            String result = (String)this.prepareQueryAndMap("SELECT timestamp_to_scn(scn_to_timestamp(?) - (? / 86400000)) FROM DUAL", st -> {
                st.setString(1, scn.toString());
                st.setLong(2, adjustment.toMillis());
            }, this.singleResultMapper(rs -> rs.getString(1), "Failed to get adjusted SCN from: " + String.valueOf(scn)));
            return Scn.valueOf(result);
        }
        catch (SQLException e) {
            if (e.getErrorCode() == 8181 || e.getErrorCode() == 8180) {
                return Scn.NULL;
            }
            throw e;
        }
    }

    public OffsetDateTime getDatabaseSystemTime() throws SQLException {
        return (OffsetDateTime)this.singleOptionalValue("SELECT SYSTIMESTAMP FROM DUAL", rs -> rs.getObject(1, OffsetDateTime.class));
    }

    public boolean isArchiveLogDestinationValid(String archiveDestinationName) throws SQLException {
        return (Boolean)this.prepareQueryAndMap("SELECT STATUS, TYPE FROM V$ARCHIVE_DEST_STATUS WHERE DEST_NAME=?", st -> st.setString(1, archiveDestinationName), rs -> {
            if (!rs.next()) {
                throw new DebeziumException(String.format("Archive log destination name '%s' is unknown to Oracle", archiveDestinationName));
            }
            return "VALID".equals(rs.getString("STATUS")) && "LOCAL".equals(rs.getString("TYPE"));
        });
    }

    public boolean isOnlyOneArchiveLogDestinationValid() throws SQLException {
        return (Boolean)this.queryAndMap("SELECT COUNT(1) FROM V$ARCHIVE_DEST_STATUS WHERE STATUS='VALID' AND TYPE='LOCAL'", rs -> {
            if (!rs.next()) {
                throw new DebeziumException("Unable to resolve number of archive log destinations");
            }
            return rs.getLong(1) == 1L;
        });
    }

    protected ColumnEditor overrideColumn(ColumnEditor column) {
        if (93 == column.jdbcType()) {
            column.length(column.scale().orElse(-1).intValue()).scale(null);
        } else if (2 == column.jdbcType()) {
            column.scale().filter(s -> s == -127).ifPresent(s -> column.scale(null));
        }
        return column;
    }

    protected Map<TableId, List<Column>> getColumnsDetails(String catalogName, String schemaName, String tableName, Tables.TableFilter tableFilter, Tables.ColumnNameFilter columnFilter, DatabaseMetaData metadata, Set<TableId> viewIds) throws SQLException {
        if (tableName != null && tableName.contains("/")) {
            tableName = tableName.replace("/", "//");
        }
        return super.getColumnsDetails(catalogName, schemaName, tableName, tableFilter, columnFilter, metadata, viewIds);
    }

    public Optional<Boolean> nullsSortLast() {
        return Optional.of(true);
    }

    public Map<String, Object> reselectColumns(Table table, List<String> columns, List<String> keyColumns, List<Object> keyValues, Struct source) throws SQLException {
        String commitScn;
        TableId oracleTableId = new TableId(null, table.id().schema(), table.id().table());
        if (source != null && !Strings.isNullOrEmpty((String)(commitScn = source.getString("commit_scn")))) {
            String query = String.format("SELECT %s FROM (SELECT * FROM %s AS OF SCN ?) WHERE %s", columns.stream().map(arg_0 -> ((OracleConnection)this).quoteIdentifier(arg_0)).collect(Collectors.joining(",")), this.quotedTableIdString(oracleTableId), keyColumns.stream().map(key -> key + "=?").collect(Collectors.joining(" AND ")));
            ArrayList<Object> bindValues = new ArrayList<Object>(keyValues.size() + 1);
            bindValues.add(commitScn);
            bindValues.addAll(keyValues);
            try {
                return this.reselectColumns(query, oracleTableId, columns, bindValues);
            }
            catch (SQLException e) {
                if (OracleConnection.shouldReselectFallbackToNonFlashbackQuery(e)) {
                    LOGGER.warn("Failed to re-select row for table {} and key columns {} with values {}. Trying to perform re-selection without flashback.", new Object[]{table.id(), keyColumns, keyValues});
                }
                throw e;
            }
        }
        String query = String.format("SELECT %s FROM %s WHERE %s", columns.stream().map(arg_0 -> ((OracleConnection)this).quoteIdentifier(arg_0)).collect(Collectors.joining(",")), this.quotedTableIdString(oracleTableId), keyColumns.stream().map(key -> key + "=?").collect(Collectors.joining(" AND ")));
        return this.reselectColumns(query, oracleTableId, columns, keyValues);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private static boolean shouldReselectFallbackToNonFlashbackQuery(SQLException exception) {
        if (ORACLE_RESELECT_ERROR_CODE_FALLBACK.contains(exception.getErrorCode())) return true;
        if (!ORACLE_RESELECT_ERROR_PREFIX_FALLBACK.stream().anyMatch(exception.getMessage()::startsWith)) return false;
        return true;
    }

    protected Map<TableId, List<Attribute>> getAttributeDetails(TableId tableId, String tableType) {
        HashMap<TableId, List<Attribute>> results = new HashMap<TableId, List<Attribute>>();
        try {
            this.getDatabaseObjectDetails(tableId, tableType, (objectId, dataObjectId) -> {
                LOGGER.info("\tRegistering '{}' attributes: object_id={}, data_object_id={}", new Object[]{tableId, objectId, dataObjectId});
                ArrayList<Attribute> attributes = new ArrayList<Attribute>();
                attributes.add(Attribute.editor().name("OBJECT_ID").value((Object)objectId).create());
                attributes.add(Attribute.editor().name("DATA_OBJECT_ID").value((Object)dataObjectId).create());
                results.put(tableId, attributes);
            });
            return results;
        }
        catch (SQLException e) {
            throw new DebeziumException("Failed to get table attributes for table: " + String.valueOf(tableId), (Throwable)e);
        }
    }

    private void getDatabaseObjectDetails(TableId tableId, String tableType, ObjectIdentifierConsumer consumer) throws SQLException {
        String query = "SELECT OBJECT_ID, DATA_OBJECT_ID FROM ALL_OBJECTS WHERE OBJECT_NAME=? AND OWNER=? AND OBJECT_TYPE=?";
        this.prepareQuery("SELECT OBJECT_ID, DATA_OBJECT_ID FROM ALL_OBJECTS WHERE OBJECT_NAME=? AND OWNER=? AND OBJECT_TYPE=?", List.of(tableId.table(), tableId.schema(), tableType), (params, rs) -> {
            if (!rs.next()) {
                throw new SQLException("Query 'SELECT OBJECT_ID, DATA_OBJECT_ID FROM ALL_OBJECTS WHERE OBJECT_NAME=? AND OWNER=? AND OBJECT_TYPE=?' returned no results.");
            }
            consumer.apply(rs.getLong(1), rs.getLong(2));
        });
    }

    public Long getTableObjectId(TableId tableId) throws SQLException {
        return (Long)this.prepareQueryAndMap("SELECT OBJECT_ID FROM ALL_OBJECTS WHERE OBJECT_TYPE='TABLE' AND OWNER=? AND OBJECT_NAME=?", ps -> {
            ps.setString(1, tableId.schema());
            ps.setString(2, tableId.table());
        }, rs -> rs.next() ? Long.valueOf(rs.getLong(1)) : null);
    }

    public Long getTableDataObjectId(TableId tableId) throws SQLException {
        return (Long)this.prepareQueryAndMap("SELECT DATA_OBJECT_ID FROM ALL_OBJECTS WHERE OBJECT_TYPE='TABLE' AND OWNER=? AND OBJECT_NAME=?", ps -> {
            ps.setString(1, tableId.schema());
            ps.setString(2, tableId.table());
        }, rs -> rs.next() ? Long.valueOf(rs.getLong(1)) : null);
    }

    public CharacterSet getNationalCharacterSet() {
        String query = "select VALUE from NLS_DATABASE_PARAMETERS where PARAMETER = 'NLS_NCHAR_CHARACTERSET'";
        try {
            String nlsCharacterSet = (String)this.queryAndMap("select VALUE from NLS_DATABASE_PARAMETERS where PARAMETER = 'NLS_NCHAR_CHARACTERSET'", rs -> {
                if (rs.next()) {
                    return rs.getString(1);
                }
                return null;
            });
            if (nlsCharacterSet != null) {
                switch (nlsCharacterSet) {
                    case "AL16UTF16": {
                        return CharacterSet.make((int)2000);
                    }
                    case "UTF8": {
                        return CharacterSet.make((int)871);
                    }
                }
            }
            throw new SQLException("An unexpected NLS_NCHAR_CHARACTERSET detected: " + nlsCharacterSet);
        }
        catch (SQLException e) {
            throw new DebeziumException("Failed to resolve Oracle's NLS_NCHAR_CHARACTERSET property", (Throwable)e);
        }
    }

    public void removeAllLogFilesFromLogMinerSession() throws SQLException {
        Set fileNames = (Set)this.queryAndMap("SELECT FILENAME AS NAME FROM V$LOGMNR_LOGS", rs -> {
            HashSet<String> results = new HashSet<String>();
            while (rs.next()) {
                results.add(rs.getString(1));
            }
            return results;
        });
        for (String fileName : fileNames) {
            LOGGER.debug("Removing file {} from LogMiner mining session.", (Object)fileName);
            String sql = "BEGIN SYS.DBMS_LOGMNR.REMOVE_LOGFILE(LOGFILENAME => '" + fileName + "');END;";
            CallableStatement statement = this.connection(false).prepareCall(sql);
            try {
                statement.execute();
            }
            finally {
                if (statement == null) continue;
                statement.close();
            }
        }
    }

    public RedoThreadState getRedoThreadState() throws SQLException {
        String query = "SELECT * FROM V$THREAD";
        try {
            return (RedoThreadState)this.queryAndMap("SELECT * FROM V$THREAD", rs -> {
                RedoThreadState.Builder builder = RedoThreadState.builder();
                while (rs.next()) {
                    int threadId = rs.getInt("THREAD#");
                    if (rs.wasNull()) continue;
                    RedoThreadState.RedoThread.Builder threadBuilder = builder.thread().threadId(threadId).status(rs.getString("STATUS")).enabled(rs.getString("ENABLED")).logGroups(rs.getLong("GROUPS")).instanceName(rs.getString("INSTANCE")).openTime(OracleConnection.readTimestampAsInstant(rs, "OPEN_TIME")).currentGroupNumber(rs.getLong("CURRENT_GROUP#")).currentSequenceNumber(rs.getLong("SEQUENCE#")).checkpointScn(OracleConnection.readScnColumnAsScn(rs, "CHECKPOINT_CHANGE#")).checkpointTime(OracleConnection.readTimestampAsInstant(rs, "CHECKPOINT_TIME")).enabledScn(OracleConnection.readScnColumnAsScn(rs, "ENABLE_CHANGE#")).enabledTime(OracleConnection.readTimestampAsInstant(rs, "ENABLE_TIME")).disabledScn(OracleConnection.readScnColumnAsScn(rs, "DISABLE_CHANGE#")).disabledTime(OracleConnection.readTimestampAsInstant(rs, "DISABLE_TIME"));
                    if (this.getOracleVersion().getMajor() >= 11) {
                        threadBuilder = threadBuilder.lastRedoSequenceNumber(rs.getLong("LAST_REDO_SEQUENCE#")).lastRedoBlock(rs.getLong("LAST_REDO_BLOCK")).lastRedoScn(OracleConnection.readScnColumnAsScn(rs, "LAST_REDO_CHANGE#")).lastRedoTime(OracleConnection.readTimestampAsInstant(rs, "LAST_REDO_TIME"));
                    }
                    if (this.getOracleVersion().getMajor() >= 12) {
                        threadBuilder = threadBuilder.conId(rs.getLong("CON_ID"));
                    }
                    builder = threadBuilder.build();
                }
                return builder.build();
            });
        }
        catch (SQLException e) {
            throw new DebeziumException("Failed to read the Oracle database redo thread state", (Throwable)e);
        }
    }

    public List<String> getSQLKeywords() {
        try {
            return Arrays.asList(this.connection().getMetaData().getSQLKeywords().split(","));
        }
        catch (SQLException e) {
            LOGGER.debug("Failed to acquire SQL keywords from JDBC driver.", (Throwable)e);
            return Collections.emptyList();
        }
    }

    public boolean hasExtendedStringSupport() {
        try {
            String value = Strings.defaultIfBlank((String)this.getDatabaseParameterValue("MAX_STRING_SIZE"), (String)"STANDARD");
            LOGGER.info("Oracle MAX_STRING_SIZE is {}", (Object)value);
            return "EXTENDED".equals(value);
        }
        catch (Exception e) {
            LOGGER.warn("Failed to check MAX_STRING_SIZE status, defaulting to STANDARD.", (Throwable)e);
            return false;
        }
    }

    public String getDatabaseParameterValue(String parameterName) throws SQLException {
        String query = "SELECT VALUE FROM V$PARAMETER WHERE UPPER(NAME) = UPPER(?)";
        return (String)this.prepareQueryAndMap("SELECT VALUE FROM V$PARAMETER WHERE UPPER(NAME) = UPPER(?)", ps -> ps.setString(1, parameterName), rs -> rs.next() ? rs.getString(1) : null);
    }

    private static Scn readScnColumnAsScn(ResultSet rs, String columnName) throws SQLException {
        String value = rs.getString(columnName);
        return Strings.isNullOrEmpty((String)value) ? Scn.NULL : Scn.valueOf(value);
    }

    private static Instant readTimestampAsInstant(ResultSet rs, String columnName) throws SQLException {
        Timestamp value = rs.getTimestamp(columnName);
        return value == null ? null : value.toInstant();
    }

    public static class NonRelationalTableException
    extends DebeziumException {
        public NonRelationalTableException(String message) {
            super(message);
        }
    }

    @FunctionalInterface
    static interface ObjectIdentifierConsumer {
        public void apply(Long var1, Long var2);
    }
}

